/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

/**
 *<br> class:		DynBass
 *<br> namespace:	caosGL::core
 *<br> inherits:	<none>
 *<br> implements:	<none>
 *<br> purpose:		the Bass sound system dynamic loader
 *
 */

#ifndef CAOSGL_CORE_DYNBASS
#define CAOSGL_CORE_DYNBASS

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes

// extern includes
#include <wtypes.h>


namespace caosGL {
	namespace core {


#ifndef BASSDEF
#define BASSDEF(f) WINAPI f
#endif

typedef DWORD HMUSIC;		// MOD music handle
typedef DWORD HSAMPLE;		// sample handle
typedef DWORD HCHANNEL;		// playing sample's channel handle
typedef DWORD HSTREAM;		// sample stream handle
typedef DWORD HSYNC;		// synchronizer handle
typedef DWORD HDSP;			// DSP handle

// Error codes returned by BASS_GetErrorCode()
#define BASS_OK				0	// all is OK
#define BASS_ERROR_MEM		1	// memory error
#define BASS_ERROR_FILEOPEN	2	// can't open the file
#define BASS_ERROR_DRIVER	3	// can't find a free/valid driver
#define BASS_ERROR_BUFLOST	4	// the sample buffer was lost - please report this!
#define BASS_ERROR_HANDLE	5	// invalid handle
#define BASS_ERROR_FORMAT	6	// unsupported format
#define BASS_ERROR_POSITION	7	// invalid playback position
#define BASS_ERROR_INIT		8	// BASS_Init has not been successfully called
#define BASS_ERROR_START	9	// BASS_Start has not been successfully called
#define BASS_ERROR_INITCD	10	// can't initialize CD
#define BASS_ERROR_CDINIT	11	// BASS_CDInit has not been successfully called
#define BASS_ERROR_NOCD		12	// no CD in drive
#define BASS_ERROR_CDTRACK	13	// can't play the selected CD track
#define BASS_ERROR_ALREADY	14	// already initialized
#define BASS_ERROR_CDVOL	15	// CD has no volume control
#define BASS_ERROR_NOPAUSE	16	// not paused
#define BASS_ERROR_NOTAUDIO	17	// not an audio track
#define BASS_ERROR_NOCHAN	18	// can't get a free channel
#define BASS_ERROR_ILLTYPE	19	// an illegal type was specified
#define BASS_ERROR_ILLPARAM	20	// an illegal parameter was specified
#define BASS_ERROR_NO3D		21	// no 3D support
#define BASS_ERROR_NOEAX	22	// no EAX support
#define BASS_ERROR_DEVICE	23	// illegal device number
#define BASS_ERROR_NOPLAY	24	// not playing
#define BASS_ERROR_FREQ		25	// illegal sample rate
#define BASS_ERROR_NOA3D	26	// A3D.DLL is not installed
#define BASS_ERROR_NOTFILE	27	// the stream is not a file stream (WAV/MP3)
#define BASS_ERROR_NOHW		29	// no hardware voices available
#define BASS_ERROR_NOSYNC	30	// synchronizers have been disabled
#define BASS_ERROR_UNKNOWN	-1	// some other mystery error

// Device setup flags
#define BASS_DEVICE_8BITS	1	// use 8 bit resolution, else 16 bit
#define BASS_DEVICE_MONO	2	// use mono, else stereo
#define BASS_DEVICE_3D		4	// enable 3D functionality
/* If the BASS_DEVICE_3D flag is not specified when initilizing BASS,
then the 3D flags (BASS_SAMPLE_3D and BASS_MUSIC_3D) are ignored when
loading/creating a sample/stream/music. */
#define BASS_DEVICE_A3D		8	// enable A3D functionality
#define BASS_DEVICE_NOSYNC	16	// disable synchronizers

// DirectSound interfaces (for use with BASS_GetDSoundObject)
#define BASS_OBJECT_DS		1	// IDirectSound
#define BASS_OBJECT_DS3DL	2	// IDirectSound3DListener

typedef struct {
	DWORD size;		// size of this struct (set this before calling the function)
	DWORD flags;	// device capabilities (DSCAPS_xxx flags)
/* The following values are irrelevant if the device doesn't have hardware
support (DSCAPS_EMULDRIVER is specified in flags) */
	DWORD hwsize;	// size of total device hardware memory
	DWORD hwfree;	// size of free device hardware memory
	DWORD freesam;	// number of free sample slots in the hardware
	DWORD free3d;	// number of free 3D sample slots in the hardware
	DWORD minrate;	// min sample rate supported by the hardware
	DWORD maxrate;	// max sample rate supported by the hardware
	BOOL eax;		// device supports EAX? (always FALSE if BASS_DEVICE_3D was not used)
	DWORD a3d;		// A3D version (0=unsupported or BASS_DEVICE_A3D was not used)
	DWORD dsver;	// DirectSound version (use to check for DX5/7 functions)
} BASS_INFO;

// BASS_INFO flags (from DSOUND.H)
#define DSCAPS_CONTINUOUSRATE	0x00000010
/* supports all sample rates between min/maxrate */
#define DSCAPS_EMULDRIVER		0x00000020
/* device does NOT have hardware DirectSound support */
#define DSCAPS_CERTIFIED		0x00000040
/* device driver has been certified by Microsoft */
/* The following flags tell what type of samples are supported by HARDWARE
mixing, all these formats are supported by SOFTWARE mixing */
#define DSCAPS_SECONDARYMONO	0x00000100	// mono
#define DSCAPS_SECONDARYSTEREO	0x00000200	// stereo
#define DSCAPS_SECONDARY8BIT	0x00000400	// 8 bit
#define DSCAPS_SECONDARY16BIT	0x00000800	// 16 bit

// Volume sliding flags
#define BASS_SLIDE_DIGITAL	1	// slide digital master volume
#define BASS_SLIDE_CD		2	// slide CD volume

// Music flags
#define BASS_MUSIC_RAMP		1	// normal ramping
#define BASS_MUSIC_RAMPS	2	// sensitive ramping
/* Ramping doesn't take a lot of extra processing and improves
the sound quality by removing "clicks". Sensitive ramping will
leave sharp attacked samples, unlike normal ramping. */
#define BASS_MUSIC_LOOP		4	// loop music
#define BASS_MUSIC_FT2MOD	16	// play .MOD as FastTracker 2 does
#define BASS_MUSIC_PT1MOD	32	// play .MOD as ProTracker 1 does
#define BASS_MUSIC_MONO		64	// force mono mixing (less CPU usage)
#define BASS_MUSIC_3D		128	// enable 3D functionality
#define BASS_MUSIC_POSRESET	256	// stop all notes when moving position

// Sample info structure & flags
typedef struct {
	DWORD freq;		// default playback rate
	DWORD volume;	// default volume (0-100)
	int pan;		// default pan (-100=left, 0=middle, 100=right)
	DWORD flags;	// BASS_SAMPLE_xxx flags
	DWORD length;	// length (in samples, not bytes)
	DWORD max;		// maximum simultaneous playbacks
	DWORD mode3d;	// BASS_3DMODE_xxx mode
	float mindist;	// minimum distance
	float maxdist;	// maximum distance
	DWORD iangle;	// angle of inside projection cone
	DWORD oangle;	// angle of outside projection cone
	DWORD outvol;	// delta-volume outside the projection cone
/* The following are the defaults used if the sample uses the DirectX 7
voice allocation/management features. */
	DWORD vam;		// voice allocation/management flags (BASS_VAM_xxx)
	DWORD priority;	// priority (0=lowest, 0xffffffff=highest)
} BASS_SAMPLE;

#define BASS_SAMPLE_8BITS		1	// 8 bit, else 16 bit
#define BASS_SAMPLE_MONO		2	// mono, else stereo
#define BASS_SAMPLE_LOOP		4	// looped
#define BASS_SAMPLE_3D			8	// 3D functionality enabled
#define BASS_SAMPLE_SOFTWARE	16	// it's NOT using hardware mixing
#define BASS_SAMPLE_MUTEMAX		32	// muted at max distance (3D only)
#define BASS_SAMPLE_VAM			64	// uses the DX7 voice allocation & management
#define BASS_SAMPLE_OVER_VOL	0x10000	// override lowest volume
#define BASS_SAMPLE_OVER_POS	0x20000	// override longest playing
#define BASS_SAMPLE_OVER_DIST	0x30000 // override furthest from listener (3D only)

#define BASS_MP3_HALFRATE	0x10000 // reduced quality MP3 (half sample rate)
#define BASS_MP3_SETPOS		0x20000 // enable seeking on the MP3

// DX7 voice allocation flags
#define BASS_VAM_HARDWARE		1
/* Play the sample in hardware. If no hardware voices are available then
the "play" call will fail */
#define BASS_VAM_SOFTWARE		2
/* Play the sample in software (ie. non-accelerated). No other VAM flags
may be used together with this flag. */

// DX7 voice management flags
/* These flags enable hardware resource stealing... if the hardware has no
available voices, a currently playing buffer will be stopped to make room for
the new buffer. NOTE: only samples loaded/created with the BASS_SAMPLE_VAM
flag are considered for termination by the DX7 voice management. */
#define BASS_VAM_TERM_TIME		4
/* If there are no free hardware voices, the buffer to be terminated will be
the one with the least time left to play. */
#define BASS_VAM_TERM_DIST		8
/* If there are no free hardware voices, the buffer to be terminated will be
one that was loaded/created with the BASS_SAMPLE_MUTEMAX flag and is beyond
it's max distance. If there are no buffers that match this criteria, then the
"play" call will fail. */
#define BASS_VAM_TERM_PRIO		16
/* If there are no free hardware voices, the buffer to be terminated will be
the one with the lowest priority. */


// 3D vector (for 3D positions/velocities/orientations)
typedef struct {
	float x;	// +=right, -=left
	float y;	// +=up, -=down
	float z;	// +=front, -=behind
} BASS_3DVECTOR;

// 3D channel modes
#define BASS_3DMODE_NORMAL		0
/* normal 3D processing */
#define BASS_3DMODE_RELATIVE	1
/* The channel's 3D position (position/velocity/orientation) are relative to
the listener. When the listener's position/velocity/orientation is changed
with BASS_Set3DPosition, the channel's position relative to the listener does
not change. */
#define BASS_3DMODE_OFF			2
/* Turn off 3D processing on the channel, the sound will be played
in the center. */

// EAX environments, use with BASS_SetEAXParameters
enum
{
	EAX_ENVIRONMENT_GENERIC,
	EAX_ENVIRONMENT_PADDEDCELL,
	EAX_ENVIRONMENT_ROOM,
	EAX_ENVIRONMENT_BATHROOM,
	EAX_ENVIRONMENT_LIVINGROOM,
	EAX_ENVIRONMENT_STONEROOM,
	EAX_ENVIRONMENT_AUDITORIUM,
	EAX_ENVIRONMENT_CONCERTHALL,
	EAX_ENVIRONMENT_CAVE,
	EAX_ENVIRONMENT_ARENA,
	EAX_ENVIRONMENT_HANGAR,
	EAX_ENVIRONMENT_CARPETEDHALLWAY,
	EAX_ENVIRONMENT_HALLWAY,
	EAX_ENVIRONMENT_STONECORRIDOR,
	EAX_ENVIRONMENT_ALLEY,
	EAX_ENVIRONMENT_FOREST,
	EAX_ENVIRONMENT_CITY,
	EAX_ENVIRONMENT_MOUNTAINS,
	EAX_ENVIRONMENT_QUARRY,
	EAX_ENVIRONMENT_PLAIN,
	EAX_ENVIRONMENT_PARKINGLOT,
	EAX_ENVIRONMENT_SEWERPIPE,
	EAX_ENVIRONMENT_UNDERWATER,
	EAX_ENVIRONMENT_DRUGGED,
	EAX_ENVIRONMENT_DIZZY,
	EAX_ENVIRONMENT_PSYCHOTIC,

	EAX_ENVIRONMENT_COUNT			// total number of environments
};

// EAX presets, usage: BASS_SetEAXParameters(EAX_PRESET_xxx)
#define EAX_PRESET_GENERIC         EAX_ENVIRONMENT_GENERIC,0.5F,1.493F,0.5F
#define EAX_PRESET_PADDEDCELL      EAX_ENVIRONMENT_PADDEDCELL,0.25F,0.1F,0.0F
#define EAX_PRESET_ROOM            EAX_ENVIRONMENT_ROOM,0.417F,0.4F,0.666F
#define EAX_PRESET_BATHROOM        EAX_ENVIRONMENT_BATHROOM,0.653F,1.499F,0.166F
#define EAX_PRESET_LIVINGROOM      EAX_ENVIRONMENT_LIVINGROOM,0.208F,0.478F,0.0F
#define EAX_PRESET_STONEROOM       EAX_ENVIRONMENT_STONEROOM,0.5F,2.309F,0.888F
#define EAX_PRESET_AUDITORIUM      EAX_ENVIRONMENT_AUDITORIUM,0.403F,4.279F,0.5F
#define EAX_PRESET_CONCERTHALL     EAX_ENVIRONMENT_CONCERTHALL,0.5F,3.961F,0.5F
#define EAX_PRESET_CAVE            EAX_ENVIRONMENT_CAVE,0.5F,2.886F,1.304F
#define EAX_PRESET_ARENA           EAX_ENVIRONMENT_ARENA,0.361F,7.284F,0.332F
#define EAX_PRESET_HANGAR          EAX_ENVIRONMENT_HANGAR,0.5F,10.0F,0.3F
#define EAX_PRESET_CARPETEDHALLWAY EAX_ENVIRONMENT_CARPETEDHALLWAY,0.153F,0.259F,2.0F
#define EAX_PRESET_HALLWAY         EAX_ENVIRONMENT_HALLWAY,0.361F,1.493F,0.0F
#define EAX_PRESET_STONECORRIDOR   EAX_ENVIRONMENT_STONECORRIDOR,0.444F,2.697F,0.638F
#define EAX_PRESET_ALLEY           EAX_ENVIRONMENT_ALLEY,0.25F,1.752F,0.776F
#define EAX_PRESET_FOREST          EAX_ENVIRONMENT_FOREST,0.111F,3.145F,0.472F
#define EAX_PRESET_CITY            EAX_ENVIRONMENT_CITY,0.111F,2.767F,0.224F
#define EAX_PRESET_MOUNTAINS       EAX_ENVIRONMENT_MOUNTAINS,0.194F,7.841F,0.472F
#define EAX_PRESET_QUARRY          EAX_ENVIRONMENT_QUARRY,1.0F,1.499F,0.5F
#define EAX_PRESET_PLAIN           EAX_ENVIRONMENT_PLAIN,0.097F,2.767F,0.224F
#define EAX_PRESET_PARKINGLOT      EAX_ENVIRONMENT_PARKINGLOT,0.208F,1.652F,1.5F
#define EAX_PRESET_SEWERPIPE       EAX_ENVIRONMENT_SEWERPIPE,0.652F,2.886F,0.25F
#define EAX_PRESET_UNDERWATER      EAX_ENVIRONMENT_UNDERWATER,1.0F,1.499F,0.0F
#define EAX_PRESET_DRUGGED         EAX_ENVIRONMENT_DRUGGED,0.875F,8.392F,1.388F
#define EAX_PRESET_DIZZY           EAX_ENVIRONMENT_DIZZY,0.139F,17.234F,0.666F
#define EAX_PRESET_PSYCHOTIC       EAX_ENVIRONMENT_PSYCHOTIC,0.486F,7.563F,0.806F

// A3D resource manager modes
#define A3D_RESMODE_OFF				0	// off
#define A3D_RESMODE_NOTIFY			1	// notify when there are no free hardware channels
#define A3D_RESMODE_DYNAMIC     	2	// non-looping channels are managed
#define A3D_RESMODE_DYNAMIC_LOOPERS	3	// all (inc. looping) channels are managed (req A3D 1.2)

// software 3D mixing algorithm modes (used with BASS_Set3DAlgorithm)
#define BASS_3DALG_DEFAULT	0
/* default algorithm (currently translates to BASS_3DALG_OFF) */
#define BASS_3DALG_OFF		1
/* Uses normal left and right panning. The vertical axis is ignored except for
scaling of volume due to distance. Doppler shift and volume scaling are still
applied, but the 3D filtering is not performed. This is the most CPU efficient
software implementation, but provides no virtual 3D audio effect. Head Related
Transfer Function processing will not be done. Since only normal stereo panning
is used, a channel using this algorithm may be accelerated by a 2D hardware
voice if no free 3D hardware voices are available. */
#define BASS_3DALG_FULL		2
/* This algorithm gives the highest quality 3D audio effect, but uses more CPU.
Requires Windows 98 2nd Edition or Windows 2000 that uses WDM drivers, if this
mode is not available then BASS_3DALG_OFF will be used instead. */
#define BASS_3DALG_LIGHT	3
/* This algorithm gives a good 3D audio effect, and uses less CPU than the FULL
mode. Requires Windows 98 2nd Edition or Windows 2000 that uses WDM drivers, if
this mode is not available then BASS_3DALG_OFF will be used instead. */

typedef DWORD (CALLBACK STREAMPROC)(HSTREAM handle, void *buffer, DWORD length, DWORD user);
/* Stream callback function. NOTE: A stream function should obviously be as quick
as possible, other streams (and MOD musics) can't be mixed until it's finished.
handle : The stream that needs writing
buffer : Buffer to write the samples in
length : Number of bytes to write
user   : The 'user' parameter value given when calling BASS_StreamCreate
RETURN : Number of bytes written. If less than "length" then the
		 stream is assumed to be at the end, and is stopped. */

/* Sync types (with BASS_ChannelSetSync() "param" and SYNCPROC "data"
definitions) & flags. */
#define BASS_SYNC_MUSICPOS	0
/* Sync when a music reaches a position.
param: LOWORD=order (0=first, -1=all) HIWORD=row (0=first, -1=all)
data : LOWORD=order HIWORD=row */
#define BASS_SYNC_MUSICINST	1
/* Sync when an instrument (sample for the non-instrument based formats)
is played in a music (not including retrigs).
param: LOWORD=instrument (1=first) HIWORD=note (0=c0...119=b9, -1=all)
data : LOWORD=note HIWORD=volume (0-64) */
#define BASS_SYNC_END		2
/* Sync when a music or file stream reaches the end.
param: not used
data : not used */
#define BASS_SYNC_MUSICFX	3
/* Sync when the "sync" effect (XM/MTM/MOD: E8x, IT/S3M: S0x) is used.
param: 0:data=pos, 1:data="x" value
data : param=0: LOWORD=order HIWORD=row, param=1: "x" value */
#define BASS_SYNC_MIXTIME	0x40000000	// FLAG: sync at mixtime, else at playtime
#define BASS_SYNC_ONETIME	0x80000000	// FLAG: sync only once, else continuously

typedef void (CALLBACK SYNCPROC)(HSYNC handle, DWORD channel, DWORD data, DWORD user);
/* Sync callback function. NOTE: a sync callback function should be very
quick (eg. just posting a message) as other syncs cannot be processed
until it has finished. If the sync is a "mixtime" sync, then other streams
and MOD musics can not be mixed until it's finished either.
handle : The sync that has occured
channel: Channel that the sync occured in
data   : Additional data associated with the sync's occurance
user   : The 'user' parameter given when calling BASS_ChannelSetSync */

typedef void (CALLBACK DSPPROC)(HSYNC handle, DWORD channel, void *buffer, DWORD length, DWORD user);
/* DSP callback function. NOTE: A DSP function should obviously be as quick as
possible... other DSP functions, streams and MOD musics can not be processed
until it's finished.
handle : The DSP handle
channel: Channel that the DSP is being applied to
buffer : Buffer to apply the DSP to
length : Number of bytes in the buffer
user   : The 'user' parameter given when calling BASS_ChannelSetDSP */

#define CDCHANNEL	0	// CD channel, for use with BASS_Channel functions

typedef DWORD        (APIENTRY* pfnBASS_GetVersion				)();
typedef BOOL         (APIENTRY* pfnBASS_GetDeviceDescription	)(int devnum, char **desc);
typedef void         (APIENTRY* pfnBASS_SetBufferLength			)(float length);
typedef void         (APIENTRY* pfnBASS_SetGlobalVolumes		)(int musvol, int samvol, int strvol);
typedef void         (APIENTRY* pfnBASS_GetGlobalVolumes		)(DWORD *musvol, DWORD *samvol, DWORD *strvol);
typedef void         (APIENTRY* pfnBASS_SetLogCurves			)(BOOL volume, BOOL pan);
typedef void         (APIENTRY* pfnBASS_Set3DAlgorithm			)(DWORD algo);
typedef DWORD        (APIENTRY* pfnBASS_ErrorGetCode			)();
typedef BOOL         (APIENTRY* pfnBASS_Init					)(int device, DWORD freq, DWORD flags, HWND win);
typedef void         (APIENTRY* pfnBASS_Free					)();
typedef void *       (APIENTRY* pfnBASS_GetDSoundObject			)(DWORD object);
typedef void         (APIENTRY* pfnBASS_GetInfo					)(BASS_INFO *info);
typedef float        (APIENTRY* pfnBASS_GetCPU					)();
typedef BOOL         (APIENTRY* pfnBASS_Start					)();
typedef BOOL         (APIENTRY* pfnBASS_Stop					)();
typedef BOOL         (APIENTRY* pfnBASS_Pause					)();
typedef BOOL         (APIENTRY* pfnBASS_SetVolume				)(DWORD volume);
typedef int          (APIENTRY* pfnBASS_GetVolume				)();
typedef BOOL         (APIENTRY* pfnBASS_Set3DFactors			)(float distf, float rollf, float doppf);
typedef BOOL         (APIENTRY* pfnBASS_Get3DFactors			)(float *distf, float *rollf, float *doppf);
typedef BOOL         (APIENTRY* pfnBASS_Set3DPosition			)(BASS_3DVECTOR *pos, BASS_3DVECTOR *vel, BASS_3DVECTOR *front, BASS_3DVECTOR *top);
typedef BOOL         (APIENTRY* pfnBASS_Get3DPosition			)(BASS_3DVECTOR *pos, BASS_3DVECTOR *vel, BASS_3DVECTOR *front, BASS_3DVECTOR *top);
typedef BOOL         (APIENTRY* pfnBASS_Apply3D					)();
typedef BOOL         (APIENTRY* pfnBASS_SetEAXParameters		)(int env, float vol, float decay, float damp);
typedef BOOL         (APIENTRY* pfnBASS_GetEAXParameters		)(DWORD *env, float *vol, float *decay, float *damp);
typedef BOOL         (APIENTRY* pfnBASS_SetA3DResManager		)(DWORD mode);
typedef DWORD        (APIENTRY* pfnBASS_GetA3DResManager		)();
typedef BOOL         (APIENTRY* pfnBASS_SetA3DHFAbsorbtion		)(float factor);
typedef BOOL         (APIENTRY* pfnBASS_GetA3DHFAbsorbtion		)(float *factor);
typedef HMUSIC       (APIENTRY* pfnBASS_MusicLoad				)(BOOL mem, void *file, DWORD offset, DWORD length, DWORD flags);
typedef void         (APIENTRY* pfnBASS_MusicFree				)(HMUSIC handle);
typedef char *       (APIENTRY* pfnBASS_MusicGetName			)(HMUSIC handle);
typedef DWORD        (APIENTRY* pfnBASS_MusicGetLength			)(HMUSIC handle);
typedef BOOL         (APIENTRY* pfnBASS_MusicPlay				)(HMUSIC handle);
typedef BOOL         (APIENTRY* pfnBASS_MusicPlayEx				)(HMUSIC handle, DWORD pos, int flags, BOOL reset);
typedef BOOL         (APIENTRY* pfnBASS_MusicSetAmplify			)(HMUSIC handle, DWORD amp);
typedef BOOL         (APIENTRY* pfnBASS_MusicSetPanSep			)(HMUSIC handle, DWORD pan);
typedef BOOL         (APIENTRY* pfnBASS_MusicSetPositionScaler	)(HMUSIC handle, DWORD scale);
typedef HSAMPLE      (APIENTRY* pfnBASS_SampleLoad				)(BOOL mem, void *file, DWORD offset, DWORD length, DWORD max, DWORD flags);
typedef void*        (APIENTRY* pfnBASS_SampleCreate			)(DWORD length, DWORD freq, DWORD max, DWORD flags);
typedef HSAMPLE      (APIENTRY* pfnBASS_SampleCreateDone		)();
typedef void         (APIENTRY* pfnBASS_SampleFree				)(HSAMPLE handle);
typedef BOOL         (APIENTRY* pfnBASS_SampleGetInfo			)(HSAMPLE handle, BASS_SAMPLE *info);
typedef BOOL         (APIENTRY* pfnBASS_SampleSetInfo			)(HSAMPLE handle, BASS_SAMPLE *info);
typedef HCHANNEL     (APIENTRY* pfnBASS_SamplePlay				)(HSAMPLE handle);
typedef HCHANNEL     (APIENTRY* pfnBASS_SamplePlayEx			)(HSAMPLE handle, DWORD start, int freq, int volume, int pan, BOOL loop);
typedef HCHANNEL     (APIENTRY* pfnBASS_SamplePlay3D			)(HSAMPLE handle, BASS_3DVECTOR *pos, BASS_3DVECTOR *orient, BASS_3DVECTOR *vel);
typedef HCHANNEL     (APIENTRY* pfnBASS_SamplePlay3DEx			)(HSAMPLE handle, BASS_3DVECTOR *pos, BASS_3DVECTOR *orient, BASS_3DVECTOR *vel, DWORD start, int freq, int volume, BOOL loop);
typedef BOOL         (APIENTRY* pfnBASS_SampleStop				)(HSAMPLE handle);
typedef HSTREAM      (APIENTRY* pfnBASS_StreamCreate			)(DWORD freq, DWORD flags, STREAMPROC *proc, DWORD user);
typedef HSTREAM      (APIENTRY* pfnBASS_StreamCreateFile		)(BOOL mem, void *file, DWORD offset, DWORD length, DWORD flags);
typedef void         (APIENTRY* pfnBASS_StreamFree				)(HSTREAM handle);
typedef DWORD        (APIENTRY* pfnBASS_StreamGetLength			)(HSTREAM handle);
typedef DWORD        (APIENTRY* pfnBASS_StreamGetBlockLength	)(HSTREAM handle);
typedef BOOL         (APIENTRY* pfnBASS_StreamPlay				)(HSTREAM handle, BOOL flush, DWORD flags);
typedef BOOL         (APIENTRY* pfnBASS_CDInit					)(char *drive);
typedef void         (APIENTRY* pfnBASS_CDFree					)();
typedef BOOL         (APIENTRY* pfnBASS_CDInDrive				)();
typedef BOOL         (APIENTRY* pfnBASS_CDPlay					)(DWORD track, BOOL loop, BOOL wait);
typedef BOOL         (APIENTRY* pfnBASS_ChannelIsActive			)(DWORD handle);
typedef DWORD        (APIENTRY* pfnBASS_ChannelGetFlags			)(DWORD handle);
typedef BOOL         (APIENTRY* pfnBASS_ChannelStop				)(DWORD handle);
typedef BOOL         (APIENTRY* pfnBASS_ChannelPause			)(DWORD handle);
typedef BOOL         (APIENTRY* pfnBASS_ChannelResume			)(DWORD handle);
typedef BOOL         (APIENTRY* pfnBASS_ChannelSetAttributes	)(DWORD handle, int freq, int volume, int pan);
typedef BOOL         (APIENTRY* pfnBASS_ChannelGetAttributes	)(DWORD handle, DWORD *freq, DWORD *volume, int *pan);
typedef BOOL         (APIENTRY* pfnBASS_ChannelSet3DAttributes	)(DWORD handle, int mode, float min, float max, int iangle, int oangle, int outvol);
typedef BOOL         (APIENTRY* pfnBASS_ChannelGet3DAttributes	)(DWORD handle, DWORD *mode, float *min, float *max, DWORD *iangle, DWORD *oangle, DWORD *outvol);
typedef BOOL         (APIENTRY* pfnBASS_ChannelSet3DPosition	)(DWORD handle, BASS_3DVECTOR *pos, BASS_3DVECTOR *orient, BASS_3DVECTOR *vel);
typedef BOOL         (APIENTRY* pfnBASS_ChannelGet3DPosition	)(DWORD handle, BASS_3DVECTOR *pos, BASS_3DVECTOR *orient, BASS_3DVECTOR *vel);
typedef BOOL         (APIENTRY* pfnBASS_ChannelSetPosition		)(DWORD handle, DWORD pos);
typedef DWORD        (APIENTRY* pfnBASS_ChannelGetPosition		)(DWORD handle);
typedef DWORD        (APIENTRY* pfnBASS_ChannelGetLevel			)(DWORD handle);
typedef DWORD        (APIENTRY* pfnBASS_ChannelGetData			)(DWORD handle, void *buffer, DWORD length);
typedef HSYNC        (APIENTRY* pfnBASS_ChannelSetSync			)(DWORD handle, DWORD type, DWORD param, SYNCPROC *proc, DWORD user);
typedef BOOL         (APIENTRY* pfnBASS_ChannelRemoveSync		)(DWORD handle, HSYNC sync);
typedef HDSP         (APIENTRY* pfnBASS_ChannelSetDSP			)(DWORD handle, DSPPROC *proc, DWORD user);
typedef BOOL         (APIENTRY* pfnBASS_ChannelRemoveDSP		)(DWORD handle, HDSP dsp);
typedef BOOL         (APIENTRY* pfnBASS_ChannelSetEAXMix		)(DWORD handle, float mix);
typedef BOOL         (APIENTRY* pfnBASS_ChannelGetEAXMix		)(DWORD handle, float *mix);

extern pfnBASS_GetVersion				BASS_GetVersion;
extern pfnBASS_GetDeviceDescription		BASS_GetDeviceDescription;
extern pfnBASS_SetBufferLength			BASS_SetBufferLength;
extern pfnBASS_SetGlobalVolumes			BASS_SetGlobalVolumes;
extern pfnBASS_GetGlobalVolumes			BASS_GetGlobalVolumes;
extern pfnBASS_SetLogCurves				BASS_SetLogCurves;
extern pfnBASS_Set3DAlgorithm			BASS_Set3DAlgorithm;
extern pfnBASS_ErrorGetCode				BASS_ErrorGetCode;
extern pfnBASS_Init						BASS_Init;
extern pfnBASS_Free						BASS_Free;
extern pfnBASS_GetDSoundObject			BASS_GetDSoundObject;
extern pfnBASS_GetInfo					BASS_GetInfo;
extern pfnBASS_GetCPU					BASS_GetCPU;
extern pfnBASS_Start					BASS_Start;
extern pfnBASS_Stop						BASS_Stop;
extern pfnBASS_Pause					BASS_Pause;
extern pfnBASS_SetVolume				BASS_SetVolume;
extern pfnBASS_GetVolume				BASS_GetVolume;
extern pfnBASS_Set3DFactors				BASS_Set3DFactors;
extern pfnBASS_Get3DFactors				BASS_Get3DFactors;
extern pfnBASS_Set3DPosition			BASS_Set3DPosition;
extern pfnBASS_Get3DPosition			BASS_Get3DPosition;
extern pfnBASS_Apply3D					BASS_Apply3D;
extern pfnBASS_SetEAXParameters			BASS_SetEAXParameters;
extern pfnBASS_GetEAXParameters			BASS_GetEAXParameters;
extern pfnBASS_SetA3DResManager			BASS_SetA3DResManager;
extern pfnBASS_GetA3DResManager			BASS_GetA3DResManager;
extern pfnBASS_SetA3DHFAbsorbtion		BASS_SetA3DHFAbsorbtion;
extern pfnBASS_GetA3DHFAbsorbtion		BASS_GetA3DHFAbsorbtion;
extern pfnBASS_MusicLoad				BASS_MusicLoad;
extern pfnBASS_MusicFree				BASS_MusicFree;
extern pfnBASS_MusicGetName				BASS_MusicGetName;
extern pfnBASS_MusicGetLength			BASS_MusicGetLength;
extern pfnBASS_MusicPlay				BASS_MusicPlay;
extern pfnBASS_MusicPlayEx				BASS_MusicPlayEx;
extern pfnBASS_MusicSetAmplify			BASS_MusicSetAmplify;
extern pfnBASS_MusicSetPanSep			BASS_MusicSetPanSep;
extern pfnBASS_MusicSetPositionScaler	BASS_MusicSetPositionScaler;
extern pfnBASS_SampleLoad				BASS_SampleLoad;
extern pfnBASS_SampleCreate				BASS_SampleCreate;
extern pfnBASS_SampleCreateDone			BASS_SampleCreateDone;
extern pfnBASS_SampleFree				BASS_SampleFree;
extern pfnBASS_SampleGetInfo			BASS_SampleGetInfo;
extern pfnBASS_SampleSetInfo			BASS_SampleSetInfo;
extern pfnBASS_SamplePlay				BASS_SamplePlay;
extern pfnBASS_SamplePlayEx				BASS_SamplePlayEx;
extern pfnBASS_SamplePlay3D				BASS_SamplePlay3D;
extern pfnBASS_SamplePlay3DEx			BASS_SamplePlay3DEx;
extern pfnBASS_SampleStop				BASS_SampleStop;
extern pfnBASS_StreamCreate				BASS_StreamCreate;
extern pfnBASS_StreamCreateFile			BASS_StreamCreateFile;
extern pfnBASS_StreamFree				BASS_StreamFree;
extern pfnBASS_StreamGetLength			BASS_StreamGetLength;
extern pfnBASS_StreamGetBlockLength		BASS_StreamGetBlockLength;
extern pfnBASS_StreamPlay				BASS_StreamPlay;
extern pfnBASS_CDInit					BASS_CDInit;
extern pfnBASS_CDFree					BASS_CDFree;
extern pfnBASS_CDInDrive				BASS_CDInDrive;
extern pfnBASS_CDPlay					BASS_CDPlay;
extern pfnBASS_ChannelIsActive			BASS_ChannelIsActive;
extern pfnBASS_ChannelGetFlags			BASS_ChannelGetFlags;
extern pfnBASS_ChannelStop				BASS_ChannelStop;
extern pfnBASS_ChannelPause				BASS_ChannelPause;
extern pfnBASS_ChannelResume			BASS_ChannelResume;
extern pfnBASS_ChannelSetAttributes		BASS_ChannelSetAttributes;
extern pfnBASS_ChannelGetAttributes		BASS_ChannelGetAttributes;
extern pfnBASS_ChannelSet3DAttributes	BASS_ChannelSet3DAttributes;
extern pfnBASS_ChannelGet3DAttributes	BASS_ChannelGet3DAttributes;
extern pfnBASS_ChannelSet3DPosition		BASS_ChannelSet3DPosition;
extern pfnBASS_ChannelGet3DPosition		BASS_ChannelGet3DPosition;
extern pfnBASS_ChannelSetPosition		BASS_ChannelSetPosition;
extern pfnBASS_ChannelGetPosition		BASS_ChannelGetPosition;
extern pfnBASS_ChannelGetLevel			BASS_ChannelGetLevel;
extern pfnBASS_ChannelGetData			BASS_ChannelGetData;
extern pfnBASS_ChannelSetSync			BASS_ChannelSetSync;
extern pfnBASS_ChannelRemoveSync		BASS_ChannelRemoveSync;
extern pfnBASS_ChannelSetDSP			BASS_ChannelSetDSP;
extern pfnBASS_ChannelRemoveDSP			BASS_ChannelRemoveDSP;
extern pfnBASS_ChannelSetEAXMix			BASS_ChannelSetEAXMix;
extern pfnBASS_ChannelGetEAXMix			BASS_ChannelGetEAXMix;

int loadBass ();

	}
}

#endif // CAOSGL_CORE_DYNBASS

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof