#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "\sr\h\sr.h"
#include "\lib\h\control.h"
#include "\lib\h\memory.h"
#include "\lib\h\bitmap.h"
#include "h\inside.h"

#define XW 18               // ground width
#define SIZ 512
#define MAPW 128
#define FOGDEPTH 2.5

static char        *palette=NULL,*sunpic=NULL,*flarepic[5]={NULL},*flareshd=NULL;
static char        *qtable=NULL,*tmap=NULL,*fog=NULL;
static char        *envmap=NULL;
static int         flight_tag;

static SR_BITMAPOBJ *sun=NULL,*flare[10];

static SR_SCENE    *scene;
static SR_FILE     *shipsro[10];
static SR_MESH     *obj,*ship[10];
static SR_MAT      *mat,*emat;
static SR_CAMERA   *camera;
static SR_OBJ      *control;
static VECTOR3     shiploc[10],controlloc,camloc;
static MATRIX3     shipmtx[10],controlmtx,cammtx;
static float       fogval = 0.8;
static int         UVlut[(XW+1)*2];
static FLOAT       velocity = SIZ/30.0;
static int         flarecount=0;

#pragma pack(1);

typedef struct
{
	int y;
	int g;
} mapstruct;

mapstruct *map;

#pragma pack();

//-----------------------------------------------------------------------------

float getsin (int i, float loop)
{
	return (sin ((float)((PI/2.0)*i*loop)/MAPW));
}

//-----------------------------------------------------------------------------

float getcos (int i, float loop)
{
	return (cos ((float)((PI/2.0)*i*loop)/MAPW));
}

//-----------------------------------------------------------------------------

static void init_camera()
{
    camera = MEM_allocate_named(sizeof(SR_CAMERA),"camera");
	SR_camera_init(camera);
    SR_add_object  (scene, (SR_OBJ *)camera);

    scene->camera = camera;
}

//-----------------------------------------------------------------------------

static void calculate_lights()
{
	int x,y;
	float t,d,g;

	for (y=0; y<MAPW; y++)
	for (x=0; x<MAPW; x++)
	{
		d = -(map[((x+1)%MAPW)+((y)%MAPW)*MAPW].y-
			map[x+y*MAPW].y);
		t = atan(d/SIZ);
		t-= 0.7*(PI/2);

		g = cos(t)*64*16;
		if (g<10*16) g = 10*16;
		map[x+y*MAPW].g = g;
	}
}

//-----------------------------------------------------------------------------

static void generate_landscape()
{
    int   i,j;
    float r;
    SR_FILE *sro;

    obj    = MEM_allocate_named(sizeof(SR_MESH) + (XW+1)*(XW+1)*sizeof(SR_VERT) + XW*XW*sizeof(SR_POLY)*2,"grid landscape");

    sro    = MEM_allocate(400000);

    SR_create_grid (sro, XW, XW, 1, 1);
    SR_init_object((SR_OBJ *)obj);
    SR_load_object (obj, sro, SIZ*65536, mat); // initialize to dummy material
    SR_set_object_render(obj,SR_TEX_GH);
    SR_add_object(scene,(SR_OBJ*)obj);

    MEM_free (sro);

    for (i=0; i<(XW+1)*2; i++)
        UVlut[i] = (256*16.0*i)/XW;

    for (i=0; i<MAPW/2; i++)
    for (j=0; j<MAPW/2; j++)
    {
        r = SIZ/2.0;

        r              = 1.0*r*getsin (i+MAPW/2,2.0)+
                         0.7*r*getsin (j-11,4.0)+
                         0.7*r*getcos (i-13,4.3)+
                         1.0*r*getcos (j+17,6.0)+
                         1.7*r*getsin ((i*j*0.3),9.0)+
                         1.9*r*getcos (j+10,16.0)+
                         0.5*r*getcos (i-7,45.0)+
                         0.8*r*getsin (j+3,39.0)+
                         0.9*r*getsin ((j+1)*i,64.0)+
                         0.8*r*getcos (j*i,90.0)+
                         0.6*r*getsin ((i+1)*j,72.0);

        if (r > SIZ/2.0) r = 0;
            map[i*MAPW+j].y=r;

     }

    for (i=0; i<(MAPW/2)-1; i++)
    for (j=0; j<(MAPW/2)-1; j++)
    {
        r = (map[i*MAPW+j].y +
            map[i*MAPW+j+1].y +
            map[(i+1)*MAPW+j+1].y +
            map[(i+1)*MAPW+j+1].y)/4;

        map[i*MAPW+j].y = r;
    }

    for (i=0; i<MAPW/2; i++)
    for (j=0; j<MAPW/2; j++)
    {
        map[i*MAPW+j+MAPW/2].y = map[(MAPW/2)-j-1+(i*MAPW)].y;
    }

    for (i=0; i<MAPW/2; i++)
    for (j=0; j<MAPW; j++)
    {
        map[(i+(MAPW/2))*MAPW+j].y =
            map[j+((MAPW/2)-i-1)*MAPW].y;
    }

    obj->flags|=SR_NOBOUND|SR_NOCULL;
    SR_calc_poly_normals (obj);
    SR_calc_vertex_normals (obj);

}


//-----------------------------------------------------------------------------

static void initialize_ships()
{
    int i,j,k;

    ship[0] = MEM_allocate_named (SR_SIZE(shipsro[0]),"flight ship 0");
    ship[1] = MEM_allocate_named (SR_SIZE(shipsro[0]),"flight ship 1");
    ship[2] = MEM_allocate_named (SR_SIZE(shipsro[0]),"flight ship 2");
    SR_init_object((SR_OBJ *)ship[0]);
    SR_load_object(ship[0],shipsro[0],65536*0.1,emat);
    SR_set_object_render(ship[0],SR_TEX);
    ship[0]->flags|=SR_ENVLOCAL;
    SR_mesh_copy (ship[1],ship[0]);
    SR_mesh_copy (ship[2],ship[0]);

    j = rand()%(SIZ*30);
    k = rand()%(SIZ*30);

    for (i=0; i<=2; i++)
    {
        SR_vmake (shiploc[i],j+rand()%(SIZ*10),i*SIZ*5,k-i*SIZ*5);
        SR_munit (shipmtx[i]);
    }

    SR_add_object(scene,(SR_OBJ*)ship[0]);
    SR_add_object(scene,(SR_OBJ*)ship[1]);
    SR_add_object(scene,(SR_OBJ*)ship[2]);


}

//-----------------------------------------------------------------------------

static void cast_grid (SR_MESH *grid,SR_CAMERA *cam)
{
	int x,z,gx,gz,vv,mx,mz;
	SR_VERT *v;
	mapstruct *m,*m2;

    grid->x = ((cam->x)/SIZ);
    grid->z = ((cam->z)/SIZ);
	grid->x*=SIZ;
	grid->z*=SIZ;
	gx = grid->x/SIZ;
	gz = grid->z/SIZ;
	mx = gx;
	mz = gz;
	while (mx<0) mx+=MAPW;
	while (mz<0) mz+=MAPW;
	while (gx<0) gx+=XW;
	while (gz<0) gz+=XW;
	gx%=XW;
	gz%=XW;
	mx%=MAPW;
	mz%=MAPW;

	v = grid->vertex;

	for (z=0; z<=XW; z++)
	{
		m = &(map[MAPW*((mz+z)&(MAPW-1))]);
		vv = UVlut[gz+z];

		for (x=0; x<=XW; x++,v++)
		{
			m2 = m+((mx+x)&(MAPW-1));
			v->u = UVlut[gx+x];
			v->v = vv;
			v->y = m2->y;
			v->g = m2->g;
		}
	}
}

//-----------------------------------------------------------------------------

static float dist(float x, float y, float z)
{
	return sqrt(x*x+y*y+z*z);
}

//-----------------------------------------------------------------------------

static void fog_object (SR_MESH *obj, SR_CAMERA *cam)
{
	SR_VERT *v;
    SR_POLY *p;

    float camx,camy,camz;
    FLOAT d,fog;
	int FOGLIMIT;

    FOGLIMIT = fogval*((SIZ*(XW/2.0-(FOGDEPTH))));

    camx = cam->x%SIZ;
    camy = cam->y;
    camz = cam->z%SIZ;

	for (v=obj->vertex;v;v=v->next)
	if (!(v->flags&SR_DISABLE))
	{
        d = dist (camx-v->x,camy-v->y,camz-v->z)-FOGLIMIT;

		if (d<0) fog = 0; else
			fog = d/(FOGDEPTH*SIZ)*63*16;

		v->h=fog;

        if (v->h<0)     v->h = 0;
        if (v->h>63*16) v->h = 63*16;

	}

    for (p=obj->poly; p; p=p->next)
        if (p->p1->h==0 && p->p2->h == 0 && p->p3->h == 0)
            p->render_func = SR_TEX_G; else
            p->render_func = SR_TEX_GH;
}


//-----------------------------------------------------------------------------

static void ship_ai (int no, float speed)
{
    VECTOR3 v,w;
    FLOAT r;


    SR_vcopy (v,shiploc[no]);
	SR_vmake (w,0,0,speed);
    SR_vmove (v,w,SR_vlen(w),shipmtx[no]);
    SR_vcopy (shiploc[no],v);

    r = DEGREE_TO_RAD(0.6)*getsin ((demotimer*0.12+no*30.6)*1.1,(1.7+no/4.0))+
        DEGREE_TO_RAD(1.2)*getcos ((demotimer*0.12+91-no*17),1.2)+
        DEGREE_TO_RAD(-0.8)*getcos ((demotimer*0.12-42+no*91),-1.71)-
        DEGREE_TO_RAD(-0.4)*getsin ((demotimer*0.12-81-no*13.5), 2.31);

    if (fabs(r)>DEGREE_TO_RAD(0.4))
        SR_myaw (shipmtx[no],r);

    shiploc[no][1]*=0.99;

    if (no>0)
    {
        if (shiploc[no][0]>shiploc[0][0]) shiploc[no][0]-=speed*0.65;
        if (shiploc[no][0]<shiploc[0][0]) shiploc[no][0]+=speed*0.65;

        if (shiploc[no][2]>shiploc[0][2]) shiploc[no][2]-=speed*0.65;
        if (shiploc[no][2]<shiploc[0][2]) shiploc[no][2]+=speed*0.65;
    }

    if (shiploc[no][1]>-SIZ*(1.7+no*0.3)) shiploc[no][1]=-(SIZ*(1.7+no*0.3));

}

//-----------------------------------------------------------------------------

static void move_camera(float speed)
{
    VECTOR3 w,v1;

	SR_vmake (w,0,0,speed);
    SR_vmove (camloc,w,SR_vlen(w),cammtx);

    camloc[1]*=0.995;

    if (camloc[1]>-SIZ*2.0)
      camloc[1]=-SIZ*2.0;

    SR_vcopy (v1,controlloc);
    SR_vsub (v1,camloc);
    SR_vec_to_matrix (cammtx,v1);

    SR_mrotz (cammtx,DEGREE_TO_RAD(180));
    SR_mpitch (cammtx,-DEGREE_TO_RAD(12));
}

//-----------------------------------------------------------------------------

static void move_control_point()
{
    SR_vmake (controlloc,0,0,0);
    SR_vadd  (controlloc,shiploc[0]);
    SR_munit (controlmtx);
}

//-----------------------------------------------------------------------------

void flight_irq()
{
    ship_ai (0,velocity*0.4);
    ship_ai (1,velocity*0.3);
    ship_ai (2,velocity*0.2);

    move_control_point();
    move_camera(velocity*0.3);
}

//-----------------------------------------------------------------------------

static SR_BITMAPOBJ *make_flare ()
{
    SR_BITMAPOBJ *b;

    b = MEM_allocate_named(sizeof(SR_BITMAPOBJ),"flight sun flare");

    SR_bitmap_init (b);

    b->pipeline = SR_BMOpipeline;
    b->parent = NULL;

    b->flags = 0;
    b->render_func = &SR_BMP;
    b->bitmap = sunpic;
    b->qtable = NULL;
    b->xsiz   = 128;
    b->ysiz   = 128;
    b->physwidth = 128;
    b->refscale = 1;
    b->scale = 70.0;
    b->depth = 0;

    SR_add_object (scene,(SR_OBJ *)b);
    return b;

}

//-----------------------------------------------------------------------------

#define MILJUUNA 100*256*65536

static SR_BITMAPOBJ *make_flare_reflection (char *bit, int xs, int ys, float refscale)
{
    SR_BITMAPOBJ *b;

    b = MEM_allocate_named(sizeof(SR_BITMAPOBJ),"flight flare");

    SR_bitmap_init (b);

    b->pipeline = SR_BMOpipeline;
    b->parent = NULL;

    b->flags = 0;
    b->render_func = &SR_BMP_I;
    b->bitmap = bit;
    b->qtable = flareshd;
    b->xsiz   = xs;
    b->ysiz   = ys;
    b->physwidth = xs;
    b->refscale = refscale;
    b->scale = 64.0;
    b->depth = -MILJUUNA;

    flarecount++;

    SR_add_object (scene,(SR_OBJ *)b);

    return b;

}

//-----------------------------------------------------------------------------

static void initialize_bitmaps()
{
    flarecount=0;

    sun = make_flare();

    flare[0] = make_flare_reflection(flarepic[3],12,11, 0.0);
    flare[1] = make_flare_reflection(flarepic[4],4,4  ,-0.2);
    flare[2] = make_flare_reflection(flarepic[4],4,4  , 0.5);
    flare[3] = make_flare_reflection(flarepic[3],12,11, 0.3);
    flare[4] = make_flare_reflection(flarepic[4],4,4,-1.3);
    flare[5] = make_flare_reflection(flarepic[0],74,63,-0.9);
    flare[6] = make_flare_reflection(flarepic[1],52,43,-0.47);
    flare[7] = make_flare_reflection(flarepic[2],31,23,-2.0);

}

//-----------------------------------------------------------------------------

void check_reflections()
{
    VECTOR2 s;
    VECTOR3 l;
    int     i;

    for (i=0; i<flarecount; i++)
    {
        flare[i]->x = sun->x;
        flare[i]->y = sun->y;
        flare[i]->z = sun->z;
    }

    SR_vmake (l,sun->x,sun->y,sun->z);
    i = SR_project (scene,s,l);

    if (i==0 && s[0]>=scene->xstart-12 && s[0]<scene->xstop+12 &&
                s[1]>=scene->ystart-12 && s[1]<scene->ystop+12)
    {
        for (i=0; i<flarecount; i++)
            flare[i]->flags&=~SR_DISABLE;
    } else
        for (i=0; i<flarecount; i++)
            flare[i]->flags|=SR_DISABLE;
}

//-----------------------------------------------------------------------------

static void check_camera_collisions()
{
    int i,j,k,r;

    for (r=0; r<=2; r++)
    {
        if (dist (shiploc[r][0]-camloc[0],shiploc[r][1]-camloc[1],shiploc[r][2]-camloc[2])<
         0.30*SIZ)
         {
                i=(shiploc[r][0]-camloc[0])*(5+rand()%10);
                j=(shiploc[r][1]-camloc[1])*(2+rand()%5);
                k=(shiploc[r][2]-camloc[2])*(5+rand()%10);

                camloc[0]-=i;
                camloc[1]-=j;
                camloc[2]-=k;
         }
    }
}

//-----------------------------------------------------------------------------

static void copy_locations()
{

    int i;

    for (i=0; i<=2; i++)
    {
        ship[i]->x = shiploc[i][0];
        ship[i]->y = shiploc[i][1];
        ship[i]->z = shiploc[i][2];
        SR_mcopy (ship[i]->matrix,shipmtx[i]);
    }

    camera->x = camloc[0];      // copy camera parameters
    camera->y = camloc[1];
    camera->z = camloc[2];

    SR_mcopy (camera->matrix,cammtx);

    control->x = controlloc[0];      // copy camera parameters
    control->y = controlloc[1];
    control->z = controlloc[2];

    SR_mcopy (control->matrix,controlmtx);

    sun->x = control->x-280*SIZ;
    sun->z = control->z+0*SIZ;
    sun->y = -40*SIZ;

}

//-----------------------------------------------------------------------------

static void flight_main_loop()
{

    scene->buffer = buffer1;

    copy_locations();
    check_camera_collisions();

    check_reflections();
    cast_grid (obj,camera);
    fog_object (obj,camera);

    SR_clear_view (scene,221);
    SR_render_view(scene);

    show_info();
    flip_page();

}

//-----------------------------------------------------------------------------

void flight_load()
{
    flight_tag = MEM_get_unused_tag();
    MEM_set_tag (flight_tag);

    srand (4582140);

    map         = MEM_allocate_named (MAPW*MAPW*sizeof(mapstruct),"map heightfield");
    flarepic[0] = load_file ("flight/lens1.raw");
    flarepic[1] = load_file ("flight/lens2.raw");
    flarepic[2] = load_file ("flight/lens3.raw");
    flarepic[3] = load_file ("flight/lens4.raw");
    flarepic[4] = load_file ("flight/lens5.raw");
    flareshd    = load_file ("flight/flare.shd");
    sunpic      = load_file ("flight/sun.raw");
    tmap        = load_file ("flight/map.raw");
    fog         = load_file ("flight/map.fog");
    qtable      = load_file ("flight/map.shd");
    envmap      = load_file ("flight/river.raw");
    shipsro[0]  = load_file ("flight/ship.sro");
    palette     = load_file ("flight/map.pal");

    mat         = MEM_allocate_named(sizeof(SR_MAT),"flight landscape material");
    emat        = MEM_allocate_named(sizeof(SR_MAT),"flight envmap material");

    scene = MEM_allocate_named(sizeof(SR_SCENE),"flight scene");     // allocate memory for scene

    SR_init_scene (scene);                  // initialize scene to default parameters
    SR_set_viewport(scene,320,200,0,0,320,200);

    scene->perspective = 200;

    SR_set_material (mat,tmap,NULL,qtable,fog,511,511,9,9);
    SR_set_material (emat,envmap,NULL,NULL,fog,255,255,8,8);

    init_camera();

    control = MEM_allocate_named(sizeof(SR_OBJ),"flight control obj");

    SR_init_object (control);
    SR_set_location (control,0,-SIZ,-SIZ/2);
    SR_add_object(scene,control);

    generate_landscape();

    calculate_lights();
    initialize_ships();
    initialize_bitmaps();

    SR_munit (cammtx);
    SR_vmake (camloc,rand()%(SIZ*5),0,rand()%(SIZ*4));
    SR_vadd  (camloc,shiploc[0]);
    SR_munit (controlmtx);
    SR_vmake (controlloc,rand()%(SIZ*5),0,rand()%(SIZ*4));
    SR_vadd  (controlloc,shiploc[0]);

}

//-----------------------------------------------------------------------------

void flight_init()
{
    MEM_set (buffer1,0,320*200);
    MEM_set (buffer2,0,320*200);

    set_base_palette(palette);
    set_fade(1.0,60);
    set_demo_irq  (&flight_irq);
    set_demo_loop (&flight_main_loop);
}

//-----------------------------------------------------------------------------

void flight_exit()
{
    set_demo_irq  (NULL);
    set_demo_loop (NULL);

    MEM_free(map);

//    MEM_free_tagged (flight_tag); // !!!!!!!!!!!!!!!!!!!
}

//-----------------------------------------------------------------------------

