//
//                       __      __      _  __ 
//                   .: (__ (__ (--) >-< ( (-_ :.
//
//
// (c) Copyright 1997 Mikko E. Mononen <memon@inside.org>
//


#include <math.h>
#include <stdio.h>
#include "clax.h"
#include "misc.hh"


extern
float
fsqrt( float a );
#pragma aux fsqrt modify nomemory [EBX EDX] parm [EAX] value no8087 = \
"   MOV     EDX,EAX              " \
"   AND     EAX,0x007FFFFF       " \
"   SUB     EDX,0x3F800000       " \
"   MOV     EBX,EDX              " \
"   AND     EDX,0xFF000000       " \
"   SAR     EDX,1                " \
"   AND     EBX,0x00800000       " \
"   ADD     EDX,0x3F800000       " \
"   OR      EAX,EBX              " \
"   SHR     EAX,16               " \
"   MOV     EBX,OFFSET sqrttab   " \
"   DB      0x8B, 0x04, 0x83     " \
"   ADD     EAX,EDX              "

// Note that DB ... means MOV EAX,[4*EAX+EBX] which WC won't accept :(


#define MED_CONST       (11.0/32.0)
#define MIN_CONST       (1.0/4.0)
float
vec_apxlength( c_VECTOR *a )
{
  float   maxc, medc, minc, tmp;

  maxc = a->x;
  medc = a->y;
  minc = a->z;
  ffabs( &maxc );
  ffabs( &medc );
  ffabs( &minc );

  // maxc, medc, and minc are positive values, so
  // we can use integer comparisions.
  if ( *(long *)&maxc < *(long *)&medc ) {
    tmp = medc;
    medc = maxc;
    maxc = tmp;
  }
  if ( *(long *)&maxc < *(long *)&minc) {
    tmp = minc;
    minc = maxc;
    maxc = tmp;
  }
  if ( *(long *)&medc < *(long *)&minc) {
    tmp = minc;
    minc = medc;
    medc = tmp;
  }

  return (maxc + medc * MED_CONST + minc * MIN_CONST);
}

void
vec_apxnormalize( c_VECTOR *a, c_VECTOR *res )
{
  float  d = vec_apxlength( a );
  
  if ( ispos( &d ) ) {
    d = 1.0 / d;
    res->x = a->x * d;
    res->y = a->y * d;
    res->z = a->z * d;
  } else {
    res->x = 0.0;
    res->y = 0.0;
    res->z = 0.0;
  }
}

float
vec_fastlength( c_VECTOR *a )
{
  return fsqrt( a->x * a->x + a->y * a->y + a->z * a->z );
}

void
vec_fastnormalize( c_VECTOR *a, c_VECTOR *res )
{
  float  d = vec_fastlength( a );
  
  if ( ispos( &d ) ) {
    d = 1.0 / d;
    res->x = a->x * d;
    res->y = a->y * d;
    res->z = a->z * d;
  } else {
    res->x = 0.0;
    res->y = 0.0;
    res->z = 0.0;
  }
}

int32
mat_inverse_c (c_MATRIX a, c_MATRIX out)
{
/*
  mat_inverse: inverse matrix calculation (non-singular).
*/
  int16    i, j;
  float    scale;
  c_MATRIX temp;

  scale = a[0][X]*a[0][X] + a[0][Y]*a[0][Y] + a[0][Z]*a[0][Z];
  if (scale == 0.0) {
    mat_identity (out);
    return clax_err_singular;
  }
  scale = 1.0 / scale;

  for (i = 0; i < 3; i++) {
    for (j = 0; j < 3; j++) temp[i][j] = a[j][i] * scale;
    temp[i][W] = -(temp[i][X]*a[X][W] +
                   temp[i][Y]*a[Y][W] +
                   temp[i][Z]*a[Z][W]);
  }
  mat_copy (temp, out);
  return clax_err_ok;
}

int
exists( char* file )
{
  FILE* fp;
  
  if ( (fp = fopen( file, "rb" )) != 0 ) {
    fclose( fp );
    return -1;
  }

  fclose( fp );
  return 0;
}

int32
insideViewCone( float ratioX, float ratioY, c_VECTOR* vec, float r )
{

  int    result;
  float  x = vec->x * ratioX;
  float  y = vec->y * ratioY;
  float  z = vec->z;
  float  d;

  const float  dx = 1.0 / fsqrt( 1 + ratioX * ratioX );
  const float  dy = 1.0 / fsqrt( 1 + ratioY * ratioY );

  result = 0;

  // left
  d = (-z - x) * dx;
  ffabs( &d );
  if ( x < -z && d > r )
    result |= 1;

  // right
  d = (z - x) * dx;
  ffabs( &d );
  if ( x > z && d > r )
    result |= 2;

  // top
  d = (z - y) *dy;
  ffabs( &d );
  if ( y > z && d > r )
    result |= 4;

  // bottom
  d = (-z - y) * dy;
  ffabs( &d );
  if ( y < -z && d > r )
    result |= 8;
  
  return result;
}
