//
//                       __      __      _  __ 
//                   .: (__ (__ (--) >-< ( (-_ :.
//
//
// (c) Copyright 1997 Mikko E. Mononen <memon@inside.org>
//

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <malloc.h>

#include "globals.hh"
#include "material.hh"
#include "draw.hh"
#include "clax.h"
#include "ie.hh"
#include "file_tga.hh"
#include "misc.hh"



c_SCENE*   scene;
c_CAMERA*  camera;

int        maxfaces, maxverts;

int        numVisibleFaces;
Sort*      sortList;
Sort*      tmpList;

int        numLights;
c_LIGHT**  lightList;

const float  IE_minZ = 1.0;


static
void
putPixel( int x, int y, char color )
{
  if ( (x >= 0) && (x < 320) && (y >= 0) && (y < 200) )
    WritePagePtr[x + y * 320] = color;
}



#define ByteOf(x) (((x) >> bitsOffset) & 0xff)

static
void
radix( short bitsOffset, ulong N, Sort *source, Sort *dest )
{
  // suppressed the need for index as it is reported in count
  ulong count[256];

  // added temp variables to simplify writing, understanding and compiler optimization job
  // most of them will be allocated as registers
  ulong  *cp;
  Sort   *sp;
  ulong  s, c, i;

  // faster than MemSet
  cp = count;
  for (i = 256; i > 0; --i, ++cp)
    *cp = 0;

  // count occurences of every byte value
  sp = source;
  for (i = N; i > 0; --i, ++sp) {
    cp = count + ByteOf( sp->z );
    (*cp)++;
  }

  // transform count into index by summing elements and storing into same array
  s = 0;
  cp = count;
  for (i = 256; i > 0; --i, ++cp) {
    c = *cp;
    *cp = s;
    s += c;
  }

  // fill dest with the right values in the right place
  sp = source;
  for (i = N; i > 0; --i, ++sp) {
    cp = count + ByteOf( sp->z );
    dest[*cp].z    = sp->z;
    dest[*cp].face = sp->face;
    (*cp)++;
  }
}


void
transform()
{
  w_NODE    *node, *from, *to;
  c_OBJECT  *obj, *o1, *o2;
  c_VERTEX  *v1, *v2, *vert;
  c_FACE    *f1, *f2, *face;
  c_LIGHT   *lit;
  c_VECTOR  vec, sight;
  c_MATRIX  objmat, normat, invobjmat;
  float     alpha, ratio, dot, z;
  int32     i, zclip, sort;
  int32     vertFlags, faceFlags;

  float  hx = camera->sizeX / 2.0;  
  float  hy = camera->sizeY / 2.0;  

  float  tanfov = tan( (camera->fov / 2.0) / 180 * M_PI );
  float  ratioX = 1.0 / tanfov;
  float  ratioY = 1.0 / (tanfov * camera->sizeY / camera->sizeX);


  numVisibleFaces = 0;
  numLights = 0;
  
  // transform lights
  for ( node = scene->world; node; node = node->next ) {
    if (node->type == clax_obj_light) {
      lit = (c_LIGHT *)node->object;
      mat_mulvecto( camera->matrix, &lit->pos, &lit->ppos );
      if ( lit->flags & clax_light_spot ) {
        mat_mulvec (camera->matrix, &lit->target, &lit->ptarget);
        vec_sub( &lit->ptarget, &lit->ppos, &vec );
        vec_fastnormalize( &vec, &lit->dir );
      }
      lightList[numLights] = lit;
      numLights++;
    }
  }

  c_LIGHT**  localLightList = new c_LIGHT * [numLights];
  int        numLocalLights;


  // transform objects  
  for ( node = scene->world; node; node = node->next ) {

    if ( node->type == clax_obj_object ) {
      obj = (c_OBJECT *)node->object;

      if (((obj->flags & clax_obj_hidden) == 0)&&
           (obj->flags & clax_obj_chidden) == 0) {

        mat_mul( camera->matrix, obj->matrix, objmat );
        
        mat_mulvec( objmat, &obj->center, &vec );

        zclip = 1;
        vertFlags = 0;
        faceFlags = 0;

        // check objects visibility
        if ( (vec.z + obj->radius) < IE_minZ )
          // object is completely behind near clipping plane
          continue;
        else if ( (vec.z - obj->radius) > IE_minZ )
          // object does not need near plane clipping
          zclip = 0;
        else
          // object needs near plane clipping
          zclip = 1;

          
        if ( (vec.z - obj->radius) > IE_globals.maxZ )
          // object is too far.
          continue;

        // depth cue this object?
        if ( (vec.z + obj->radius) > IE_globals.depthCue ) {
          vertFlags |= clax_vertex_depthcue;
          faceFlags |= clax_face_depthcue;
        }

        // depth fog this object?
        if ( (IE_globals.fogEnabled) && ((vec.z + obj->radius) > IE_globals.fogDepth) ) {
          vertFlags |= clax_vertex_fog;
          faceFlags |= clax_face_fog;
        }

        // object inside viewcone?
        if ( insideViewCone( ratioX, ratioX, &vec, obj->radius ) != 0 )
          continue; // nop

        mat_inverse_c( obj->matrix, invobjmat );
        
        if ( obj->flags & clax_obj_nolight )
          faceFlags |= clax_face_nolight;

        if ( obj->flags & clax_obj_morph ) {

          clax_byid( obj->morph.from, &from );
          clax_byid( obj->morph.to, &to );
          o1 = (c_OBJECT *)from->object;
          o2 = (c_OBJECT *)to->object;
          v1 = o1->vertices;
          v2 = o2->vertices;
          f1 = o1->faces;
          f2 = o2->faces;
          alpha = obj->morph.alpha;
          vec_lerp( &o1->bbox.min, &o2->bbox.min, alpha, &vec );
          mat_mulvec( obj->matrix, &vec, &obj->pbbox.min );
          vec_lerp( &o1->bbox.max, &o2->bbox.max, alpha, &vec );
          mat_mulvec( obj->matrix, &vec, &obj->pbbox.max );

          if ( zclip ) {
            for ( i = 0; i < obj->numverts; i++ ) {
              vec_lerp( &v1[i].vert, &v2[i].vert, alpha, &vec );
              mat_mulvecto( objmat, &vec, &obj->vertices[i].pvert );
              obj->vertices[i].flags = 0;    // clear zclip flag
    
              if ( obj->vertices[i].pvert.z < IE_minZ )
                obj->vertices[i].flags |= clax_vertex_zclip;   // set zclip flag
              else {
                obj->vertices[i].flags |= vertFlags;
                obj->vertices[i].sz = 1.0 / obj->vertices[i].pvert.z;
                obj->vertices[i].sx = hx + obj->vertices[i].pvert.x * camera->perspX * obj->vertices[i].sz;
                obj->vertices[i].sy = hy - obj->vertices[i].pvert.y * camera->perspY * obj->vertices[i].sz;
              }
            }
          } else {
            for ( i = 0; i < obj->numverts; i++ ) {
              vec_lerp( &v1[i].vert, &v2[i].vert, alpha, &vec );
              mat_mulvecto( objmat, &vec, &obj->vertices[i].pvert );
              obj->vertices[i].flags = 0;    // clear zclip flag
              obj->vertices[i].flags |= vertFlags;
              obj->vertices[i].sz = 1.0 / obj->vertices[i].pvert.z;
              obj->vertices[i].sx = hx + obj->vertices[i].pvert.x * camera->perspX * obj->vertices[i].sz;
              obj->vertices[i].sy = hy - obj->vertices[i].pvert.y * camera->perspY * obj->vertices[i].sz;
            }
          }

          for ( i = 0; i < obj->numverts; i++ ) {
           vec_lerp( &v1[i].norm, &v2[i].norm, alpha, &vec );
            mat_mulnorm( objmat, &vec, &obj->vertices[i].pnorm );
          }

          // world space backface culling

          for ( i = 0; i < obj->numfaces; i++ ) {
            vec_lerp ( &f1[i].norm, &f2[i].norm, alpha, &vec );
            mat_mulnormto ( objmat, &vec, &obj->faces[i].pnorm );

            dot = vec_dotunit( &obj->faces[i].p[0]->pvert, &obj->faces[i].pnorm );
            if ( ispos( &dot ) ) {
              obj->faces[i].p[0]->flags |= clax_vertex_visible;
              obj->faces[i].p[1]->flags |= clax_vertex_visible;
              obj->faces[i].p[2]->flags |= clax_vertex_visible;
              obj->faces[i].flags = clax_face_visible | faceFlags;
              sortList[numVisibleFaces].face = &obj->faces[i];
              numVisibleFaces++;
            }
          }

        } else {

          // Object space backface culling

          face = obj->faces;
          mat_mulvecto( invobjmat, &camera->pos, &vec );
          for ( i = obj->numfaces; i > 0 ; --i ) {
            sight.x = face->p[0]->vert.x - vec.x;
            sight.y = face->p[0]->vert.y - vec.y;
            sight.z = face->p[0]->vert.z - vec.z;

            dot = vec_dotunit( &face->norm, &sight );
            if ( ispos( &dot ) ) {
              face->p[0]->flags = clax_vertex_visible;
              face->p[1]->flags = clax_vertex_visible;
              face->p[2]->flags = clax_vertex_visible;
              face->flags = clax_face_visible | faceFlags;

              sortList[numVisibleFaces].face = face;
              numVisibleFaces++;
            } 
            face++;
          }

          vert = obj->vertices;
          if ( zclip ) {
            for ( i = obj->numverts; i > 0 ; --i ) {
              if ( vert->flags & clax_vertex_visible ) {
                mat_mulvecto( objmat, &vert->vert, &vert->pvert );
                mat_mulnormto( objmat, &vert->norm, &vert->pnorm );
                vert->flags = clax_vertex_visible;    // clear flag

                if ( vert->pvert.z < IE_minZ )
                  vert->flags |= clax_vertex_zclip;   // set zclip flag
                else {
                  vert->flags |= vertFlags;
                  vert->sz = 1.0 / vert->pvert.z;
                  vert->sx = hx + vert->pvert.x * camera->perspX * vert->sz;
                  vert->sy = hy - vert->pvert.y * camera->perspY * vert->sz;
                }
              }
              vert++;
            }
          } else {
            for ( i = obj->numverts; i > 0 ; --i ) {
              if ( vert->flags & clax_vertex_visible ) {
                mat_mulvecto( objmat, &vert->vert, &vert->pvert );
                mat_mulnormto( objmat, &vert->norm, &vert->pnorm );
                vert->flags = clax_vertex_visible;
                vert->flags |= vertFlags;
                vert->sz = 1.0 / vert->pvert.z;
                vert->sx = hx + vert->pvert.x * camera->perspX * vert->sz;
                vert->sy = hy - vert->pvert.y * camera->perspY * vert->sz;
              }
              vert++;
            }
          }

        } // if morph
      } // if object visible
    }
  }

  int  currVisible = numVisibleFaces;
  numVisibleFaces = 0;

  for ( i = 0; i < currVisible; i++ ) {
    face = sortList[i].face;

    // farthest point
/*     z = face->p[0]->pvert.z;
    if (*((long *)&face->p[1]->pvert.z) > *((long *)&z) )
      z = face->p[1]->pvert.z;
    if (*((long *)&face->p[2]->pvert.z) > *((long *)&z) )
      z = face->p[2]->pvert.z;*/

     z = (face->p[0]->pvert.z + face->p[1]->pvert.z + face->p[2]->pvert.z ) * (1.0 / 3.0);

    sortList[numVisibleFaces].z = *((ulong*)&z);
    numVisibleFaces++;
  }

}


void
sort()
{
  radix( 0, numVisibleFaces, sortList, tmpList );
  radix( 8, numVisibleFaces, tmpList, sortList );
  radix( 16, numVisibleFaces, sortList, tmpList );
  radix( 24, numVisibleFaces, tmpList, sortList );
}


void
processLights()
{
  float      dot, depthRatio, fogRatio, s, falloff, lenght;
  float      lightIntensity;
  c_LIGHT*   light;
  c_VERTEX*  vert;
  c_VECTOR   vec;

  depthRatio = 1.0 / (IE_globals.maxZ - IE_globals.depthCue);
  fogRatio = 1.0 / (IE_globals.maxZ - IE_globals.fogDepth);

  for ( int i = 0; i < numVisibleFaces; i++ ) {

    // Test fog and depth cueuing more accurately.
    if ( !(sortList[i].z > *(long *)&IE_globals.depthCue) )
      sortList[i].face->flags &= ~clax_face_depthcue;

    if ( !(sortList[i].z > *(long *)&IE_globals.fogDepth) )
      sortList[i].face->flags &= ~clax_face_fog;

    if ( sortList[i].face->flags & clax_face_nolight ) {

      for ( int k = 0; k < 3; k++ ) {
        vert = sortList[i].face->p[k];
        if ( vert->flags & clax_vertex_visible ) {


          if ( *((long *)&vert->pvert.z) < *((long *)&IE_globals.maxZ) ) {
            // perform depth cueuing
            vert->intensity = 0.70;
            vert->fog = 0.0;
            if ( (vert->flags & clax_vertex_depthcue) &&
                 (*((long *)&vert->pvert.z) > *((long *)&IE_globals.depthCue)) ) {
              s = (IE_globals.maxZ - vert->pvert.z) * depthRatio;
              vert->intensity = s * vert->intensity;
            }
            // fog
            if ( (vert->flags & clax_vertex_fog) &&
                 (*((long *)&vert->pvert.z) > *((long *)&IE_globals.fogDepth)) ) {
              vert->fog = 1.0 - (IE_globals.maxZ - vert->pvert.z) * fogRatio;
            }
          } else {
            vert->flags |= clax_vertex_maxz;
            vert->intensity = 0.0;
            vert->fog = 1.0;
          }

          vert->flags ^= clax_vertex_visible;
        }

      }
    } else {

      // full lighting
      for ( int k = 0; k < 3; k++ ) {
        vert = sortList[i].face->p[k];
        if ( vert->flags & clax_vertex_visible ) {


          if ( *((long *)&vert->pvert.z) < *((long *)&IE_globals.maxZ) ) {
            vert->intensity = 0.0;
            vert->fog = 0.0;
            for ( int j = 0; j < numLights; j++ ) {
              light = lightList[j];
              
              lightIntensity = 0.301 * light->color.r +
                               0.586 * light->color.g +
                               0.113 * light->color.b;

              vec_sub( &vert->pvert, &light->ppos, &vec );
              lenght = vec_fastlength( &vec );
              if ( (light->flags & clax_light_attenuate) &&
                   (*(long *)&lenght > *(long *)&light->outerrange) )
                continue;

              vec_scale( &vec, 1.0 / lenght, &vec );
              dot = vec_dotunit( &vec, &vert->pnorm );
            
              if ( (light->flags & clax_light_attenuate) &&
                   (*(long *)&lenght > *(long *)&light->innerrange) )
                dot *= (light->outerrange - lenght) / (light->outerrange - light->innerrange);
            
              if ( ispos( &dot ) ) {
                dot *= lightIntensity;
                if ( light->flags & clax_light_spot ) {

                  // spot light
                  falloff = vec_dotunit( &vec, &light->dir );
                  if ( (*((long *)&falloff) > *((long *)&light->falloff)) ) {
                    if ( *((long *)&falloff) < *((long *)&light->hotspot) )
                     vert->intensity += dot * ((light->falloff - falloff) * light->falloffinv);
                    else
                       vert->intensity += dot;
                  }
                } else {

                  // omni light
                  vert->intensity += dot;
                }
              }
            }

        
            // depth cueuing
            if ( (vert->flags & clax_vertex_depthcue) &&
                 (*((long *)&vert->pvert.z) > *((long *)&IE_globals.depthCue)) ) {
              s = (IE_globals.maxZ - vert->pvert.z) * depthRatio;
              vert->intensity = s * vert->intensity;
            }
            // fog
            if ( (vert->flags & clax_vertex_fog) &&
                 (*((long *)&vert->pvert.z) > *((long *)&IE_globals.fogDepth)) ) {
              vert->fog = 1.0 - (IE_globals.maxZ - vert->pvert.z) * fogRatio;
            }

            // clamp intensity
            if ( vert->intensity > 1.0 )
              vert->intensity = 1.0;
          } else {
            vert->flags |= clax_vertex_maxz;
            vert->intensity = 0.0;
            vert->fog = 1.0;
          }

          vert->flags ^= clax_vertex_visible;
        }
      }
    }
  }
}



void
drawPolys()
{
  char*     texture;
  int       npoints;
  float     ratio, intensity, fog, u, v;
  c_VECTOR  I;
  c_VERTEX  *start, *end, *vlist[3];
  c_FACE*   face;
  vertexuvfloat  points[5];

  float  hx = camera->sizeX / 2.0;  
  float  hy = camera->sizeY / 2.0;  


  // draw backwards 'cos radix sorts 'em that way..

  for ( int i = numVisibleFaces - 1; i >= 0 ; i-- ) {
    
      {
      face = sortList[i].face;

      if ( (face->p[0]->flags & clax_vertex_maxz) &&
           (face->p[1]->flags & clax_vertex_maxz) &&
           (face->p[2]->flags & clax_vertex_maxz) )
        continue;

      if ( ((face->p[0]->flags & clax_vertex_zclip) == 0) &&
           ((face->p[1]->flags & clax_vertex_zclip) == 0) &&
           ((face->p[2]->flags & clax_vertex_zclip) == 0) ) {
        // this poly is visible                  

        texture = 0;

        if ( face->pmat->reflection.data ) {
          points[0].x = face->p[0]->sx;
          points[0].y = face->p[0]->sy;
          points[0].i = face->p[0]->intensity * 62.5;
          points[0].f = face->p[0]->fog * 62.5;
          points[0].u = (face->p[0]->pnorm.x + 1.0) * 127.5;
          points[0].v = (face->p[0]->pnorm.y + 1.0) * 127.5;

          points[1].x = face->p[1]->sx;
          points[1].y = face->p[1]->sy;
          points[1].i = face->p[1]->intensity * 62.5;
          points[1].f = face->p[1]->fog * 62.5;
          points[1].u = (face->p[1]->pnorm.x + 1.0) * 127.5;
          points[1].v = (face->p[1]->pnorm.y + 1.0) * 127.5;

          points[2].x = face->p[2]->sx;
          points[2].y = face->p[2]->sy;
          points[2].i = face->p[2]->intensity * 62.5;
          points[2].f = face->p[2]->fog * 62.5;
          points[2].u = (face->p[2]->pnorm.x + 1.0) * 127.5;
          points[2].v = (face->p[2]->pnorm.y + 1.0) * 127.5;
          
          texture = face->pmat->reflection.data;
        } else {
          points[0].x = face->p[0]->sx;
          points[0].y = face->p[0]->sy;
          points[0].i = face->p[0]->intensity * 62.5;
          points[0].f = face->p[0]->fog * 62.5;
          points[0].u = face->p[0]->u * 255.9;
          points[0].v = face->p[0]->v * 255.9;

          points[1].x = face->p[1]->sx;
          points[1].y = face->p[1]->sy;
          points[1].i = face->p[1]->intensity * 62.5;
          points[1].f = face->p[1]->fog * 62.5;
          points[1].u = face->p[1]->u * 255.9;
          points[1].v = face->p[1]->v * 255.9;

          points[2].x = face->p[2]->sx;
          points[2].y = face->p[2]->sy;
          points[2].i = face->p[2]->intensity * 62.5;
          points[2].f = face->p[2]->fog * 62.5;
          points[2].u = face->p[2]->u * 255.9;
          points[2].v = face->p[2]->v * 255.9;

          texture = face->pmat->texture.data;
        }

        if ( texture )
          if ( (face->flags & clax_face_nolight) && (face->flags & clax_face_fog) )
            DrawFogTextureTriangle( points, texture );
          else if ( (face->flags & clax_face_nolight) && (face->flags & clax_face_depthcue) )
            DrawShadedTextureTriangle( points, texture );
          else if ( face->flags & clax_face_nolight )
            DrawTextureTriangle( points, texture );
          else if ( face->flags & clax_face_fog )
            DrawFogShadedTextureTriangle( points, texture );
          else
            DrawShadedTextureTriangle( points, texture );
        else
          if ( face->flags & clax_face_fog )
            DrawFogGouraudTriangle( points, face->pmat->color_ramp );
          else
            DrawGouraudTriangle( points, face->pmat->color_ramp );

      } else if ( ((face->p[0]->flags & clax_vertex_zclip) != 0) &&
                  ((face->p[1]->flags & clax_vertex_zclip) != 0) &&
                  ((face->p[2]->flags & clax_vertex_zclip) != 0) ) {
        // this poly is not visible
        continue;
      } else {
        // this poly must be clipped

        npoints = 0;      
        vlist[0] = face->p[0];
        vlist[1] = face->p[1];
        vlist[2] = face->p[2];
      
        start = vlist[2];
        for ( int j = 0; j < 3; j++ ) {
          end = vlist[j];

          if ( (end->flags & clax_vertex_zclip) == 0 ) {
            if ( (start->flags & clax_vertex_zclip) == 0 ) {
              // polygon edge is completely inside
              points[npoints].x = end->sx;
              points[npoints].y = end->sy;
              points[npoints].i = end->intensity * 62.5;
              points[npoints].f = end->fog * 62.5;
              points[npoints].u = end->u * 255.9;
              points[npoints].v = end->v * 255.9;
              npoints++;
            } else {
              // end point is inside start point is outside
              ratio = (IE_minZ - start->pvert.z) / (end->pvert.z - start->pvert.z);
              intensity = start->intensity + ((end->intensity - start->intensity) * ratio);
              fog = start->fog + ((end->fog - start->fog) * ratio);
              I.x = start->pvert.x + ((end->pvert.x - start->pvert.x) * ratio);
              I.y = start->pvert.y + ((end->pvert.y - start->pvert.y) * ratio);

              u = start->u + ((end->u - start->u) * ratio);
              v = start->v + ((end->v - start->v) * ratio);

              points[npoints].x = hx + I.x * camera->perspX * (1.0 / IE_minZ);
              points[npoints].y = hy - I.y * camera->perspY * (1.0 / IE_minZ);
              points[npoints].i = intensity * 62.5;
              points[npoints].f = fog * 62.5;
              points[npoints].u = u * 255.9;
              points[npoints].v = v * 255.9;
              npoints++;

              points[npoints].x = end->sx;
              points[npoints].y = end->sy;
              points[npoints].i = end->intensity * 62.5;
              points[npoints].f = end->fog * 62.5;
              points[npoints].u = end->u * 255.9;
              points[npoints].v = end->v * 255.9;
              npoints++;

            }
          } else {
            if ( (start->flags & clax_vertex_zclip) == 0 ) {
              // start point is inside end point is outside
              ratio = (IE_minZ - end->pvert.z) / (start->pvert.z - end->pvert.z);
              intensity = end->intensity + ((start->intensity - end->intensity) * ratio);
              fog = end->fog + ((start->fog - end->fog) * ratio);
              I.x = end->pvert.x + ((start->pvert.x - end->pvert.x) * ratio);
              I.y = end->pvert.y + ((start->pvert.y - end->pvert.y) * ratio);
              u = end->u + ((start->u - end->u) * ratio);
              v = end->v + ((start->v - end->v) * ratio);

              points[npoints].x = hx + I.x * camera->perspX * (1.0 / IE_minZ);
              points[npoints].y = hy - I.y * camera->perspY * (1.0 / IE_minZ);
              points[npoints].i = intensity * 62.5;
              points[npoints].f = fog * 62.5;
              points[npoints].u = u * 255.9;
              points[npoints].v = v * 255.9;
              npoints++;
            } else {
              // polygon edge is completely outside -> do nothing
            }
          }
          start = end;
        }
      
        if ( face->pmat->texture.data )
          if ( (face->flags & clax_face_nolight) && (face->flags & clax_face_fog) )
            DrawFogTextureTriangle( points, face->pmat->texture.data );
          else if ( (face->flags & clax_face_nolight) && (face->flags & clax_face_depthcue) )
            DrawShadedTextureTriangle( points, face->pmat->texture.data );
          else if ( face->flags & clax_face_nolight )
            DrawTextureTriangle( points, face->pmat->texture.data );
          else if ( face->flags & clax_face_fog )
            DrawFogShadedTextureTriangle( points, face->pmat->texture.data );
          else
            DrawShadedTextureTriangle( points, face->pmat->texture.data );
        else
          if ( face->flags & clax_face_fog )
            DrawFogGouraudTriangle( points, face->pmat->color_ramp );
          else
            DrawGouraudTriangle( points, face->pmat->color_ramp );

        if ( npoints == 4 ) {
          points[4].x = points[0].x;
          points[4].y = points[0].y;
          points[4].i = points[0].i;
          points[4].f = points[0].f;
          points[4].u = points[0].u;
          points[4].v = points[0].v;

          if ( face->pmat->texture.data )
            if ( (face->flags & clax_face_nolight) && (face->flags & clax_face_fog) )
              DrawFogTextureTriangle( &points[2], face->pmat->texture.data );
            else if ( (face->flags & clax_face_nolight) && (face->flags & clax_face_depthcue) )
              DrawShadedTextureTriangle( &points[2], face->pmat->texture.data );
            else if ( face->flags & clax_face_nolight )
              DrawTextureTriangle( &points[2], face->pmat->texture.data );
            else if ( face->flags & clax_face_fog )
              DrawFogShadedTextureTriangle( &points[2], face->pmat->texture.data );
            else
              DrawShadedTextureTriangle( &points[2], face->pmat->texture.data );
          else
            if ( face->flags & clax_face_fog )
              DrawFogGouraudTriangle( &points[2], face->pmat->color_ramp );
            else
              DrawGouraudTriangle( &points[2], face->pmat->color_ramp );
        }
      }
    }
  }
}

void
loadPalette( char* name )
{
  FILE*  fp;
  if ( (fp = fopen( name, "rb" )) != 0 ) {
    fseek( fp, -(256 * 3), SEEK_END );
    fread( IE_globals.palette, 256 * 3, 1, fp );
    fclose( fp );
  }
  for ( int i = 0; i < 256; i++ ) {
    IE_globals.palette[i * 3] >>= 2;
    IE_globals.palette[i * 3 + 1] >>= 2;
    IE_globals.palette[i * 3 + 2] >>= 2;
  }
}


void
closeIE()
{
//  printf("freeing IE..\n");
  if ( sortList ) delete [] sortList;
  if ( tmpList ) delete [] tmpList;
  if ( lightList ) delete [] lightList;
  freeGlobals();
}

int
initIE()
{
  c_OBJECT *obj;
  w_NODE   *node;
  int  err;

  clax_getactive_scene( &scene );
  clax_getactive_camera( &camera );
  
  maxfaces = 0;
  maxverts = 0;
  numLights = 0;

  for ( node = scene->world; node; node = node->next ) {
    if ( node->type == clax_obj_object ) {
      obj = (c_OBJECT *)node->object;
      maxfaces += obj->numfaces;
      maxverts += obj->numverts;
    } else if ( node->type == clax_obj_light )
      numLights++;
  }
  
  sortList = new Sort [maxfaces + numLights];
  tmpList =  new Sort [maxfaces];
  lightList = new c_LIGHT* [numLights];

  if ( (sortList == 0) || (tmpList == 0) || (lightList == 0) ) {
    closeIE();
    return clax_err_nomem;
  }

  err = initGlobals();
  if ( err != clax_err_ok ) {
    closeIE();
    return err;
  }

  loadPalette( "color.pcx" );

  if ( initOutBuffer( (int)camera->sizeX, (int)camera->sizeY ) == 0 ) {
    closeIE();
    return clax_err_nomem;
  }

  IE_globals.clipTop = 15.0;
  IE_globals.clipBottom = 165.0;
  IE_globals.clipLeft = 0.0;
  IE_globals.clipRight = 320.0;

/*
  err = initMaterials( scene, 1 );
  if ( err != clax_err_ok ) {
    closeIE();
    return err;
  }
  saveFogTable( "fog.tbl" );
  saveShadeTable( "shade.tbl" );
*/

  err = initMaterials( scene, 0 );
  if ( err != clax_err_ok ) {
    closeIE();
    return err;
  }
  loadFogTable( "fog.tbl" );
  loadShadeTable( "shade.tbl" );


  return clax_err_ok;
}
