
(*
** Sound Recognition v1.0 (SVGA version)
** Written by Bas van Gaalen and Sandor van Kollenburg in
**   September and November '96 as a schoolproject.
**
** Use numberkeys on gray keypad to change the record-
**   and detectionlevels. And press L to switch between
**   Detection mode and Learning mode.
*)

{$q- Overflowchecking off }
{$r- Rangechecking off }
{$x+ Extended syntax on }
{$s- Stackchecking off }
{$n+ Use coprocessor }
{$e- Use emulation }

{$m 32768,0,655360}

program sound_recognition;

uses
  dos,crt,_fft,types,svga,gui,u_ffpcx,u_pal,u_kb;

const
  sprpath:pathstr='spr\';                             { Soundprint directory }
  sprmask:pathstr='inst####.spr';
  leadingchar:char='0';
  signalaxis=70;                                         { Drawing constants }
  signalrange=60;
  grid=10;
  plotzoom=2;

(* Instrument-check constants *)
  pctcheck=0.05;                           { Take pctcheck percent of signal }
  check=0.10;           { If check is large, ir may choose wrong instruments }

type
  commentstr=string[20];
  shortstr=commentstr;
  sprptr=^sprrec;
  sprrec=record                                          { Soundprint record }
    name:commentstr;
    print:magstruc;
    next:sprptr;
  end;

var
  fspr,cspr,lspr:sprptr;          { Soundprint linked-list control variables }
  old1c,                                           { Pointer to timerhandler }
  oldexit:pointer;                                   { Pointer to exit-chain }
  nofsprs:word;                               { Number of soundprints in mem }

(* Global procedures/functions ---------------------------------------------*)

procedure clearstruc(var struc; size:word);            { MAKE 32-BIT CODE!!! }
begin
  fillchar(struc,size,0);
end;

function lz(num:longint; fld:byte):shortstr;
var
  tmp:shortstr;
  i:byte;
begin
  str(num:fld,tmp);
  if fld>1 then begin
    i:=1;
    while tmp[i]=' ' do begin
      tmp[i]:=leadingchar;
      inc(i);
    end;
  end;
  lz:=tmp;
end;

(* Meter draw routines -----------------------------------------------------*)

procedure draw_rv(rv:byte); { Record Volume }
var
  barlen:word;
begin
  barlen:=round(82*rv/7);
  fbox(11,yres-108,19,yres-26-barlen,black);
  fbox(11,yres-25-barlen,19,yres-25,blue);
end;

procedure draw_dv(dv:byte); { Detection Volume }
var
  barlen:word;
begin
  barlen:=round(82*dv/7);
  fbox(31,yres-108,39,yres-26-barlen,black);
  fbox(31,yres-25-barlen,39,yres-25,blue);
end;

procedure draw_dt(dt:byte); { Detection Treshold }
var
  barlen:word;
begin
  barlen:=round(82*dt/15);
  fbox(51,yres-108,59,yres-26-barlen,black);
  fbox(51,yres-25-barlen,59,yres-25,blue);
end;

(* Plot/Display routines ---------------------------------------------------*)

procedure plotsignal(pd,d:sampletype; maxvals,y:word; pc,c:byte);
var
  i,max:word;
  tmp:integer;
  col:byte;
begin
  { Draw horizontal axis }
  max:=maxvals;
  if max>xres-1 then max:=xres-1;
  { Draw the signal }
  for i:=1 to max do begin
    tmp:=round(pd[slack+i])-128;
    if tmp<-signalrange then tmp:=-signalrange
    else if tmp>signalrange then tmp:=signalrange;
    if ((y+tmp) mod grid=0) or (i mod grid=0) then col:=darkgray else col:=pc;
    if tmp=0 then col:=lightgray;
    putpixel(i,y+tmp,col);

    tmp:=round(d[slack+i])-128;
    if tmp<-signalrange then tmp:=-signalrange
    else if tmp>signalrange then tmp:=signalrange;
    putpixel(i,y+tmp,c);
  end;
end;

procedure plotmag(pm,m:magstruc; maxvals,x,y:word; zf,pc,c:byte);
var
  i,j:word;
begin
  { Draw the spectrum }
  for i:=1 to maxvals do
    for j:=0 to zf-1 do begin
      verline((i-1)*zf+j+x,y-round(pm[i]),y,pc);
      verline((i-1)*zf+j+x,y-round(m[i]),y,c);
    end;
end;

(* Global exit-procedure ---------------------------------------------------*)

procedure newexit; far;
var
  tspr:sprptr;
  msg:string[80];
begin
  { Restore exit-chain }
  exitproc:=oldexit;
  { Dispose reserved memory for Soundprints }
  cspr:=fspr;
  while cspr<>lspr do begin
    tspr:=cspr;
    cspr:=cspr^.next;
    dispose(tspr);
  end;
  { Optionally display error message }
  if (exitcode>0) and (erroraddr=nil) then begin
    case exitcode of
       2:msg:='No VESA support';
       3:msg:='Requested VESA mode not supported';
      10:msg:='Fontfile not found';
      11:msg:='Header of fontfile corrupt - not a valid font';
      12:msg:='Error reading data of fontfile';
    else msg:='Unknown error...';
    end;
    writeln('Error: ',msg);
  end;
end;

(* Initialization ----------------------------------------------------------*)

procedure init;
type
  bufptr=^buftyp;
  buftyp=array[0..65000] of byte;
var
  f:file;
  di:searchrec;
  pal:pal_type;
  buf:bufptr;
  i:word;
begin
  { Add new exit-procedure to exit-chain }
  oldexit:=exitproc;
  exitproc:=@newexit;

  { Detect GUI-errorcodes }
  if GUI_Error<>GUI_Err_Okay then halt(GUI_Error);

  { 'Create' credits area }
  horline(0,yres-115,xres-1,white);
  horline(1,yres-114,xres-2,white);
  verline(0,yres-114,yres-1,white);
  verline(1,yres-114,yres-1,white);
  horline(1,yres-1,xres-1,darkgray);
  horline(2,yres-2,xres-2,darkgray);
  verline(xres-1,yres-114,yres-2,darkgray);
  verline(xres-2,yres-113,yres-3,darkgray);
  fbox(2,yres-113,xres-3,yres-3,lightgray);
  horline(xres-110,yres-109,xres-9,darkgray);
  verline(xres-110,yres-108,yres-8,darkgray);
  horline(xres-110,yres-7,xres-8,white);
  verline(xres-8,yres-109,yres-8,white);
  sdrawstr(260,yres-95,'Sound Recognition v1.0',yellow,darkgray,mainfont);
  sdrawstr(260,yres-75,'Design and Implementation by:',white,darkgray,mainfont);
  sdrawstr(280,yres-61,'Bas van Gaalen & Sandor van Kollenburg',white,darkgray,mainfont);
  sdrawstr(260,yres-41,'Soundblaster hardware routines by:',white,darkgray,mainfont);
  sdrawstr(280,yres-27,'Romesh Prakashpalan',white,darkgray,mainfont);

  { Load and display image }
  getmem(buf,4+100*100);
  assign(f,'sr.raw');
  reset(f,1);
  blockread(f,pal,768);
  blockread(f,buf^[4],100*100);
  close(f);
  setpal(pal);
  buf^[0]:=100; buf^[1]:=0;
  buf^[2]:=100; buf^[3]:=0;
  drawimage(xres-109,yres-108,buf^);
  freemem(buf,4+100*100);

  { 'Create' information area }
  sdrawstr(80,yres-100,'rv = record volume',lightgreen,darkgray,mainfont);
  sdrawstr(80,yres-86,'dv = detection volume',lightgreen,darkgray,mainfont);
  sdrawstr(80,yres-72,'dt = detection treshold',lightgreen,darkgray,mainfont);
  sdrawstr(80,yres-52,'''L'' = Switch between',lightcyan,darkgray,mainfont);
  sdrawstr(80,yres-38,'        Learning mode and',lightcyan,darkgray,mainfont);
  sdrawstr(80,yres-24,'        Detection mode',lightcyan,darkgray,mainfont);

  { Level-meters }
  sdrawstr(10,yres-20,'rv',white,darkgray,mainfont);
  sdrawstr(30,yres-20,'dv',white,darkgray,mainfont);
  sdrawstr(50,yres-20,'dt',white,darkgray,mainfont);
  for i:=0 to 2 do begin
    horline(10+i*20,yres-109,19+i*20,darkgray);
    verline(10+i*20,yres-108,yres-25,darkgray);
    horline(10+i*20,yres-24,20+i*20,white);
    verline(20+i*20,yres-109,yres-24,white);
  end;
  draw_rv(recordvolume);
  draw_dv(detectvolume);
  draw_dt(detecttreshold);

  { Drawgrid }
  i:=signalaxis-signalrange;
  while i<=signalaxis+signalrange do begin
    horline(0,i,xres-1,darkgray);
    inc(i,grid);
  end;
  i:=0;
  while i<=xres-1 do begin
    verline(i,signalaxis-signalrange,signalaxis+signalrange,darkgray);
    inc(i,grid);
  end;
  horline(0,signalaxis,xres-1,lightgray);
  putpixel(0,signalaxis-detecttreshold,white);
  putpixel(0,signalaxis+detecttreshold,white);

  { Draw boxes }
  drawstr(80,150,'Input Signal',white,mainfont);
  drawstr(360,150,'Reference Signal',white,mainfont);
  horline(79,251,80+plotzoom*maxfpr,darkgray);
  horline(79,250-81,80+plotzoom*maxfpr,darkgray);
  verline(79,250-80,250,darkgray);
  verline(80+plotzoom*maxfpr,250-80,250,darkgray);
  horline(359,251,360+plotzoom*maxfpr,darkgray);
  horline(359,250-81,360+plotzoom*maxfpr,darkgray);
  verline(359,250-80,250,darkgray);
  verline(360+plotzoom*maxfpr,250-80,250,darkgray);

  { Read the Soundprint data }
  nofsprs:=0;
  new(fspr);
  fspr^.next:=nil;
  cspr:=fspr;
  findfirst(sprpath+'*.spr',anyfile,di);
  while doserror=0 do begin
    {
    if fspr=nil then begin
      new(fspr);
      cspr:=fspr;
    end
    else begin
      new(cspr^.next);
      cspr:=cspr^.next;
    end;
    cspr^.next:=nil;
    lspr:=cspr;
    }
    assign(f,sprpath+di.name);
    reset(f,1);
    blockread(f,cspr^.name,sizeof(commentstr));
    blockread(f,cspr^.print,sizeof(magstruc));
    close(f);
    new(cspr^.next);
    cspr:=cspr^.next;
    findnext(di);
    inc(nofsprs);
  end;
  lspr:=cspr;
  drawstr(xres-20,yres-130,lz(nofsprs,0),white,mainfont);
end;

(*--------------------------------------------------------------------------*)

function compare(m:magstruc; maxvals:word):commentstr;
var
  cmp:real;
  i:word;
  found:boolean;
begin
  cspr:=fspr;
  found:=false;
  while (not found) and (cspr<>lspr) do begin
    cmp:=0;
    for i:=1 to maxvals do
      cmp:=cmp+abs((m[i]/psize)-(cspr^.print[i]/psize));
    found:=cmp*pctcheck<check;
    if not found then cspr:=cspr^.next;
  end;
  if found then compare:=cspr^.name else compare:='';
end;

(* SuperVGA input procedure ------------------------------------------------*)

procedure svgainput(x,y:word; var s:string; max:byte);
var
  tmp:string;
  key:word;
  i:byte;
  leave:boolean;
begin
  fillchar(tmp,sizeof(tmp),0);
  i:=0;
  leave:=false;
  repeat
    while (port[$3da] and 8)=8 do;
    while (port[$3da] and 8)<>8 do;
    fbox(x,y,x+pLength(tmp,mainfont)+10,y+12,black);
    drawstr(x,y,tmp+'.',white,mainfont);
    key:=getekey;
    case key of
      ord(#32),ord(#35)..ord(#165):if i<max then begin
        if i=0 then tmp:=tmp+upcase(chr(key))
        else tmp:=tmp+chr(key);
        inc(i);
      end;
      crsrbs:if i>0 then begin
        delete(tmp,length(tmp),1);
        dec(i);
      end;
      crsrcr:begin
        s:=tmp;
        leave:=true;
      end;
      crsresc:leave:=true;
    end;
  until leave;
end;

(*--------------------------------------------------------------------------*)

procedure main;
var
  f:file;
  outfname:pathstr;
  sndprtname,numstr,name:commentstr;
  data:datastruc;
  pmag,mag,pdmag,dmag:magstruc;
  pbuf,buf:sampletype;
  i,key:word;
  saverv,savedv,savedt:byte;
  learningmode:boolean;
begin
  learningmode:=false;
  clearstruc(buf,sizeof(buf));
  clearstruc(mag,sizeof(mag));
  clearstruc(dmag,sizeof(dmag));
  saverv:=recordvolume;
  savedv:=detectvolume;
  savedt:=detecttreshold;
  repeat
    if detectinput then begin { Detect key or input signal: key }
      key:=getekey;
      case key of
        crsrhome,ord('7'):if recordvolume<7 then begin
          inc(recordvolume);
          draw_rv(recordvolume);
        end;
        crsrend,ord('1'):if recordvolume>1 then begin
          dec(recordvolume);
          draw_rv(recordvolume);
        end;
        crsrup,ord('8'):if detectvolume<7 then begin
          inc(detectvolume);
          draw_dv(detectvolume);
        end;
        crsrdown,ord('2'):if detectvolume>1 then begin
          dec(detectvolume);
          draw_dv(detectvolume);
        end;
        crsrpgup,ord('9'):if detecttreshold<15 then begin
          putpixel(0,signalaxis-detecttreshold,darkgray);
          putpixel(0,signalaxis+detecttreshold,darkgray);
          inc(detecttreshold);
          draw_dt(detecttreshold);
          putpixel(0,signalaxis-detecttreshold,white);
          putpixel(0,signalaxis+detecttreshold,white);
        end;
        crsrpgdn,ord('3'):if detecttreshold>1 then begin
          putpixel(0,signalaxis-detecttreshold,darkgray);
          putpixel(0,signalaxis+detecttreshold,darkgray);
          dec(detecttreshold);
          draw_dt(detecttreshold);
          putpixel(0,signalaxis-detecttreshold,white);
          putpixel(0,signalaxis+detecttreshold,white);
        end;
        crsrleft,ord('4'):begin
          recordvolume:=saverv;
          draw_rv(recordvolume);
        end;
        19456,ord('5'):begin
          detectvolume:=savedv;
          draw_dv(detectvolume);
        end;
        crsrright,ord('6'):begin
          detecttreshold:=savedt;
          draw_dt(detecttreshold);
        end;
        ord('l'),ord('L'):begin
          learningmode:=not learningmode;
          if learningmode then begin
            fbox(359,150,639,251,black);
            drawstr(360,150,'Learningmode active!',white,mainfont);
          end
          else begin
            fbox(359,150,639,251,black);
            drawstr(360,150,'Reference Signal',white,mainfont);
            horline(359,251,360+plotzoom*maxfpr,darkgray);
            horline(359,250-81,360+plotzoom*maxfpr,darkgray);
            verline(359,250-80,250,darkgray);
            verline(360+plotzoom*maxfpr,250-80,250,darkgray);
          end;
        end;
        crsrcr:if learningmode then begin
          drawstr(360,170,'Instrumentname:',white,mainfont);
          fillchar(name,sizeof(name),0);
          svgainput(458,170,name,20);
          fbox(360,170,639,182,black);
          if name<>'' then begin
            outfname:=sprmask;
            numstr:=lz(nofsprs,4);
            move(numstr[1],outfname[pos('#',outfname)],4);
            { Write to mem }
            cspr:=lspr;
            cspr^.name:=name;
            cspr^.print:=mag;
            new(cspr^.next);
            cspr:=cspr^.next;
            lspr:=cspr;

            {new(lspr^.next);
            cspr:=lspr^.next;
            cspr^.name:=name;
            cspr^.print:=mag;
            cspr^.next:=nil;
            lspr:=cspr;}
            { Write to disk }
            assign(f,sprpath+outfname);
            rewrite(f,1);
            blockwrite(f,name,sizeof(name));
            blockwrite(f,mag,sizeof(mag));
            close(f);
            { Update info }
            inc(nofsprs);
            fbox(xres-20,yres-130,xres-1,yres-118,black);
            drawstr(xres-20,yres-130,lz(nofsprs,0),white,mainfont);
          end;
        end;
        ord('S'):begin { Signal Save = Shift-'S' }
          assign(f,'signal.dat');
          rewrite(f,1);
          blockwrite(f,buf,sizeof(buf));
          blockwrite(f,mag,sizeof(mag));
          close(f);
        end;
        ord(' '):fbox(100,320,xres-100,332,black);
        crsresc:exit;
      end;
    end
    else begin { ...input signal }
      move(buf,pbuf,sizeof(buf));
      sampler(buf);
      plotsignal(pbuf,buf,fftsize,signalaxis,black,white);
      clearstruc(data,sizeof(data));
      for i:=0 to fftsize-1 do data[i].re:=buf[slack+i];
      fft(data,fftsize);
      move(mag,pmag,sizeof(mag));
      convert(data,mag,maxfpr);
      plotmag(pmag,mag,maxfpr,80,250,plotzoom,black,lightgreen);
      if not learningmode then begin                   { <--- Detection-mode }
        sndprtname:=compare(mag,maxfpr);
        move(dmag,pdmag,sizeof(dmag));
        clearstruc(dmag,sizeof(dmag));
        if sndprtname<>'' then begin
          fbox(100,320,xres-100,332,black);
          drawstr((xres-plength(sndprtname,mainfont)) div 2,320,sndprtname,white,mainfont);
          move(cspr^.print,dmag,sizeof(magstruc));
        end;
        plotmag(pdmag,dmag,maxfpr,360,250,plotzoom,black,lightgreen);
      end
      else begin                                        { <--- Learning-mode }
        waitforend;
      end;
    end;
  until false;
end;

(*--------------------------------------------------------------------------*)

begin
  init;
  main;
end.
