//
// Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

#ifndef __PTC_CONSOLE_H
#define __PTC_CONSOLE_H

// include files
#include "Core/Base.h"
#include "Core/Copy.h"
#include "Core/Mode.h"
#include "Core/Types.h"
#include "Core/Clear.h"
#include "Core/Format.h"
#include "Core/Config.h"
#include "Core/Palette.h"
#include "Win32/Window.h"
#include "Win32/Keyboard.h"
#include "DirectX/Library.h"
#include "DirectX/Display.h"
#include "DirectX/Primary.h"




class Console : public BaseConsole
{
    public:

        // setup
        DLLAPI Console();
        DLLAPI ~Console();

        // console configuration
        DLLAPI virtual void PTCAPI configure(const char file[]);

        // console option string
        DLLAPI virtual bool PTCAPI option(const char option[]);

        // console modes
        DLLAPI virtual const Mode* PTCAPI modes();

        // console management
        DLLAPI virtual void PTCAPI open(const char title[],int pages = 0);
        DLLAPI virtual void PTCAPI open(const char title[],const Format &format,int pages = 0);
        DLLAPI virtual void PTCAPI open(const char title[],int width,int height,const Format &format,int pages = 0);
        DLLAPI virtual void PTCAPI open(const char title[],const Mode &mode,int pages = 0);
        DLLAPI virtual void PTCAPI close();
               
        // synchronization
        DLLAPI virtual void PTCAPI flush();
        DLLAPI virtual void PTCAPI finish();
        DLLAPI virtual void PTCAPI update();
        DLLAPI virtual void PTCAPI update(const Area &area);

        // keyboard input
        DLLAPI virtual bool PTCAPI key();
        DLLAPI virtual Key PTCAPI read();

        // copy to surface
        DLLAPI virtual void PTCAPI copy(BaseSurface &surface);
        DLLAPI virtual void PTCAPI copy(BaseSurface &surface,const Area &source,const Area &destination);

        // memory access
        DLLAPI virtual void* PTCAPI lock();
        DLLAPI virtual void PTCAPI unlock();

        // load pixels to console
        DLLAPI virtual void PTCAPI load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette);
        DLLAPI virtual void PTCAPI load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination);

        // save console pixels
        DLLAPI virtual void PTCAPI save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette);
        DLLAPI virtual void PTCAPI save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination);

        // clear console
        DLLAPI virtual void PTCAPI clear();
        DLLAPI virtual void PTCAPI clear(const Color &color);
        DLLAPI virtual void PTCAPI clear(const Color &color,const Area &area);

        // console palette
        DLLAPI virtual void PTCAPI palette(const Palette &palette);
        DLLAPI virtual const Palette& PTCAPI palette() const;

        // console clip area
        DLLAPI virtual void PTCAPI clip(const Area &area);

        // data access
        DLLAPI virtual int PTCAPI width() const;
        DLLAPI virtual int PTCAPI height() const;
        DLLAPI virtual int PTCAPI pages() const;
        DLLAPI virtual int PTCAPI pitch() const;
        DLLAPI virtual const Area& PTCAPI area() const;
        DLLAPI virtual const Area& PTCAPI clip() const;
        DLLAPI virtual const Format& PTCAPI format() const;
        DLLAPI virtual const char* PTCAPI name() const;
        DLLAPI virtual const char* PTCAPI title() const;
        DLLAPI virtual const char* PTCAPI information() const;

        // extension functions
        DLLAPI HWND PTCAPI window() const;
        DLLAPI LPDIRECTDRAW lpDD() const;
        DLLAPI LPDIRECTDRAW2 lpDD2() const;
        DLLAPI LPDIRECTDRAWSURFACE lpDDS() const;

    private:

        // internal console management routines
        void internal_pre_open_setup(const char title[]);
        void internal_open_fullscreen_start();
        void internal_open_fullscreen(int width,int height,const Format &format);
        void internal_open_fullscreen_finish(int pages);
        void internal_open_windowed(int width,int height,int pages);
        void internal_post_open_setup();
        void internal_reset();
        void internal_close();

        // disallow copy construction
        Console(const Console &console);

        // disallow console assignment
        Console& operator =(const Console &console);

        // console debug checks
        void check_open() const;
        void check_unlocked() const;

        // data
        Mode m_modes[256];
        char m_title[1024];
        char m_information[1024];

        // flags
        bool m_open;
        bool m_locked;

        // output type
        enum Output
        {
            DEFAULT,
            WINDOWED,
            FULLSCREEN
        };

        // option data
        int m_default_width;
        int m_default_height;
        int m_default_pages;
        Format m_default_format;
        Output m_output_mode;

        // objects
        Copy m_copy;

        // Win32 objects
        Win32Window *m_window;
        Win32Keyboard *m_keyboard;

        // DirectX objects
        DirectXLibrary m_library;
        DirectXDisplay m_display;
        DirectXPrimary *m_primary;
};




#endif
