//
// Format class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// include files
#include "Error.h"
#include "Config.h"
#include "Format.h"
#include "Hermes.h"




Format::Format()
{
    // defaults
    m_format.r = 0;
    m_format.g = 0;
    m_format.b = 0;
    m_format.a = 0;
    m_format.bits = 0;
    m_format.indexed = 0;
}


Format::Format(int bits)
{
    // check bits per pixel
    if (bits!=8) throw Error("unsupported bits per pixel");

    // indexed color
    m_format.r = 0;
    m_format.g = 0;
    m_format.b = 0;
    m_format.a = 0;
    m_format.bits = bits;
    m_format.indexed = 1;
}


Format::Format(int bits,int32 r,int32 g,int32 b,int32 a)
{
    // check bits per pixel
    if (bits%8 || bits<=0 || bits>32) throw Error("unsupported bits per pixel");

    // direct color
    m_format.r = r;
    m_format.g = g;
    m_format.b = b;
    m_format.a = a;
    m_format.bits = bits;
    m_format.indexed = 0;
}


Format::Format(const Format &format)
{
#ifdef __COMPILER_SUPPORTS_CONST_CAST__

    // copy constructor
    Hermes_FormatCopy(const_cast<HermesFormat*>(&format.m_format),&m_format);

#else

    // copy constructor
    Hermes_FormatCopy((HermesFormat*)&format.m_format,&m_format);

#endif
}


Format::~Format()
{
    // destructor
}




int32 Format::r() const
{
    // get r
    return m_format.r;
}


int32 Format::g() const
{
    // get g
    return m_format.g;
}


int32 Format::b() const
{
    // get b
    return m_format.b;
}


int32 Format::a() const
{
    // get a
    return m_format.a;
}


int Format::bits() const
{
    // get bits
    return m_format.bits;
}


int Format::bytes() const
{
    // get bytes
    return m_format.bits / 8;
}


bool Format::direct() const
{
    // is direct color?
    return m_format.indexed==0;
}


bool Format::indexed() const
{
    // is indexed color?
    return m_format.indexed==1;
}




Format& Format::operator =(const Format &format)
{
    // check for self assignment
    if (this==&format) throw Error("self assignment is not allowed");

#ifdef __COMPILER_SUPPORTS_CONST_CAST__

    // copy other hermes format to this hermes format
    Hermes_FormatCopy(const_cast<HermesFormat*>(&format.m_format),&m_format);

#else

    // copy other hermes format to this hermes format
    Hermes_FormatCopy((HermesFormat*)&format.m_format,&m_format);

#endif

    // return reference
    return *this;
}


bool Format::operator ==(const Format &format) const
{
#ifdef __COMPILER_SUPPORTS_CONST_CAST__

    // is equal to
    return Hermes_FormatEquals(const_cast<HermesFormat*>(&format.m_format),const_cast<HermesFormat*>(&m_format))==1;

#else

    // is equal to
    return Hermes_FormatEquals((HermesFormat*)&format.m_format,(HermesFormat*)&m_format)==1;

#endif
}


bool Format::operator !=(const Format &format) const
{
    // not equal to
    return !(*this==format);
}
