//
// Surface class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "Error.h"
#include "Surface.h"
#include <memory.h>




Surface::Surface(int width,int height,const Format& format)
{
    // setup data
    m_width = width;
    m_height = height;
    m_format = format;
    m_pixels = 0;

    // calculate size of pixels
    int size = width * height * format.bits()/8;
 
    // check size
    if (!size) throw Error("zero surface size");

    // allocate pixels
    m_pixels = new char8[size];
}


Surface::~Surface()
{
    // free surface
    delete [] m_pixels;
}


void Surface::copy(BaseSurface &surface)
{ surface.load(m_pixels,m_width,m_height,m_format,m_palette);
}


void Surface::copy(BaseSurface &surface,const Area &source,
  const Area &destination)
{ surface.load(m_pixels,m_width,m_height,source,destination,m_format,
	       m_palette);
}


void* Surface::lock()
{
    // lock
    return m_pixels;
}


void Surface::unlock()
{
    // unlock
}


void Surface::load(const void *pixels,int width,int height,
  const Format &format,const int32 palette[])
{
  bool stretch=(m_width!=width)||(m_height!=height);

  Copy::request(format,m_format,stretch);

  Copy::palette(palette);

  // Lock and copy (lock() is hidden somewhere in that line, watch out! :)
  Copy::copy(pixels,0,0,width,height,width*format.bits()/8,
             lock(),0,0,m_width,m_height,pitch());

  unlock();
}


void Surface::load(const void *pixels,int width,int height,
  const Area &source,const Area &destination,const Format &format,
  const int32 palette[])
{
  bool stretch=(source.width()!=destination.width())||
               (source.height()!=destination.height());

  // set copy formats
  Copy::request(format,m_format,stretch);
    
  // set copy palette
  Copy::palette(palette);

  // copy pixels to primary
  Copy::copy(pixels,source.left(),source.top(),source.width(),source.height(),
             width*format.bits()/8,lock(),destination.left(),
             destination.top(),destination.width(),destination.height(),
             pitch());

  // unlock
  unlock();
}




void Surface::save(void *pixels,int width,int height,
  const Format &format,const int32 palette[]) const
{
    // save surface pixels
    throw Error("surface save is not implemented");
}


void Surface::save(void *pixels,int width,int height,const Area &source,
  const Area &destination,const Format &format,const int32 palette[]) const
{
    // save surface pixels
    throw Error("surface save is not implemented");
}


void Surface::palette(int32 palette[])
{
    // set palette
    memcpy(m_palette,palette,256*4);
}



int Surface::width()
{
    // get width
    return m_width;
}


int Surface::height()
{
    // get height
    return m_height;
}


int Surface::pitch()
{
    // get pitch
    return m_width * m_format.bits()/8;
}


const Format& Surface::format()
{
    // get format
    return m_format;
}

