//
// VGA Interface class for OpenPTC 1.0 C++ API Implementation [DOS extension]
// Copyright (c) 1998 Jonathan Matthew (jmatthew@uq.net.au)
// The OpenPTC 1.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

#include <stdio.h>
#include <conio.h>

#include "dos\dpmi.h"
#include "fakemode\vga.h"
#include "dos\near.h"

#ifdef __WATCOMC__

void vgamode();
#pragma aux vgamode = "mov ax, 13h" "int 10h" modify [ax];

void biostextmode();
#pragma aux biostextmode = "mov ax, 3" "int 10h" modify [ax];

#else

void biostextmode()
{
    DPMI dpmi;
    DPMI::dpmi_regs dr;
    memset(&dr, 0, sizeof(DPMI::dpmi_regs));
    dr.l.eax = 3;
    dpmi.dos_int(0x10, &dr, &dr);
}

void vgamode()
{
    // there has to be a better way to do this.
    DPMI dpmi;
    DPMI::dpmi_regs dr;
    memset(&dr, 0, sizeof(DPMI::dpmi_regs));
    dr.l.eax = 0x13;
    dpmi.dos_int(0x10, &dr, &dr);
}

#endif

// MUST use assembler fakemode conversion with DJGPP.
#ifdef __DJGPP__
#define X86_ASSEMBLER
#endif

#ifdef X86_ASSEMBLER

// Assembler fakemode conversion
extern "C" {
#ifdef __WATCOMC__

int32 *PlaneBlt1_RGB(char *src, int32 *dest, int rows);
int32 *PlaneBlt1_RBG(char *src, int32 *dest, int rows);
int32 *PlaneBlt1_GRB(char *src, int32 *dest, int rows);
int32 *PlaneBlt2_RBG(char *src, int32 *dest, int rows);
int32 *PlaneBlt2_GBR(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_RGBRGB(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_GRBGRB(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_RBGRBG(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_GRBRBG(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_RBGGRB(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_RGBR(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_GRBG(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_RBGR(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_GRBR(char *src, int32 *dest, int rows);
int32 *PlaneBlt3_RBGG(char *src, int32 *dest, int rows);

#pragma aux PlaneBlt1_RGB "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt1_RBG "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt1_GRB "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt2_RBG "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt2_GBR "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_RGBRGB "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_GRBGRB "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_RBGRBG "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_GRBRBG "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_RBGGRB "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_RGBR "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_GRBG "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_RBGR "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_GRBR "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];
#pragma aux PlaneBlt3_RBGG "_*" parm [EAX] [EDX] [ECX] value [EAX] modify [ECX EDX];

#else

int32 *PlaneBlt1_RGB(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt1_RBG(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt1_GRB(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt2_RBG(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt2_GBR(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_RGBRGB(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_GRBGRB(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_RBGRBG(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_GRBRBG(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_RBGGRB(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_RGBR(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_GRBG(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_RBGR(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_GRBR(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));
int32 *PlaneBlt3_RBGG(char *src, int32 *dest, int rows) __attribute__ ((regparm(3)));

#endif
}

#else

// C++ fakemode conversion
static int32 *PlaneBlt1_RGB(char *src, int32 *dest, int rows);
static int32 *PlaneBlt1_RBG(char *src, int32 *dest, int rows);
static int32 *PlaneBlt1_GRB(char *src, int32 *dest, int rows);
static int32 *PlaneBlt2_RBG(char *src, int32 *dest, int rows);
static int32 *PlaneBlt2_GBR(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_RGBRGB(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_GRBGRB(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_RBGRBG(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_GRBRBG(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_RBGGRB(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_RGBR(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_GRBG(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_RBGR(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_GRBR(char *src, int32 *dest, int rows);
static int32 *PlaneBlt3_RBGG(char *src, int32 *dest, int rows);


#endif

VGA::VGA(int xres, int yres, int modetype, int faketype)
{

    // check dimensions
    if ((xres > 320) || (yres > 200)) {
        throw Error("no vga mode available");
    }

    // set up variables
    m_pages = 1;
    m_width = xres;
    m_height = yres;
    m_mode_type = modetype;

    // set up display offset to centre image on display
    m_dispoffset = ((100-(yres/2))*320) + (160-(xres/2));

    // check fakemode type
    if ((faketype != FAKEMODE1A) && (faketype != FAKEMODE1B) &&
        (faketype != FAKEMODE1C) && (faketype != FAKEMODE2A) &&
        (faketype != FAKEMODE2B) && (faketype != FAKEMODE2C) &&
        (faketype != FAKEMODE3A) && (faketype != FAKEMODE3B) &&
        (faketype != FAKEMODE3C)) faketype = FAKEMODE2A;
    m_fake_type = faketype;

    // set mode
    vgamode();
    m_ptr = (char *)0xA0000;

    // set up format
    if (modetype == RGB332) {
        m_pitch = 320;
        m_format = Format(8, 0xE0, 0x1C, 0x3);
    } else if (modetype == INDEX8) {
        m_pitch = 320;
        m_format = Format(8);
    } else if (modetype == FAKEMODE) {
        m_pitch = 640;
        m_width = 320;
        m_height = 200;         // no other size possible in fakemodes
        m_dispoffset = 0;
        m_format = Format(16, 0xF800, 0x7E0, 0x1F);

        // tweak to required resolution
        wait_retrace();
        if ((m_fake_type == FAKEMODE1A) || (m_fake_type == FAKEMODE1B) ||
            (m_fake_type == FAKEMODE1C)) {
            // FAKEMODE1x - 320x600
            outp(0x3D4,0x11);
            outp(0x3D5,inp(0x3D5) & 0x7F);
            outp(0x3C2,0xE7);
            outp(0x3D4,0x00); outp(0x3D5,0x5F);
            outp(0x3D4,0x01); outp(0x3D5,0x4F);
            outp(0x3D4,0x02); outp(0x3D5,0x50);
            outp(0x3D4,0x03); outp(0x3D5,0x82);
            outp(0x3D4,0x04); outp(0x3D5,0x54);
            outp(0x3D4,0x05); outp(0x3D5,0x80);
            outp(0x3D4,0x06); outp(0x3D5,0x70);
            outp(0x3D4,0x07); outp(0x3D5,0xF0);
            outp(0x3D4,0x08); outp(0x3D5,0x00);
            outp(0x3D4,0x09); outp(0x3D5,0x60);
            outp(0x3D4,0x10); outp(0x3D5,0x5B);
            outp(0x3D4,0x11); outp(0x3D5,0x8C);
            outp(0x3D4,0x12); outp(0x3D5,0x57);
            outp(0x3D4,0x13); outp(0x3D5,0x28);
            outp(0x3D4,0x14); outp(0x3D5,0x00);
            outp(0x3D4,0x15); outp(0x3D5,0x58);
            outp(0x3D4,0x16); outp(0x3D5,0x70);
            outp(0x3D4,0x17); outp(0x3D5,0xE3);
            outp(0x3C4,0x01); outp(0x3C5,0x01);
            outp(0x3C4,0x04); outp(0x3C5,0x06);
            outp(0x3CE,0x05); outp(0x3CF,0x40);
            outp(0x3CE,0x06); outp(0x3CF,0x05);
            outp(0x3CE,0x06); outp(0x3CF,0x05);
        } else {
            // FAKEMODE{2,3}x - 320x400
            outp(0x3D4,0x11); outp(0x3D5,inp(0x3D5) & 0x7F);
            outp(0x3C2,0x63);
            outp(0x3D4,0x0);  outp(0x3D5,0x5F);
            outp(0x3D4,0x1);  outp(0x3D5,0x4F);
            outp(0x3D4,0x2);  outp(0x3D5,0x50);
            outp(0x3D4,0x3);  outp(0x3D5,0x82);
            outp(0x3D4,0x4);  outp(0x3D5,0x54);
            outp(0x3D4,0x5);  outp(0x3D5,0x80);
            outp(0x3D4,0x6);  outp(0x3D5,0xBF);
            outp(0x3D4,0x7);  outp(0x3D5,0x1F);
            outp(0x3D4,0x8);  outp(0x3D5,0x00);
            outp(0x3D4,0x9);  outp(0x3D5,0x40);
            outp(0x3D4,0x10); outp(0x3D5,0x9C);
            outp(0x3D4,0x11); outp(0x3D5,0x8E);
            outp(0x3D4,0x12); outp(0x3D5,0x8F);
            outp(0x3D4,0x13); outp(0x3D5,0x28);
            outp(0x3D4,0x14); outp(0x3D5,0x00);
            outp(0x3D4,0x15); outp(0x3D5,0x96);
            outp(0x3D4,0x16); outp(0x3D5,0xB9);
            outp(0x3D4,0x17); outp(0x3D5,0xE3);
            outp(0x3C4,0x01); outp(0x3C5,0x01);
            outp(0x3C4,0x04); outp(0x3C5,0x06);
            outp(0x3CE,0x05); outp(0x3CF,0x40);
            outp(0x3CE,0x06); outp(0x3CF,0x05);
            outp(0x3CE,0x06); outp(0x3CF,0x05);
        }
        clear_memory();
    } else {
        // something weird is going on
        throw Error("zoon is an incompetent dickhead.");
    }


    // set palette for RGB332 and FAKEMODE modes
    if (modetype == RGB332) {
        int pal[256];
        for (int i=0; i<256; i++) {
            int r = (int)(((i&0xE0)>>5) * (255.0 / 7.0));
            int g = (int)(((i&0x1C)>>2) * (255.0 / 7.0));
            int b = (int)(((i&0x03)   ) * (255.0 / 3.0));
            pal[i] = (r<<16)|(g<<8)|(b);
        }
        palette((const int32 *)pal);
    } else if (modetype == FAKEMODE) {

        int pal[256];

        if ((m_fake_type == FAKEMODE2A) || (m_fake_type == FAKEMODE2B) ||
            (m_fake_type == FAKEMODE2C)) {
            // FAKEMODE2 palette
            for (int i=0; i<256; i++) {
                // taken from PTC 0.73
                if (!(i&0x80)) {
                    // bit 7 = 0 (top section)
                    // red (4 bits)
                    int r = (int)(((i&0x78)>>3)*(255.0/15.0));
                    // blue (3 bits)
                    int b = (int)((i&0x07) * (255.0 / 7.0));
                    pal[i] = (r<<16) | (b);
                } else {
                    // bit 7 = 1 (bottom section)
                    // green
                    int g = (int)((i&0x1F) * (255.0 / 31.0));
                    pal[i] = g<<8;
                }
            }
        } else {
            // FAKEMODE{1,3} palette
            for (int i=0; i<64; i++) {
                int z = (int)(i * (255.0/63.0));
                pal[i] = z<<16;
                pal[i+64] = z<<8;
                pal[i+128] = z;
                pal[i+192] = (z<<16)|(z<<8)|(z);
            }
        }
        palette((const int32 *)pal);
    }
}

VGA::~VGA() {
    if (m_init) {
        biostextmode();         // DJGPP: replace
        m_init = 0;
    }
}

void VGA::palette(const int32 *data)
{
    outp(0x3c8, 0);
    for (int i=0; i<256; i++) {
        int c = (data[i] >> 2) & 0x003F3F3F;
        outp(0x3c9, c>>16);
        outp(0x3c9, c>>8);
        outp(0x3c9, c);
    }
}

void VGA::wait_retrace()
{
    while ( inp(0x3DA) & 8) {}
    while (!(inp(0x3DA) & 8)) {}
}

void VGA::clear_memory()
{
    lock_near_base();
    wait_retrace();

    char *dest = (char *)adjust_near_pointer((void *)0xA0000);
    for (int strip=0; strip<32; strip++) {
        outpw(0x3C4, 0x102);
        memset(dest, 0, 2048);
        outpw(0x3C4, 0x202);
        memset(dest, 0, 2048);
        outpw(0x3C4, 0x402);
        memset(dest, 0, 2048);
        outpw(0x3C4, 0x802);
        memset(dest, 0, 2048);
        dest+=2048;
    }
    unlock_near_base();
}

void *VGA::lock()
{
    lock_near_base();
    return adjust_near_pointer((void *)((char *)m_ptr+m_dispoffset));
}

void VGA::unlock()
{
    unlock_near_base();
}

Format &VGA::format()
{
    return m_format;
}

void VGA::fakemode_load(char *src, int wvr)
{
    int32 *dest = (int32 *)lock();
    int w, s;
    int32 *d;
    switch (m_fake_type) {
    case FAKEMODE1A:
        for (w=0; w<25; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            PlaneBlt1_RGB(src+0, dest, 8);

            // plane 1
            outpw(0x3C4, 0x202);
            PlaneBlt1_RGB(src+2, dest, 8);

            // plane 2
            outpw(0x3C4, 0x402);
            PlaneBlt1_RGB(src+4, dest, 8);

            // plane 3
            outpw(0x3C4, 0x802);
            dest = PlaneBlt1_RGB(src+6, dest, 8);
            src += 320*4*4;
        }
        break;
    case FAKEMODE1B:
        for (w=0; w<25; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            d = PlaneBlt1_RBG(src+(4*4*2*20*0), dest, 1);
            d = PlaneBlt1_GRB(src+(4*4*2*20*1), d, 1);
            d = PlaneBlt1_RBG(src+(4*4*2*20*2), d, 1);
            d = PlaneBlt1_GRB(src+(4*4*2*20*3), d, 1);
            d = PlaneBlt1_RBG(src+(4*4*2*20*4), d, 1);
            d = PlaneBlt1_GRB(src+(4*4*2*20*5), d, 1);
            d = PlaneBlt1_RBG(src+(4*4*2*20*6), d, 1);
            d = PlaneBlt1_GRB(src+(4*4*2*20*7), d, 1);

            // plane 1
            outpw(0x3C4, 0x202);
            d = PlaneBlt1_GRB(src+2+(4*4*2*20*0), dest, 1);
            d = PlaneBlt1_RBG(src+2+(4*4*2*20*1), d, 1);
            d = PlaneBlt1_GRB(src+2+(4*4*2*20*2), d, 1);
            d = PlaneBlt1_RBG(src+2+(4*4*2*20*3), d, 1);
            d = PlaneBlt1_GRB(src+2+(4*4*2*20*4), d, 1);
            d = PlaneBlt1_RBG(src+2+(4*4*2*20*5), d, 1);
            d = PlaneBlt1_GRB(src+2+(4*4*2*20*6), d, 1);
            d = PlaneBlt1_RBG(src+2+(4*4*2*20*7), d, 1);

            // plane 2
            outpw(0x3C4, 0x402);
            d = PlaneBlt1_RBG(src+4+(4*4*2*20*0), dest, 1);
            d = PlaneBlt1_GRB(src+4+(4*4*2*20*1), d, 1);
            d = PlaneBlt1_RBG(src+4+(4*4*2*20*2), d, 1);
            d = PlaneBlt1_GRB(src+4+(4*4*2*20*3), d, 1);
            d = PlaneBlt1_RBG(src+4+(4*4*2*20*4), d, 1);
            d = PlaneBlt1_GRB(src+4+(4*4*2*20*5), d, 1);
            d = PlaneBlt1_RBG(src+4+(4*4*2*20*6), d, 1);
            d = PlaneBlt1_GRB(src+4+(4*4*2*20*7), d, 1);

            // plane 3
            outpw(0x3C4, 0x802);
            d = PlaneBlt1_GRB(src+6+(4*4*2*20*0), dest, 1);
            d = PlaneBlt1_RBG(src+6+(4*4*2*20*1), d, 1);
            d = PlaneBlt1_GRB(src+6+(4*4*2*20*2), d, 1);
            d = PlaneBlt1_RBG(src+6+(4*4*2*20*3), d, 1);
            d = PlaneBlt1_GRB(src+6+(4*4*2*20*4), d, 1);
            d = PlaneBlt1_RBG(src+6+(4*4*2*20*5), d, 1);
            d = PlaneBlt1_GRB(src+6+(4*4*2*20*6), d, 1);
            dest = PlaneBlt1_RBG(src+6+(4*4*2*20*7), d, 1);

            src += 320*4*4;
        }
        break;
    case FAKEMODE1C:
        for (w=0; w<25; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            PlaneBlt1_RBG(src+0, dest, 8);

            // plane 1
            outpw(0x3C4, 0x202);
            PlaneBlt1_GRB(src+2, dest, 8);

            // plane 2
            outpw(0x3C4, 0x402);
            PlaneBlt1_RBG(src+4, dest, 8);

            // plane 3
            outpw(0x3C4, 0x802);
            dest = PlaneBlt1_GRB(src+6, dest, 8);
            src += 320*4*4;
        }
        break;
    case FAKEMODE2A:
        for (w=0; w<25; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            PlaneBlt2_RBG(src+0, dest, 8);

            // plane 1
            outpw(0x3C4, 0x202);
            PlaneBlt2_RBG(src+2, dest, 8);

            // plane 2
            outpw(0x3C4, 0x402);
            PlaneBlt2_RBG(src+4, dest, 8);

            // plane 3
            outpw(0x3C4, 0x802);
            dest = PlaneBlt2_RBG(src+6, dest, 8);
            src += 320*4*4;
        }
        break;
    case FAKEMODE2B:
        for (w=0; w<25; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            d = PlaneBlt2_RBG(src+(4*4*2*20*0), dest, 1);
            d = PlaneBlt2_GBR(src+(4*4*2*20*1), d, 1);
            d = PlaneBlt2_RBG(src+(4*4*2*20*2), d, 1);
            d = PlaneBlt2_GBR(src+(4*4*2*20*3), d, 1);
            d = PlaneBlt2_RBG(src+(4*4*2*20*4), d, 1);
            d = PlaneBlt2_GBR(src+(4*4*2*20*5), d, 1);
            d = PlaneBlt2_RBG(src+(4*4*2*20*6), d, 1);
            d = PlaneBlt2_GBR(src+(4*4*2*20*7), d, 1);

            // plane 1
            outpw(0x3C4, 0x202);
            d = PlaneBlt2_GBR(src+2+(4*4*2*20*0), dest, 1);
            d = PlaneBlt2_RBG(src+2+(4*4*2*20*1), d, 1);
            d = PlaneBlt2_GBR(src+2+(4*4*2*20*2), d, 1);
            d = PlaneBlt2_RBG(src+2+(4*4*2*20*3), d, 1);
            d = PlaneBlt2_GBR(src+2+(4*4*2*20*4), d, 1);
            d = PlaneBlt2_RBG(src+2+(4*4*2*20*5), d, 1);
            d = PlaneBlt2_GBR(src+2+(4*4*2*20*6), d, 1);
            d = PlaneBlt2_RBG(src+2+(4*4*2*20*7), d, 1);

            // plane 2
            outpw(0x3C4, 0x402);
            d = PlaneBlt2_RBG(src+4+(4*4*2*20*0), dest, 1);
            d = PlaneBlt2_GBR(src+4+(4*4*2*20*1), d, 1);
            d = PlaneBlt2_RBG(src+4+(4*4*2*20*2), d, 1);
            d = PlaneBlt2_GBR(src+4+(4*4*2*20*3), d, 1);
            d = PlaneBlt2_RBG(src+4+(4*4*2*20*4), d, 1);
            d = PlaneBlt2_GBR(src+4+(4*4*2*20*5), d, 1);
            d = PlaneBlt2_RBG(src+4+(4*4*2*20*6), d, 1);
            d = PlaneBlt2_GBR(src+4+(4*4*2*20*7), d, 1);

            // plane 3
            outpw(0x3C4, 0x802);
            d = PlaneBlt2_GBR(src+6+(4*4*2*20*0), dest, 1);
            d = PlaneBlt2_RBG(src+6+(4*4*2*20*1), d, 1);
            d = PlaneBlt2_GBR(src+6+(4*4*2*20*2), d, 1);
            d = PlaneBlt2_RBG(src+6+(4*4*2*20*3), d, 1);
            d = PlaneBlt2_GBR(src+6+(4*4*2*20*4), d, 1);
            d = PlaneBlt2_RBG(src+6+(4*4*2*20*5), d, 1);
            d = PlaneBlt2_GBR(src+6+(4*4*2*20*6), d, 1);
            dest = PlaneBlt2_RBG(src+6+(4*4*2*20*7), d, 1);

            src += 320*4*4;
        }
        break;
    case FAKEMODE2C:
        for (w=0; w<25; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            PlaneBlt2_RBG(src+0, dest, 8);

            // plane 1
            outpw(0x3C4, 0x202);
            PlaneBlt2_GBR(src+2, dest, 8);

            // plane 2
            outpw(0x3C4, 0x402);
            PlaneBlt2_RBG(src+4, dest, 8);

            // plane 3
            outpw(0x3C4, 0x802);
            dest = PlaneBlt2_GBR(src+6, dest, 8);
            src += 320*4*4;
        }
        break;
    case FAKEMODE3A:
        for (w=0; w<16; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            PlaneBlt3_RGBRGB(src+0, dest, 4);

            // plane 1
            outpw(0x3C4, 0x202);
            PlaneBlt3_RGBRGB(src+2, dest, 4);

            // plane 2
            outpw(0x3C4, 0x402);
            PlaneBlt3_RGBRGB(src+4, dest, 4);

            // plane 3
            outpw(0x3C4, 0x802);
            dest = PlaneBlt3_RGBRGB(src+6, dest, 4);
            src += 320*4*2*3;
        }
        s = (4*4*2*20) + (320*2*2*2);

        outpw(0x3C4, 0x102);
        d = PlaneBlt3_RGBRGB(src, dest, 2);
        PlaneBlt3_RGBR(src+s, d, 1);

        outpw(0x3C4, 0x202);
        d = PlaneBlt3_RGBRGB(src+2, dest, 2);
        PlaneBlt3_RGBR(src+s+2, d, 1);

        outpw(0x3C4, 0x402);
        d = PlaneBlt3_RGBRGB(src+4, dest, 2);
        PlaneBlt3_RGBR(src+s+4, d, 1);

        outpw(0x3C4, 0x802);
        d = PlaneBlt3_RGBRGB(src+6, dest, 2);
        PlaneBlt3_RGBR(src+s+6, d, 1);

        break;
    case FAKEMODE3B:
        for (w=0; w<16; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            PlaneBlt3_GRBRBG(src+0, dest, 4);

            // plane 1
            outpw(0x3C4, 0x202);
            PlaneBlt3_RBGGRB(src+2, dest, 4);

            // plane 2
            outpw(0x3C4, 0x402);
            PlaneBlt3_GRBRBG(src+4, dest, 4);

            // plane 3
            outpw(0x3C4, 0x802);
            dest = PlaneBlt3_RBGGRB(src+6, dest, 4);
            src += 320*4*2*3;
        }
        s = (4*4*2*20) + (320*2*2*2);

        outpw(0x3C4, 0x102);
        d = PlaneBlt3_GRBRBG(src, dest, 2);
        PlaneBlt3_GRBR(src+s, d, 1);

        outpw(0x3C4, 0x202);
        d = PlaneBlt3_RBGGRB(src+2, dest, 2);
        PlaneBlt3_RBGG(src+s+2, d, 1);

        outpw(0x3C4, 0x402);
        d = PlaneBlt3_GRBRBG(src+4, dest, 2);
        PlaneBlt3_GRBR(src+s+4, d, 1);

        outpw(0x3C4, 0x802);
        d = PlaneBlt3_RBGGRB(src+6, dest, 2);
        PlaneBlt3_RBGG(src+s+6, d, 1);
        break;
    case FAKEMODE3C:
        for (w=0; w<16; w++) {
            // plane 0
            outpw(0x3C4, 0x102);
            PlaneBlt3_GRBGRB(src+0, dest, 4);

            // plane 1
            outpw(0x3C4, 0x202);
            PlaneBlt3_RBGRBG(src+2, dest, 4);

            // plane 2
            outpw(0x3C4, 0x402);
            PlaneBlt3_GRBGRB(src+4, dest, 4);

            // plane 3
            outpw(0x3C4, 0x802);
            dest = PlaneBlt3_RBGRBG(src+6, dest, 4);
            src += 320*4*2*3;
        }
        s = (4*4*2*20) + (320*2*2*2);

        outpw(0x3C4, 0x102);
        d = PlaneBlt3_GRBGRB(src, dest, 2);
        PlaneBlt3_GRBG(src+s, d, 1);

        outpw(0x3C4, 0x202);
        d = PlaneBlt3_RBGRBG(src+2, dest, 2);
        PlaneBlt3_RBGR(src+s+2, d, 1);

        outpw(0x3C4, 0x402);
        d = PlaneBlt3_GRBGRB(src+4, dest, 2);
        PlaneBlt3_GRBG(src+s+4, d, 1);

        outpw(0x3C4, 0x802);
        d = PlaneBlt3_RBGRBG(src+6, dest, 2);
        PlaneBlt3_RBGR(src+s+6, d, 1);
        break;
    }
    if (wvr) wait_retrace();
    unlock();
}

#ifndef X86_ASSEMBLER


static int32 *PlaneBlt1_RGB(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            unsigned int b = gl;
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 3;

            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            r &= 0xF8F8F8F8;
            r >>= 2;

            gl += gh;
            gl |= 0x40404040;

            dest[0] = r;
            dest[20] = gl;
            dest[40] = b;

            dest++;
            src += 4*4*2;
        }
        dest += 40;
    }
    return dest;
}

static int32 *PlaneBlt1_RBG(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            unsigned int b = gl;
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 3;

            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            r &= 0xF8F8F8F8;
            r >>= 2;

            gl += gh;
            gl |= 0x40404040;

            dest[0] = r;
            dest[20] = b;
            dest[40] = gl;

            dest++;
            src += 4*4*2;
        }
        dest += 40;
    }
    return dest;
}

static int32 *PlaneBlt1_GRB(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            unsigned int b = gl;
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 3;

            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            r &= 0xF8F8F8F8;
            r >>= 2;

            gl += gh;
            gl |= 0x40404040;

            dest[0] = gl;
            dest[20] = r;
            dest[40] = b;

            dest++;
            src += 4*4*2;
        }
        dest += 40;
    }
    return dest;
}

static int32 *PlaneBlt2_RBG(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {

            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            unsigned int b = gl;
            gl &= 0xC0C0C0C0;
            gl >>= 6;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 2;

            b &=  0x1C1C1C1C;
            b >>= 2;

            r &=  0xF0F0F0F0;
            r >>= 1;

            r += b;
            gl += gh;
            gl |= 0x80808080;

            dest[0] = r;
            dest[20] = gl;

            src += 4*4*2;
            dest++;
        }
        dest += 20;
    }
    return dest;
}

static int32 *PlaneBlt2_GBR(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            unsigned int b = gl;
            gl &= 0xC0C0C0C0;
            gl >>= 6;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 2;

            b &=  0x1C1C1C1C;
            b >>= 2;

            r &=  0xF0F0F0F0;
            r >>= 1;

            r += b;
            gl += gh;
            gl |= 0x80808080;

            dest[0] = gl;
            dest[20] = r;

            src += 4*4*2;
            dest++;
        }
        dest += 20;
    }
    return dest;
}

static int32 *PlaneBlt3_RGBRGB(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 3;

            unsigned int b =  ((src[ 0+(320*2)])<< 0) | ((src[ 8+(320*2)])<< 8) |
                              ((src[16+(320*2)])<<16) | ((src[24+(320*2)])<<24);
            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            r &= 0xF8F8F8F8;
            r >>= 2;

            gl += gh;
            gl |= 0x40404040;

            dest[0] = r;
            dest[20] = gl;
            dest[40] = b;

            r =  ((src[ 1+(320*2)])<< 0) | ((src[ 9+(320*2)])<< 8) |
                 ((src[17+(320*2)])<<16) | ((src[25+(320*2)])<<24);
            r &= 0xF8F8F8F8;
            r >>= 2;

            gl =  ((src[ 0+(640*2)])<< 0) | ((src[ 8+(640*2)])<< 8) |
                  ((src[16+(640*2)])<<16) | ((src[24+(640*2)])<<24);
            b = gl;
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            gh =  ((src[ 1+(640*2)])<< 0) | ((src[ 9+(640*2)])<< 8) |
                  ((src[17+(640*2)])<<16) | ((src[25+(640*2)])<<24);
            gh &= 0x07070707;
            gh <<= 3;

            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            gl += gh;
            gl |= 0x40404040;

            dest[60] = r;
            dest[80] = gl;
            dest[100] = b;

            dest++;
            src += 4*4*2;
        }
        dest += 100;
        src += 320*2*2;
    }
    return dest;
}

static int32 *PlaneBlt3_GRBGRB(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 3;

            unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            r &= 0xF8F8F8F8;
            r >>= 2;

            gl += gh;
            gl |= 0x40404040;

            dest[0] = gl;
            dest[20] = r;
            dest[40] = b;

            r =  ((src[ 1+320*2])<< 0) | ((src[ 9+320*2])<< 8) |
                 ((src[17+320*2])<<16) | ((src[25+320*2])<<24);
            r &= 0xF8F8F8F8;
            r >>= 2;

            gl =  ((src[ 0+640*2])<< 0) | ((src[ 8+640*2])<< 8) |
                  ((src[16+640*2])<<16) | ((src[24+640*2])<<24);
            b = gl;
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            gh =  ((src[ 1+640*2])<< 0) | ((src[ 9+640*2])<< 8) |
                  ((src[17+640*2])<<16) | ((src[25+640*2])<<24);
            gh &= 0x07070707;
            gh <<= 3;

            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            gl += gh;
            gl |= 0x40404040;

            dest[60] = gl;
            dest[80] = r;
            dest[100] = b;

            dest++;
            src += 4*4*2;
        }
        dest += 100;
        src += 320*2*2;
    }
    return dest;
}

static int32 *PlaneBlt3_RBGRBG(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 3;

            unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            r &= 0xF8F8F8F8;
            r >>= 2;

            gl += gh;
            gl |= 0x40404040;

            dest[0] = r;
            dest[20] = b;
            dest[40] = gl;

            r =  ((src[ 1+320*2])<< 0) | ((src[ 9+320*2])<< 8) |
                 ((src[17+320*2])<<16) | ((src[25+320*2])<<24);
            r &= 0xF8F8F8F8;
            r >>= 2;

            gl =  ((src[ 0+640*2])<< 0) | ((src[ 8+640*2])<< 8) |
                  ((src[16+640*2])<<16) | ((src[24+640*2])<<24);
            b = gl;
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            gh =  ((src[ 1+640*2])<< 0) | ((src[ 9+640*2])<< 8) |
                  ((src[17+640*2])<<16) | ((src[25+640*2])<<24);
            gh &= 0x07070707;
            gh <<= 3;

            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            gl += gh;
            gl |= 0x40404040;

            dest[60] = r;
            dest[80] = b;
            dest[100] = gl;

            dest++;
            src += 4*4*2;
        }
        dest += 100;
        src += 320*2*2;
    }
    return dest;
}

static int32 *PlaneBlt3_GRBRBG(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 3;

            unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            r &= 0xF8F8F8F8;
            r >>= 2;

            gl += gh;
            gl |= 0x40404040;

            dest[0] = gl;
            dest[20] = r;
            dest[40] = b;

            r =  ((src[ 1+320*2])<< 0) | ((src[ 9+320*2])<< 8) |
                 ((src[17+320*2])<<16) | ((src[25+320*2])<<24);
            r &= 0xF8F8F8F8;
            r >>= 2;

            gl =  ((src[ 0+640*2])<< 0) | ((src[ 8+640*2])<< 8) |
                  ((src[16+640*2])<<16) | ((src[24+640*2])<<24);
            b = gl;
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            gh =  ((src[ 1+640*2])<< 0) | ((src[ 9+640*2])<< 8) |
                  ((src[17+640*2])<<16) | ((src[25+640*2])<<24);
            gh &= 0x07070707;
            gh <<= 3;

            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            gl += gh;
            gl |= 0x40404040;

            dest[60] = r;
            dest[80] = b;
            dest[100] = gl;

            dest++;
            src += 4*4*2;
        }
        dest += 100;
        src += 320*2*2;
    }
    return dest;
}

static int32 *PlaneBlt3_RBGGRB(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 3;

            unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            r &= 0xF8F8F8F8;
            r >>= 2;

            gl += gh;
            gl |= 0x40404040;

            dest[0] = r;
            dest[20] = b;
            dest[40] = gl;

            r =  ((src[ 1+320*2])<< 0) | ((src[ 9+320*2])<< 8) |
                 ((src[17+320*2])<<16) | ((src[25+320*2])<<24);
            r &= 0xF8F8F8F8;
            r >>= 2;

            gl =  ((src[ 0+640*2])<< 0) | ((src[ 8+640*2])<< 8) |
                  ((src[16+640*2])<<16) | ((src[24+640*2])<<24);
            b = gl;
            gl &= 0xE0E0E0E0;
            gl >>= 5;

            gh =  ((src[ 1+640*2])<< 0) | ((src[ 9+640*2])<< 8) |
                  ((src[17+640*2])<<16) | ((src[25+640*2])<<24);
            gh &= 0x07070707;
            gh <<= 3;

            b &= 0x1F1F1F1F;
            b <<= 1;
            b |= 0x80808080;

            gl += gh;
            gl |= 0x40404040;

            dest[60] = gl;
            dest[80] = r;
            dest[100] = b;

            dest++;
            src += 4*4*2;
        }
        dest += 100;
        src += 320*2*2;
    }
    return dest;
}

static int32 *PlaneBlt3_RGBR(char *src, int32 *dest, int rows)
{
    for (int col=0; col<20; col++) {
        unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                          ((src[16])<<16) | ((src[24])<<24);
        gl &= 0xE0E0E0E0;
        gl >>= 5;

        unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                          ((src[17])<<16) | ((src[25])<<24);
        unsigned int r = gh;
        gh &= 0x07070707;
        gh <<= 3;

        unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
        b &= 0x1F1F1F1F;
        b <<= 1;
        b |= 0x80808080;

        r &= 0xF8F8F8F8;
        r >>= 2;

        gl += gh;
        gl |= 0x40404040;

        dest[0] = r;
        dest[20] = gl;
        dest[40] = b;

        r =  ((src[ 1+320*2])<< 0) | ((src[ 9+320*2])<< 8) |
             ((src[17+320*2])<<16) | ((src[25+320*2])<<24);
        r &= 0xF8F8F8F8;
        r >>= 2;

        dest[60] = r;

        dest++;
        src += 4*4*2;
    }
    return dest;
}

static int32 *PlaneBlt3_GRBG(char *src, int32 *dest, int rows)
{
    for (int col=0; col<20; col++) {
        unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                          ((src[16])<<16) | ((src[24])<<24);
        gl &= 0xE0E0E0E0;
        gl >>= 5;

        unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                          ((src[17])<<16) | ((src[25])<<24);
        unsigned int r = gh;
        gh &= 0x07070707;
        gh <<= 3;

        unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
        b &= 0x1F1F1F1F;
        b <<= 1;
        b |= 0x80808080;

        r &= 0xF8F8F8F8;
        r >>= 2;

        gl += gh;
        gl |= 0x40404040;

        dest[0] = gl;
        dest[20] = r;
        dest[40] = b;

        gl =  ((src[ 0+640*2])<< 0) | ((src[ 8+640*2])<< 8) |
              ((src[16+640*2])<<16) | ((src[24+640*2])<<24);
        gl &= 0xE0E0E0E0;
        gl >>= 5;

        gh =  ((src[ 1+640*2])<< 0) | ((src[ 9+640*2])<< 8) |
              ((src[17+640*2])<<16) | ((src[25+640*2])<<24);
        gh &= 0x07070707;
        gh <<= 3;

        gl += gh;
        gl |= 0x40404040;

        dest[60] = gl;

        dest++;
        src += 4*4*2;
    }
    return dest;
}

static int32 *PlaneBlt3_RBGR(char *src, int32 *dest, int rows)
{
    for (int col=0; col<20; col++) {
        unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                          ((src[16])<<16) | ((src[24])<<24);
        gl &= 0xE0E0E0E0;
        gl >>= 5;

        unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                          ((src[17])<<16) | ((src[25])<<24);
        unsigned int r = gh;
        gh &= 0x07070707;
        gh <<= 3;

        unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
        b &= 0x1F1F1F1F;
        b <<= 1;
        b |= 0x80808080;

        r &= 0xF8F8F8F8;
        r >>= 2;

        gl += gh;
        gl |= 0x40404040;

        dest[0] = r;
        dest[20] = b;
        dest[40] = gl;

        r =  ((src[ 1+320*2])<< 0) | ((src[ 9+320*2])<< 8) |
             ((src[17+320*2])<<16) | ((src[25+320*2])<<24);
        r &= 0xF8F8F8F8;
        r >>= 2;

        dest[60] = r;

        dest++;
        src += 4*4*2;
    }
    return dest;
}

static int32 *PlaneBlt3_GRBR(char *src, int32 *dest, int rows)
{
    for (int col=0; col<20; col++) {
        unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                          ((src[16])<<16) | ((src[24])<<24);
        gl &= 0xE0E0E0E0;
        gl >>= 5;

        unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                          ((src[17])<<16) | ((src[25])<<24);
        unsigned int r = gh;
        gh &= 0x07070707;
        gh <<= 3;

        unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
        b &= 0x1F1F1F1F;
        b <<= 1;
        b |= 0x80808080;

        r &= 0xF8F8F8F8;
        r >>= 2;

        gl += gh;
        gl |= 0x40404040;

        dest[0] = gl;
        dest[20] = r;
        dest[40] = b;

        r =  ((src[ 1+320*2])<< 0) | ((src[ 9+320*2])<< 8) |
             ((src[17+320*2])<<16) | ((src[25+320*2])<<24);
        r &= 0xF8F8F8F8;
        r >>= 2;

        dest[60] = r;

        dest++;
        src += 4*4*2;
    }
    return dest;
}

static int32 *PlaneBlt3_RBGG(char *src, int32 *dest, int rows)
{
    for (int col=0; col<20; col++) {
        unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                          ((src[16])<<16) | ((src[24])<<24);
        gl &= 0xE0E0E0E0;
        gl >>= 5;

        unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                          ((src[17])<<16) | ((src[25])<<24);
        unsigned int r = gh;
        gh &= 0x07070707;
        gh <<= 3;

        unsigned int b =  ((src[ 0+320*2])<< 0) | ((src[ 8+320*2])<< 8) |
                              ((src[16+320*2])<<16) | ((src[24+320*2])<<24);
        b &= 0x1F1F1F1F;
        b <<= 1;
        b |= 0x80808080;

        r &= 0xF8F8F8F8;
        r >>= 2;

        gl += gh;
        gl |= 0x40404040;

        dest[0] = r;
        dest[20] = b;
        dest[40] = gl;

        gl =  ((src[ 0+640*2])<< 0) | ((src[ 8+640*2])<< 8) |
              ((src[16+640*2])<<16) | ((src[24+640*2])<<24);
        gl &= 0xE0E0E0E0;
        gl >>= 5;

        gh =  ((src[ 1+640*2])<< 0) | ((src[ 9+640*2])<< 8) |
              ((src[17+640*2])<<16) | ((src[25+640*2])<<24);
        gh &= 0x07070707;
        gh <<= 3;

        gl += gh;
        gl |= 0x40404040;

        dest[60] = gl;

        dest++;
        src += 4*4*2;
    }
    return dest;
}


#endif
