//
// Fakemode Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// include files
#include <stdio.h>
#include <string.h>
#include <fstream.h>
#include <stdlib.h>
#include "Core/Key.h"
#include "Core/Area.h"
#include "Core/Color.h"
#include "Core/Error.h"
#include "Core/Config.h"
#include "Core/Clipper.h"
#include "Core/Console.h"
#include "Core/Surface.h"
#include "Fakemode/console.h"
#include "vga.h"

// define defaults
#define DEFAULT_WIDTH 320;
#define DEFAULT_HEIGHT 200;
#define DEFAULT_FORMAT Format(32,0x00FF0000,0x0000FF00,0x000000FF);



DLLAPI PTCAPI FakemodeConsole::FakemodeConsole()
{
    // defaults
    m_open   = false;
    m_locked = false;

    // clear strings
    m_title[0] = 0;
    m_information[0] = 0;

    // default option data
    m_default_width = DEFAULT_WIDTH;
    m_default_height = DEFAULT_HEIGHT;
    m_default_format = DEFAULT_FORMAT;

    m_primary=0;
    m_vga=0;
    m_fakemode_type = FAKEMODE2A;
    m_wait_retrace = true;


    // configure console
    configure("ptc.cfg");

}


DLLAPI PTCAPI FakemodeConsole::~FakemodeConsole()
{
    // close
    close();
}




DLLAPI void PTCAPI FakemodeConsole::configure(const char file[])
{
    // open configuration file
    ifstream is(file,ios::in | ios::nocreate);

    // check stream
    if (!is) return;

    // read option strings
    while (!is.eof())
    {
        // option line
        char line[1024];

        // read line from stream
        is.getline(line,1024);

        // process option
        option(line);
    }
}




DLLAPI bool PTCAPI FakemodeConsole::option(const char option[])
{
    // check for "default width" option
    if (strnicmp(option,"default width",13)==0)
    {
        // look for parameter
        if (option[13]!=0)
        {
            // we have a parameter
            m_default_width = atoi(&option[13]);
        }
        else
        {
            // no parameter
            m_default_width = DEFAULT_WIDTH;
        }
    }

    // check for "default height" option
    if (strnicmp(option,"default height",14)==0)
    {
        // look for parameter
        if (option[14]!=0)
        {
            // we have a parameter
            m_default_height = atoi(&option[14]);
        }
        else
        {
            // no parameter
            m_default_height = DEFAULT_HEIGHT;
        }

        // recognized
        return true;
    }

    // check for "default bits" option
    if (strnicmp(option,"default bits",12)==0)
    {
        // look for parameter
        if (option[12]!=0)
        {
            // we have a parameter
            int bits = atoi(&option[12]);

            // setup format
            switch (bits)
            {
                case 8:  m_default_format = Format(8); break;
                case 16: m_default_format = Format(16,0xF800,0x07E0,0x001F); break;
                case 24: m_default_format = Format(24,0x00FF0000,0x0000FF00,0x000000FF); break;
                case 32: m_default_format = Format(32,0x00FF0000,0x0000FF00,0x000000FF); break;
                default: return false;
            }
        }
        else
        {
            // no parameter
            m_default_height = DEFAULT_HEIGHT;
        }

        // recognized
        return true;
    }

    // pass the option to the copy object
    return m_copy.option(option);
}




DLLAPI const Mode* PTCAPI FakemodeConsole::modes()
{
    // bogus imitation mode list! :)
    m_modes[0] = Mode(640,480,Format(8));
    m_modes[1] = Mode(800,600,Format(8));
    m_modes[2] = Mode(1024,768,Format(8));
    m_modes[3] = Mode(640,480,Format(16,0xF800,0x07E0,0x001F));
    m_modes[4] = Mode(800,600,Format(16,0xF800,0x07E0,0x001F));
    m_modes[5] = Mode(1024,768,Format(16,0xF800,0x07E0,0x001F));
    m_modes[6] = Mode(640,480,Format(32,0x00FF0000,0x0000FF00,0x000000FF));
    m_modes[7] = Mode(800,600,Format(32,0x00FF0000,0x0000FF00,0x000000FF));
    m_modes[8] = Mode(1024,768,Format(32,0x00FF0000,0x0000FF00,0x000000FF));

    // return modes
    return m_modes;
}




DLLAPI void PTCAPI FakemodeConsole::open(const char title[],int pages)
{
    // open console
    open(title,m_default_format,pages);
}


DLLAPI void PTCAPI FakemodeConsole::open(const char title[],const Format &format,int pages)
{
    // open console
    open(title,m_default_width,m_default_height,m_default_format,pages);
}


DLLAPI void PTCAPI FakemodeConsole::open(const char title[],int width,int height,const Format &format,int pages)
{
    // open exact mode (windowed output is our backup...)
    open(title,Mode(width,height,format),pages);
}


DLLAPI void PTCAPI FakemodeConsole::open(const char title[],const Mode &mode,int pages)
{
    // check that the mode is valid
    if (!mode.valid()) throw Error("invalid mode");

    // get mode information
    int width = mode.width();
    int height = mode.height();
    Format format = mode.format();

    // setup before open
    internal_pre_open_setup(title);

    // check output mode

                // start internal fullscreen open
                internal_open_fullscreen_start();

                // internal open fullscreen mode
                internal_open_fullscreen(width,height,format);

                // finish internal fullscreen open
                internal_open_fullscreen_finish(pages);

    // setup after open
    internal_post_open_setup();
}


DLLAPI void PTCAPI FakemodeConsole::close()
{
    // check if open
    if (m_open)
    {
        // the console must be unlocked when closed
        if (m_locked) throw Error("console is still locked");

        // flush all key presses
        while (key()) read();
    }

    // close console
    internal_close();
}





DLLAPI void PTCAPI FakemodeConsole::flush()
{
    // debug checks
    check_open();
    check_unlocked();

    // [platform dependent code to flush all console operations]

}


DLLAPI void PTCAPI FakemodeConsole::finish()
{
    // debug check
    check_open();
    check_unlocked();

    // [platform dependent code to finish all console operations]

}


DLLAPI void PTCAPI FakemodeConsole::update()
{
    // debug check
    check_open();
    check_unlocked();

    // update primary surface


    char *m_offscreen=(char *)m_primary->lock();

        if (m_vga->fakemode()) {
            m_vga->fakemode_load(m_offscreen, m_wait_retrace);
        } else {
            if (m_wait_retrace) m_vga->wait_retrace();
            memcpy(m_vga->lock(), m_offscreen, pitch()*height());
            m_vga->unlock();
        }

    m_primary->unlock();
    // update window
}


DLLAPI void PTCAPI FakemodeConsole::update(const Area &area)
{
    // update
    update();
}




DLLAPI bool PTCAPI FakemodeConsole::key()
{
    // debug check
    check_open();

    // check for key press
    return m_keyboard->key();
}


DLLAPI Key PTCAPI FakemodeConsole::read()
{
    // debug check
    check_open();

    // read key press from keyboard
    return m_keyboard->read();
}




DLLAPI void PTCAPI FakemodeConsole::copy(BaseSurface &surface)
{
    // debug check
    check_open();
    check_unlocked();

    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console to surface",error);
    }
}


DLLAPI void PTCAPI FakemodeConsole::copy(BaseSurface &surface,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();

    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette(),source,destination);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console area to surface area",error);
    }
}




DLLAPI void* PTCAPI FakemodeConsole::lock()
{
    // debug check
    check_open();

    // fail if the console is already locked
    if (m_locked) throw Error("console is already locked");

    // lock primary surface
    void *pixels = m_primary->lock();

    //} else if (m_vga) {
    //    pixels = (void *)m_offscreen;

    // surface is locked
    m_locked = true;

    // return pixels
    return pixels;
}


DLLAPI void PTCAPI FakemodeConsole::unlock()
{
    // debug check
    check_open();

    // fail if the console is not locked
    if (!m_locked) throw Error("console is not locked");

    // unlock primary surface
    m_primary->unlock();

    // we are unlocked
    m_locked = false;
}




DLLAPI void PTCAPI FakemodeConsole::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
    // check clip area
    if (clip()==area())
    {
        // debug check
        check_open();
        check_unlocked();

        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(format,this->format());

            // set copy palettes
            m_copy.palette(palette,this->palette());

            // copy pixels to surface
            m_copy.copy(pixels,0,0,width,height,pitch,console_pixels,0,0,this->width(),this->height(),this->pitch());

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();

            // error message
            throw Error("failed to load pixels to console",error);
        }
    }
    else
    {
        // load explicit areas
        load(pixels,width,height,pitch,format,palette,Area(0,0,width,height),area());
    }
}


DLLAPI void PTCAPI FakemodeConsole::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();

    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,Area(0,0,width,height),clipped_source,destination,clip(),clipped_destination);

        // request format conversion
        m_copy.request(format,this->format());

        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch,
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to load pixels to console area",error);
    }
}




DLLAPI void PTCAPI FakemodeConsole::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
    // debug check
    check_open();
    check_unlocked();

    // check clip area
    if (clip()==area())
    {
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(this->format(),format);

            // set copy palettes
            m_copy.palette(this->palette(),palette);

            // copy console pixels to 'pixels' buffer
            m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,pitch);

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();

            // error message
            throw Error("failed to save console pixels",error);
        }
    }
    else
    {
        // save explicit areas
        save(pixels,width,height,pitch,format,palette,area(),Area(0,0,width,height));
    }
}


DLLAPI void PTCAPI FakemodeConsole::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();

    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,clip(),clipped_source,destination,Area(0,0,width,height),clipped_destination);

        // request format conversion
        m_copy.request(this->format(),format);

        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),this->pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch);
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to save console area pixels",error);
    }
}




DLLAPI void PTCAPI FakemodeConsole::clear()
{
    // debug check
    check_open();
    check_unlocked();

    // check console format
    if (format().direct())
    {
        // direct color
        clear(Color(0,0,0,0));
    }
    else
    {
        // indexed color
        clear(Color(0));
    }
}


DLLAPI void PTCAPI FakemodeConsole::clear(const Color &color)
{
    // debug check
    check_open();
    check_unlocked();

    // clear console
    clear(color,area());
}


DLLAPI void PTCAPI FakemodeConsole::clear(const Color &color,const Area &area)
{
    // debug check
    check_open();
    check_unlocked();

    // clear primary surface
    m_primary->clear(color,area);
}




DLLAPI void PTCAPI FakemodeConsole::palette(const Palette &palette)
{
    // debug check
    check_open();

    // set primary surface palette
    m_primary->palette(palette);

    {
        m_vga->palette(palette.data());
    }
}


DLLAPI const Palette& PTCAPI FakemodeConsole::palette() const
{
    // debug check
    check_open();

    // get primary surface palette
    return m_primary->palette();
}




DLLAPI void PTCAPI FakemodeConsole::clip(const Area &area)
{
    // debug check
    check_open();

    // set clip area
    m_primary->clip(area);
}




DLLAPI int PTCAPI FakemodeConsole::width() const
{
    // debug check
    check_open();

    // get primary width
    return m_primary->width();
}


DLLAPI int PTCAPI FakemodeConsole::height() const
{
    // debug check
    check_open();

    // get primary height
    return m_primary->height();
}


DLLAPI int PTCAPI FakemodeConsole::pages() const
{
    // debug check
    check_open();

    // get primary pages
    return 2;//m_primary->pages();
}


DLLAPI int PTCAPI FakemodeConsole::pitch() const
{
    // debug check
    check_open();

    // get primary pitch
    return m_primary->pitch();
}


DLLAPI const Area& PTCAPI FakemodeConsole::area() const
{
    // debug check
    check_open();

    // get primary area
    return m_primary->area();
}


DLLAPI const Area& PTCAPI FakemodeConsole::clip() const
{
    // debug check
    check_open();

    // get primary clip
    return m_primary->clip();
}


DLLAPI const Format& PTCAPI FakemodeConsole::format() const
{
    // debug check
    check_open();

    // get primary format
    return m_primary->format();
}


DLLAPI const char* PTCAPI FakemodeConsole::name() const
{
    // get name
    return "Fakemode";
}


DLLAPI const char* PTCAPI FakemodeConsole::title() const
{
    // debug check
    check_open();

    // get title
    return m_title;
}


DLLAPI const char* PTCAPI FakemodeConsole::information() const
{
    // debug check
    check_open();

    // get information
    return m_information;
}






void FakemodeConsole::internal_pre_open_setup(const char title[])
{
    // close down
    internal_close();

    // check length of console title
    if (strlen(title)+1<=sizeof(m_title))
    {
        // update console title
        strcpy(m_title,title);
    }
    else
    {
        // update console title truncated
        memcpy(m_title,title,sizeof(m_title));
        m_title[sizeof(m_title)-1] = 0;
    }
}


void FakemodeConsole::internal_open_fullscreen_start()
{
    // create fullscreen window

    // set cooperative level
}


void FakemodeConsole::internal_open_fullscreen(int width,int height,const Format &format)
{
    // open display
//    m_display.open(width,height,format);


        int vgamodetype = 0;    // assume fakemode
        if (format.bits() == 8) {
            vgamodetype = 2;
            // rgb332?
            if (format.direct()) vgamodetype = 1;
        }

        m_vga = new VGA(width, height, vgamodetype, m_fakemode_type);

        m_primary=new Surface(m_vga->width(),m_vga->height(),m_vga->format());
        // setup offscreen buffer
        //m_offscreen = new char[m_height*m_pitch];
        //if (!m_offscreen) throw Error("out of memory");
        //memset(m_offscreen, 0, m_height*m_pitch);
        //m_bufmode = m_vga->pages();
        m_vga->palette(m_primary->palette().data());
}


void FakemodeConsole::internal_open_fullscreen_finish(int pages)
{
    try
    {
        // n buffered video memory primary surface
//        m_primary = new Surface(_library.lpDD2(),pages,true,true);
    }
    catch (Error&)
    {
        // failure
        m_primary = 0;
    }

}




void FakemodeConsole::internal_post_open_setup()
{
    // create win32 keyboard
    m_keyboard = new DosKeyboard();//m_window->handle(),m_window->thread(),false);

    // temporary platform dependent information fudge
    sprintf(m_information,"dos version x.xx.x\nvesa version x.xx\nvesa driver name xxxxx\ndisplay driver vendor xxxxx\ncertified driver? x\n");

    // set open flag
    m_open = true;
}


void FakemodeConsole::internal_reset()
{
    // delete primary
    delete m_primary;

    // delete keyboard
    delete m_keyboard;

    // clear pointers
    m_primary = 0;
    m_keyboard = 0;
}


void FakemodeConsole::internal_close()
{
    // delete primary
    delete m_primary;
    m_primary=0;

    // close display
//    m_display.close();
        if (m_vga) {
            delete m_vga;
            m_vga = 0;
        }

    // delete keyboard
    delete m_keyboard;

    // clear pointers
    m_primary = 0;
    m_keyboard = 0;
}




void FakemodeConsole::check_open() const
{
    #ifdef __DEBUG__

        // check that console is open
        if (!m_open) throw Error("console is not open");

    #else

        // no checking in release build

    #endif
}


void FakemodeConsole::check_unlocked() const
{
    #ifdef __DEBUG__

        // check that console is unlocked
        if (m_locked) throw Error("console is not unlocked");

    #else

        // no checking in release build

    #endif
}
