//
// Timer class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include <time.h>
#include "Core\Timer.h"
#include "Core\Error.h"




Timer::Timer()
{
    // defaults
    m_old = 0.0;
    m_time = 0.0;
    m_start = 0.0;
    m_current = 0.0;
    m_running = false;
}


Timer::Timer(double time)
{
    // defaults
    m_old = 0.0;
    m_time = 0.0;
    m_start = 0.0;
    m_current = 0.0;
    m_running = false;

    // set time
    set(time);
}


Timer::Timer(const Timer &timer)
{
    // copy constructor
    *this = timer;
}


Timer::~Timer()
{
    // stop
    stop();
}




void Timer::set(double time)
{
    // set time
    m_current = time;
    m_start = clock();
    m_time = m_start + time;
    m_old = m_time - delta();
}




void Timer::start()
{
    // check if not running
    if (!m_running)
    {
        // set start time
        m_start = clock();

        // set old time
        m_old = clock();

        // timer is running
        m_running = true;
    }
}


void Timer::stop()
{
    // timer is not running
    m_running = false;
}




double Timer::time()
{
    // check if running
    if (m_running)
    {
        // get current clock time
        const double time = clock();

        // assign if time has advanced
        if (time>m_time) m_time = time;

        // calculate current time
        m_current = m_time - m_start;
    }

    // get current time
    return m_current;
}


double Timer::delta()
{
    // check if running
    if (m_running)
    {
        // get current clock time
        const double time = clock();

        // calculate delta
        double delta = time - m_old;

        // update old time
        m_old = time;

        // force positive delta
        if (delta<0.0) delta = 0.0;

        // get delta
        return delta;
    }
    else
    {
        // not running
        return 0.0;
    }
}


double Timer::resolution()
{
    //
    // Get timer resolution
    // --------------------
    //
    // This function gets the resolution of the timer in seconds.
    //
    // Currently it just returns 1/CLOCKS_PER_SEC because this timer
    // implementation depends on the <time.h> "clock" function.
    //
    // If you change the "Timer::clock" implementation make sure you
    // update the resolution function to match.
    //
    // See the "Timer::clock" function for more information
    //

    // get timer resolution
    return 1.0 / (double) CLOCKS_PER_SEC;
}




Timer& Timer::operator =(const Timer &timer)
{
    // check for self assignment
    if (this==&timer) throw Error("self assignment is not allowed");

    // assignment
    m_old = timer.m_old;
    m_time = timer.m_time;
    m_start = timer.m_start;
    m_current = timer.m_current;
    m_running = timer.m_running;
    return *this;
}


bool Timer::operator ==(const Timer &timer) const
{
    // is equal to
    if (m_old==timer.m_old &&
        m_time==timer.m_time &&
        m_start==timer.m_start &&
        m_current==timer.m_current &&
        m_running==timer.m_running) return true;

    // failure
    return false;
}


bool Timer::operator !=(const Timer &timer) const
{
    // not equal to
    return !(*this==timer);
}




double Timer::clock()
{
    //
    // Get clock time
    // --------------
    //
    // This function uses the ansi c "clock" function to determine
    // the number of seconds elapsed since the process has started.
    //
    // The "clock" function is usually pretty limited in resolution
    // (ie. CLOCKS_PER_SEC) so you should probably replace the call
    // to clock with a platform dependent function with higher time
    // resolution.
    //

    // get clock time
    clock_t time = ::clock();

    // convert to seconds and return
    return (double) time / (double) CLOCKS_PER_SEC;
}
