//
// RNI implementation of "ptc.rni.Console" for PTC 2.0 Java API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "ptc.h"
#include "rni.h"




// global console
static Console *console = 0;

// translation functions
Area translate(struct Hptc_Area *area);
Format translate(struct Hptc_Format *format);
Hptc_Format* translate(Format format);

// exception handling functions
void jthrow(Error error = Error("unknown rni exception"));

// dll import function
void import(HMODULE module,void **function,char name[]);

// hacked rni functions (removes dependency on msjava.dll)
static HObject* (__cdecl *rni_execute_java_constructor)(ExecEnv *ee, const char *classname,ClassClass *cb, const char *signature,...) = 0;
static void (__cdecl *rni_exceptionSet)( ExecEnv *ee, HObject *phThrowable ) = 0;
static Hjava_lang_String* (__cdecl *rni_makeJavaStringFromUtf8)(const char *pszUtf8) = 0;
static char* (__cdecl *rni_javaString2CString)(Hjava_lang_String *s, char *buf, int buflen) = 0;

// msjava.dll library
static HMODULE library = 0;




__declspec(dllexport) void __cdecl ptc_rni_Console_create(struct Hptc_rni_Console *object,struct Hjava_lang_String *title,long width,long height,struct Hptc_Format *format)
{
    try
    {
		// load msjava.dll
		library = LoadLibrary("msjava.dll");
        if (!library) throw Error("could not load msjava.dll");

        // import rni functions
		import(library,(void**)&rni_execute_java_constructor,"execute_java_constructor");
		import(library,(void**)&rni_exceptionSet,"exceptionSet");
		import(library,(void**)&rni_makeJavaStringFromUtf8,"makeJavaStringFromUtf8");
		import(library,(void**)&rni_javaString2CString,"javaString2CString");

        // create console
        console = new Console();

        // open console
        ptc_rni_Console_open(object,title,width,height,format);
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) void __cdecl ptc_rni_Console_destroy(struct Hptc_rni_Console *object)
{
    try
    {
		// free library
		FreeLibrary(library);

        // destroy console
        delete console;
        console = 0;
    }
    catch (Error)
    {
        // no exceptions
    }
}


__declspec(dllexport) void __cdecl ptc_rni_Console_open(struct Hptc_rni_Console *object,struct Hjava_lang_String *title,long width,long height,struct Hptc_Format *format)
{
    try
    {
        // get string
        char string[1024];
		rni_javaString2CString(title,string,1024);

        // open console
        console->open(string,width,height,translate(format));
    }
    catch (Error error)
    {
        // close console
        console->close();

        // known error
        jthrow(error);
    }
    catch (...)
    {
        // close console
        console->close();

        // unknown
        jthrow();
    }
}


__declspec(dllexport) void __cdecl ptc_rni_Console_close(struct Hptc_rni_Console *object)
{
    try
    {
        // close console
        console->close();
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) void __cdecl ptc_rni_Console_update(struct Hptc_rni_Console *object)
{
    try
    {
        // update console
        console->update();
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) long __cdecl ptc_rni_Console_key(struct Hptc_rni_Console *object)
{
    try
    {
        // check for keypress
        return console->key();
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) long __cdecl ptc_rni_Console_read(struct Hptc_rni_Console *object)
{
    try
    {
        // read key
        return console->read();
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) void __cdecl ptc_rni_Console_copy(struct Hptc_rni_Console *object, struct Hptc_base_Surface *surface)
{
    // copy to surface
    jthrow(Error("console to surface copy is not implemented"));
}


__declspec(dllexport) void __cdecl ptc_rni_Console_copy_area (struct Hptc_rni_Console *object,struct Hptc_base_Surface *surface,struct Hptc_Area *source,struct Hptc_Area *destination)
{
	// copy to surface area
	jthrow(Error("console to surface copy is not implemented"));
}


__declspec(dllexport) HObject* __cdecl ptc_rni_Console_lock(struct Hptc_rni_Console *object)
{
    // lock console
    jthrow(Error("console lock is not supported"));

    // pixels
    return 0;
}


__declspec(dllexport) void __cdecl ptc_rni_Console_unlock(struct Hptc_rni_Console *object)
{
    // unlock console
    jthrow(Error("console unlock is not supported"));
}


__declspec(dllexport) void __cdecl ptc_rni_Console_load(struct Hptc_rni_Console *object,HObject *pixels,long width,long height,struct Hptc_Format *format,struct HArrayOfInt *palette)
{
    try
    {
        // lock pixel array
        void *pixel_array = ((ClassArrayOfByte*)pixels)->body;

        // lock palette
        int32 *palette_array = (int32*) ((ClassArrayOfInt*)palette)->body;

        // write to console
        console->load(pixel_array,width,height,translate(format),palette_array);
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) void __cdecl ptc_rni_Console_load_area(struct Hptc_rni_Console *object,HObject *pixels,long width,long height,struct Hptc_Area *source,struct Hptc_Area *destination,struct Hptc_Format *format,struct HArrayOfInt *palette)
{
    try
    {
        // lock pixel array
        void *pixel_array = ((ClassArrayOfByte*)pixels)->body;

        // lock palette
        int32 *palette_array = (int32*) ((ClassArrayOfInt*)palette)->body;

        // write to console
        console->load(pixel_array,width,height,translate(source),translate(destination),translate(format),palette_array);
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) void __cdecl ptc_rni_Console_save(struct Hptc_rni_Console *object,HObject *pixels,long width,long height,struct Hptc_Format *format,struct HArrayOfInt *palette)
{
	// save surface pixels
	jthrow(Error("save surface pixels is not implemented"));
}


__declspec(dllexport) void __cdecl ptc_rni_Console_save_area(struct Hptc_rni_Console *object,HObject *pixels,long width,long height,struct Hptc_Area *source,struct Hptc_Area *destination,struct Hptc_Format *format,struct HArrayOfInt *palette)
{
	// load surface pixels
	jthrow(Error("save surface area pixels is not implemented"));
}


__declspec(dllexport) void __cdecl ptc_rni_Console_palette(struct Hptc_rni_Console *object, struct HArrayOfInt *palette)
{
    try
    {
        // lock palette
        int32 *palette_array = (int32*) ((ClassArrayOfInt*)palette)->body;

        // write to console
        console->palette(palette_array);
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) long __cdecl ptc_rni_Console_width(struct Hptc_rni_Console *object)
{
    try
    {
        // get width
        return console->width();
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) long __cdecl ptc_rni_Console_height(struct Hptc_rni_Console *object)
{
    try
    {
        // get height
        return console->height();
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) struct Hptc_Format * __cdecl ptc_rni_Console_format(struct Hptc_rni_Console *object)
{
    try
    {
        // get format
        return translate(console->format());
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}


__declspec(dllexport) struct Hjava_lang_String * __cdecl ptc_rni_Console_name(struct Hptc_rni_Console *object)
{
    try
    {
        // return console name
        return rni_makeJavaStringFromUtf8(console->name());
    }
    catch (Error error)
    {
        // known error
        jthrow(error);
    }
    catch (...)
    {
        // unknown
        jthrow();
    }
}




Area translate(struct Hptc_Area *area)
{
	// translate area class
	return Area(area->left,area->top,area->right,area->bottom);
}


Format translate(struct Hptc_Format *format)
{
    // check format type
    if (format->direct)
    {
        // direct color
        return Format(format->bits,format->r,format->g,format->b,format->a);
    }
    else
    {
        // indexed color
        return Format(format->bits);
    }
}


Hptc_Format* translate(Format format)
{
    // check format type
    if (format.direct())
    {
        // direct color
        return (Hptc_Format*) rni_execute_java_constructor(0,"ptc/Format",0,"(IIIII)",format.bits(),format.r(),format.g(),format.b(),format.a());
    }
    else
    {
        // indexed color
        return (Hptc_Format*) rni_execute_java_constructor(0,"ptc/Format",0,"(I)",format.bits());
    }
}


void jthrow(Error error)
{
    // create error string
    Hjava_lang_String* string = rni_makeJavaStringFromUtf8(error.message());

    // construct ptc.Error object
    HObject *object = rni_execute_java_constructor(0,"ptc/Error",0,"(Ljava/lang/String;)",string);

    // set java exception
    rni_exceptionSet(0,object);
}




void import(HMODULE module,void **function,char name[])
{
	// import function
	*function = (void*) GetProcAddress(module,name);

	// check import 
	if (!*function) throw Error("could not import msjava function");
}




DWORD __cdecl RNIGetCompatibleVersion()
{
    // version id
    return RNIVER;
}
