/*
    intel2gas.cc   intel to at&t assembly converter
    Copyright (c) 1999 Mikko Tiihonen (mikko.tiihonen@hut.fi)
    This source code is licensed under the GNU LGPL
  
    Please refer to the file COPYING.LIB contained in the distribution for
    licensing conditions
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <glob.h>         //  Find files
#include "i2gconf.h"

#define TAB_SIZE	  8
#define STR_LEN 	  63
#define SYNTAX_LINE_LEN   63
#define TEMP_SIZE         256
#define WORDS_PER_FILE    64

#define TAG_TEXT    "text"
#define TAG_NEWLINE "nl"
#define TAG_COMMENT "cm"
#define TAG_MORE    "more"
// #define TAG_MATHNUM "mathnum"

struct variable_t {
    char	name[STR_LEN+1];
    char	value[STR_LEN+1];
    variable_t *next;
};

struct list_t {
    char	name[STR_LEN+1];
    char        bit;

    int         length;
    char	words[WORDS_PER_FILE][STR_LEN+1];

    variable_t *assignments;

    list_t     *next;
};

struct syntax_t {
    char        parseline[SYNTAX_LINE_LEN+1];
    char        output[SYNTAX_LINE_LEN+1];
    
    variable_t *assignments;

    syntax_t   *next;
};

struct syntaxlist_t {
    char          name[STR_LEN+1];
    syntax_t     *syntax;
    variable_t   *var;

    syntaxlist_t *next;
};

list_t	       *list;

syntaxlist_t   *basesyntaxlist = NULL;
syntaxlist_t   *syntaxlist = NULL;

char           *resource_path;
char            tempstr[TEMP_SIZE];

int current_line = 0;

list_t *newList()
{
    list_t *l = new list_t;
    l->length = 0;
    l->bit = 0;
    l->assignments = NULL;

    l->next = list;
    list = l;

    return l;
}

syntax_t *newSyntax()
{
    syntax_t *s = new syntax_t;
    s->parseline[0] = '\0';
    s->output[0] = '\0';
    s->assignments = NULL;

    s->next = NULL;
    if (syntaxlist->syntax == NULL) {
	syntaxlist->syntax = s;
	return s;
    }

    // put last in list
    syntax_t *ss = syntaxlist->syntax;
    while (ss->next)
	ss = ss->next;
    ss->next = s;
    
    return s;
}

syntaxlist_t *newSyntaxList(char const *name)
{
    syntaxlist_t *s = new syntaxlist_t;
    strcpy(s->name,name);
    s->syntax = NULL;
    s->var = NULL;

    s->next = syntaxlist;
    syntaxlist = s;

    return s;
}

syntaxlist_t *findSyntax(char const *name)
{
    syntaxlist_t *l = basesyntaxlist;
    while (l) {
	if (!strcasecmp(name,l->name))
	    return l;
	l = l->next;
    }
    return NULL;
}

void setVar(char const * const varname, char const * const value)
{
    variable_t *v = syntaxlist->var;
    while (v) {
	if (!strcasecmp(v->name, varname)) {
	    // value already exists
    	    strcpy(v->value, value);
	    return;
	}
	v = v->next;
    }
    v = new variable_t;
    strcpy(v->name, varname);
    strcpy(v->value, value);
    v->next = NULL;

    if (syntaxlist->var == NULL) {
	syntaxlist->var = v;
	return;
    }

    // put last in list
    variable_t *vv = syntaxlist->var;
    while (vv->next)
	vv = vv->next;

    vv->next = v;
}

static void deleteVars()
{
    variable_t *v = syntaxlist->var;
    syntaxlist->var = NULL;
    while (v) {
	variable_t *delme = v;
	v = v->next;
	delete delme;
    }
}

static char *str_cutat(char * const str, char cut)
{
    char *pos = strchr(str,cut);
    if (!pos) return NULL;
    *pos = '\0';
    return pos+1;
}

static char *str_splitat(char const * const str, char *left, char cut)
{
    char *pos = strchr(str,cut);
    if (!pos) {
	strcpy(left,str);
	return NULL;
    }
    strncpy(left,str,pos-str);
    left[pos-str] = '\0';
    return pos+1;
}

static void str_splitstrat(char const * const str, char *left, char cut, char *right)
{
    char *pos = str_splitat(str,left,cut);
    if (!pos) {
	*right = '\0';
	return;
    }
    strcpy(right,pos);
}

static char *str_skipspace(char *str)
{
    while (*str == ' ') str++;
    return str;
}

inline int str_empty(char const * const str)
{
    return *str == '\0';
}

void HandleVarSet(char const * const str)
{
    if (str[0] != '@') return;
    char name[STR_LEN+1], value[STR_LEN+1];
    str_splitstrat(str+1,name,'=',value);
    str_cutat(name,' ');
    setVar(name, str_skipspace(value));
}

void ReadNextLine(FILE *f)
{
    if (feof(f)) {
        *tempstr = '\0';
	return;
    }
    char temp[TEMP_SIZE];
    fgets(temp,TEMP_SIZE,f);
//    str_cutat(temp,'\n');
//    str_cutat(temp,'\r');
    char *s1 = temp,
	 *s2 = tempstr;
    while (*s1 != '\0') {
	if (*s1 == '\t') {
	    s1++;
	    do {
	        *s2++ = ' ';
	    } while ((s2-tempstr)%TAB_SIZE != 0);
	    continue;
	}
	if (*s1 == '\n' || *s1 == '\r')
	    break;
	*s2++ = *s1++;
    }
    *s2 = '\0';
}

void LoadWordList(char const filename[])
{
    FILE *f;
    f = fopen(filename,"r");
    
    list_t *l = newList();
    
    // get the xnn letters of filename
    // xxxx.nn.list
    //    987654321
    int strl = strlen(filename);
    l->bit = 0x10*(filename[strl-7]-'0') +
             0x01*(filename[strl-6]-'0');
    strcpy(l->name,strrchr(filename,'/')+1);
    str_cutat(l->name,'.');

    newSyntaxList("dummy"); // all variables go here
    
    int mode = 0;
    while (!feof(f)) {
	ReadNextLine(f);
	char *str = tempstr;
	str_cutat(str,'#');
	str = str_skipspace(str);
	// do not allow empty lines (should we?)
	if (str_empty(str)) continue;
	if (mode == 0) {
	    if (!strcmp(str,"-")) {
		// steal the variables from dummy syntax
		l->assignments = syntaxlist->var;
		syntaxlist->var = NULL;
		mode++;
		continue;
	    }
	    if (str[0] == '@') {
	        HandleVarSet(str);	        
		continue;
	    }
	    printf("Unknown meaning (%s)\n",str);
	    exit(1);
	}
	if (l->length == WORDS_PER_FILE) {
	    printf("Too many words in file\n");
	    exit(1);
	}
	strcpy(l->words[l->length],str);
	l->length++;
    }
    fclose(f);
    
    // free dummy syntax
    deleteVars();
    delete syntaxlist;
    syntaxlist = NULL;
}

void LoadWordLists()
{
    glob_t globbuf;
  
    strcpy(tempstr,resource_path);
    strcat(tempstr,"*.??.list");
    if (glob(tempstr,GLOB_NOSORT,NULL,&globbuf) < 0) {
	perror("glob");
	exit(1);
    }

    for (int i=0;i<globbuf.gl_pathc; i++) {
	LoadWordList(globbuf.gl_pathv[i]);
    }
    globfree(&globbuf);
}

static void str_catchar(char *str, char ch)
{
    const int size = strlen(str);
    str[size] = ch;
    str[size+1] = '\0';
} 

void LoadSyntax(char const *filename)
{
    FILE *f;
    f = fopen(filename,"r");
    
    {
	char name[STR_LEN+1];
	strcpy(name,strrchr(filename,'/')+1);
	str_cutat(name,'.');
	newSyntaxList(name);
    }
    
    int mode = 0;
    syntax_t *s = NULL;

    while (!feof(f)) {
	ReadNextLine(f);
	char *str = tempstr;
	str_cutat(str,'#');
	str = str_skipspace(str);
	if (str_empty(str)) continue;
	if (!strcmp(tempstr,"-")) {
	    if (s == NULL) continue;
	    if (mode != 2) {
	        printf("Need result line\n");
		exit(1);
	    }
	    // steal variables
	    s->assignments = syntaxlist->var;
	    syntaxlist->var = NULL;

	    s = NULL;
	    mode = 0;
	    continue;
	}
	if (mode == 0) {
	    s = newSyntax();
	    strcpy(s->parseline,str);
	    str_catchar(s->parseline,' '); // to allow spaces after parsed line
	    mode = 1;
	    continue;
	}
	if (str[0] == '@') {
	    HandleVarSet(str);
	    continue;
	}
	if (mode == 2) {
	    printf("In %s: Only one result line allowed\n%s\n",filename,str);
	    exit(1);
	}
	mode = 2;
	strcpy(s->output,str);
    }
    fclose(f);
    deleteVars();
}

void LoadSyntaxis()
{
    glob_t globbuf;
  
    strcpy(tempstr,resource_path);
    strcat(tempstr,"*.syntax");
    if (glob(tempstr,GLOB_NOSORT,NULL,&globbuf) < 0) {
	perror("glob");
	exit(1);
    }

    for (int i=0;i<globbuf.gl_pathc; i++) {
	LoadSyntax(globbuf.gl_pathv[i]);
    }
    globfree(&globbuf);
}


inline int isnumber(char ch) {
  return ch >= '0' && ch <= '9';
}

char const *fetchVar(char const *tag)
{
    variable_t *v = syntaxlist->var;
    while (v) {
	if (!strcasecmp(tag,v->name))
	    return v->value;
	v = v->next;
    }
    return "";
//    printf("Variable not found\n");
//    exit(1);
}

void expand_string(char const *str, char *result)
{
    while (*str != '\0') {
	if (*str == '<') {
	    char tag[STR_LEN+1];
	    str = str_splitat(str+1,tag,'>');
	    if (!strcasecmp(tag,TAG_NEWLINE)) {
		*result++ = '\n';
		continue;
	    }
	    if (!strcasecmp(tag,TAG_COMMENT)) {
	        *result++ = '#';
		continue;
	    }
	    char const *var = fetchVar(tag);
	    strcpy(result,var);
	    result += strlen(var);
	    continue;
	}
	*result++ = *str++;
    }
    *result = '\0';
}

void setTheseVars(variable_t const *vars)
{
    while (vars) {
	char temp[TEMP_SIZE];
	expand_string(vars->value,temp);
	setVar(vars->name,temp);
	vars = vars->next;
    }
}

static char str_striplastnumber(char *str)
{
    int size = strlen(str);
    if (size <= 1) return '\0';
    char number = str[size-1];
    if (!isnumber(number)) return '\0';
    str[size-1] = '\0';
    return number;
}


static void find_tag_list(char const *source, char const tag[], int bits, char *result)
{
    list_t const *l = list;
    list_t const *best;
    int bestsize = 0;
    int bestword;
    *result = '\0';
    
    while (l) {
	// correct list == tag & bits
	if ((l->bit & bits) && !strcasecmp(l->name, tag)) {
	    for (int j=0; j<l->length; j++) {
		// match source chars to a word in list
		int size = strlen(l->words[j]);
		if (!strncasecmp(source, l->words[j], size)) {
		    if (size>bestsize) {
			best = l;
			bestsize = size;
			bestword = j;
		    }
		}
	    }
	}
	l = l->next;
    }
    if (bestsize) {
        // set all variables associated with the list
        setTheseVars(best->assignments);
	strncpy(result, best->words[bestword], bestsize);
        result[bestsize] = '\0';
    }
}

static void str_copyuntil(char *dest, char const *source, char end1, char end2 = '\0')
{
    while (*source != end1 && *source != end2) {
	*dest = *source;
	if (*source == '\0') return;
	dest++; source++;
    }
    *dest = '\0';
}

static int numberparse(char const *tag, char const *input, char *output)
{
    int base;
    if (!strcasecmp(tag,"hex"))
	base = 16;
    else if (!strcasecmp(tag,"dec"))
	base = 10;
    else if (!strcasecmp(tag,"bin"))
	base = 2;
    else if (!strcasecmp(tag,"oct"))
	base = 8;
    else {
	*output = '\0';
	return 0;
    }
    char *ptr;
    strtol(input,&ptr,base);
    const int len = ptr-input;
    strncpy(output,input,len);
    output[len] = '\0';
    return len;
}

int str_parse(syntaxlist_t *s, char *parse, char *outline);

int parsematch(char *text, char *parse)
{
    char const *oldpos = text;
    while (*parse != '\0') {
	if (*parse == '<') {
	    int bits;
	    char tag[STR_LEN+1];
	    char str[TEMP_SIZE];
	    char number;
	    int len;
	    parse = str_splitat(parse+1,tag,'>');
	    str_splitstrat(tag,tag,':',str);
	    if (!str_empty(str)) {
		char *dummy;
		bits = strtol(str,&dummy,16);
		str[0] = '\0';
	    } else bits = 0xff;
	    number = str_striplastnumber(tag);
	    // Try to match a new syntax file
	    syntaxlist_t *l = findSyntax(tag);
	    if (l != NULL) {
		char temp[TEMP_SIZE];
		char stop = *parse;
		if (stop == ' ') {
		    stop = *str_skipspace(parse);
		    if (stop == '<') {
			char *pos = strchr(text,' ');
			if (pos == NULL) {
			    stop = ' ';
			} else {
			    stop = *str_skipspace(pos);
			}
		    }
		}
		variable_t *old = syntaxlist->var;
		syntaxlist->var = NULL;
		str_copyuntil(temp,text,*parse);
		len = str_parse(l,temp,str);
		syntaxlist->var = old;
		if (len==0) return 0;
	    } else {
		find_tag_list(text, tag, bits, str);
		len = strlen(str);
		if (len==0) {
		    len = numberparse(tag, text, str);
		    if (len == 0) {
			if (!strcasecmp(tag,TAG_TEXT)) {
			    str_copyuntil(str,text,*parse,' ');
			    len = strlen(str);
/*
  Now parsed with mathnum.syntax file
			} else if (!strcasecmp(tag,TAG_MATHNUM)) {
	                    char *out = str;
        	            char *in = text;
			    len = 0;
                	    do {
              			int length = str_parse(findSyntax("num"),in,out);
	                        if (length==0) break;
        	                in += length;
                	        out += strlen(out);
                        	len += length;
	                        if (*in=='+' || *in=='-' || *in=='*') {
        	                    *out++ = *in++;
                	            len++;
                        	    continue;
	                        }
        	                break;
                	    } while (1);
*/
	 		} else if (!strcasecmp(tag,TAG_MORE)) {
			    if (str_empty(text)) {
				*str = '\0';
				len = 0;
			    } else {
    				variable_t *old = syntaxlist->var;
				syntaxlist->var = NULL;
				len = str_parse(findSyntax("main"),text,str);
				deleteVars();
				syntaxlist->var = old;
				if (len==0) return 0;
			    }
			} else {
			    return 0;
			}
		    }
		}
	    }    
	    str_catchar(tag,number);
	    setVar(tag,str);
	    text += len;
	    continue;
	}
	if (*parse == ' ') {
	    parse++;
	    text = str_skipspace(text);
	    continue;
	}
	if (*parse != *text) return 0;
	if (*text == '\0') return 0;
	parse++; text++;
    }
    text = str_skipspace(text);
    if (*text != '\0') return 0;
    return text-oldpos;
}

static void str_addatpos(char *dest, char const *source, int pos)
{
    int size = strlen(dest);
    if (size > pos) {
      str_catchar(dest,' ');
      strcat(dest,source);
      return;
    }
    memset(dest+size,' ',pos-size);
    strcpy(dest+pos,source);
}

static void convert_comment(char *str)
{
    while (*str == ';')
        *str++ = '#';
}

int str_parse(syntaxlist_t *sl, char *parse, char *outline)
{
    *outline = '\0';
    parse = str_skipspace(parse);
    if (*parse == '\0')
	return 0;
    
    syntaxlist_t *old = syntaxlist;
    syntaxlist = sl;
    syntax_t *s = syntaxlist->syntax;
    
    char tempstr[TEMP_SIZE];
    while (1) {
	if (s == NULL) {
	    syntaxlist = old;
	    return 0;
	}
	strcpy(tempstr, s->parseline);
	int size = parsematch(parse, tempstr);
	if (size) {
	    setTheseVars(s->assignments);
	    expand_string(s->output, outline);
	    syntaxlist = old;
	    return size;
	}
	s = s->next;
    }
    return 0;
}

static void CommentConvert(char *str)
{
#ifdef OLD_GAS
    if (!str) return;
    while (*str) {
	switch (*str) {
	  case '\'': *str = '^';
	  // what else doesn't work
	}
	str++;
    }
#endif
}

void mainparse()
{
    char *pos;
    char *comment;
    char temp[TEMP_SIZE];
    char outline[TEMP_SIZE];

    // skip starting space
    pos = tempstr;
    comment = str_cutat(pos,';');
    CommentConvert(comment);
    *outline = '\0';
    syntaxlist_t *mainsyntax = findSyntax("main");

    pos = str_skipspace(pos);
    if (*pos != '\0') {
	if (!str_parse(mainsyntax, pos, temp)) {
	    printf("WARNING, Line %d: no syntax match: \"%s\"\n",current_line,
	           tempstr);
	    return;
	}
	
	str_addatpos(outline, temp, pos-tempstr);
    }

    if (comment) {
	convert_comment(comment);
	// restore & convert also the first ';'
	comment--; *comment = '#';
	str_addatpos(outline, comment, comment-tempstr);
    }

    printf("%s\n",outline);
}

int main(void)
{
    // first try overriding the builtin directory
    resource_path = getenv("I2G_DATA");
    if (resource_path == NULL)
      resource_path = RESOURCE_PATH;
    
    LoadWordLists();
    LoadSyntaxis();
    basesyntaxlist = syntaxlist;
    if (basesyntaxlist == NULL) {
	printf("fatal error: no %smain.syntax found\n",resource_path);
	return 1;
    }

    current_line = 1;
    ReadNextLine(stdin);

    while (!feof(stdin)) {
	mainparse();
	ReadNextLine(stdin);
	current_line++;
    }
    return 0;
}
