 //
// DirectX Primary class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// include files
#include "Core/Error.h"
#include "Core/Color.h"
#include "Core/Clipper.h"
#include "DirectX/Check.h"
#include "DirectX/Primary.h"




DirectXPrimary::DirectXPrimary(Win32Window &window,LPDIRECTDRAW2 lpDD2,int pages,bool video,bool fullscreen)
{
    // check that the number of pages requested is valid
    if (pages<=0) throw Error("invalid number of pages");

    // setup ddraw
    m_lpDD2 = lpDD2;

    // clear locked pointer
    m_locked = 0;

    // setup window
    m_window = &window;

    // defaults
    m_width   = 0;
    m_height  = 0;
    m_lpDDS   = 0;
    m_back    = 0;
    m_front   = 0;
    m_pages   = 0;

    // set active flag
    m_active  = true;

    // setup fullscreen flag
    m_fullscreen = fullscreen;

    try
    {
        // create primary surface
        DDSURFACEDESC descriptor;
        descriptor.dwSize  = sizeof(descriptor);
        descriptor.dwFlags = DDSD_CAPS;
        if (pages>1) descriptor.dwFlags |= DDSD_BACKBUFFERCOUNT;
        descriptor.dwBackBufferCount = pages - 1;
        descriptor.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
        if (video) descriptor.ddsCaps.dwCaps |= DDSCAPS_VIDEOMEMORY;
        if (pages>1) descriptor.ddsCaps.dwCaps |= DDSCAPS_COMPLEX | DDSCAPS_FLIP;
        DirectXCheck(m_lpDD2->CreateSurface(&descriptor,&m_lpDDS,0));
    
        // get surface descriptor
        descriptor.dwSize = sizeof(descriptor);
        DirectXCheck(m_lpDDS->GetSurfaceDesc(&descriptor));
        
        // get pixel format
        DDPIXELFORMAT ddpf;
        ddpf.dwSize = sizeof(ddpf);
        DirectXCheck(m_lpDDS->GetPixelFormat(&ddpf));

        // setup data
        m_front  = m_lpDDS;
        m_pages  = pages;
        m_width  = descriptor.dwWidth;
        m_height = descriptor.dwHeight;
        m_format = translate(ddpf);

        // check if fullscreen
        if (!fullscreen)
        {
            // get window client area
            RECT rectangle;
            GetClientRect(m_window->handle(),&rectangle);

            // set data to match window
            m_width  = rectangle.right;
            m_height = rectangle.bottom;
        }
        
        // setup primary surface area
        m_area = Area(0,0,m_width,m_height);

        // setup clip area
        m_clip = m_area;

        // check pages
        if (pages>1)
        {
            // get back surface
            DDSCAPS capabilities;
            capabilities.dwCaps = DDSCAPS_BACKBUFFER;
            DirectXCheck(m_front->GetAttachedSurface(&capabilities,&m_back));
        }
        else
        {
            // only one surface
            m_back = m_front;
        }
        
        // clear all pages
        while (pages--)
        {
            // clear
            clear();
            update();
        }
    }
    catch (Error &error)
    {
        // close
        close();

        // error message
        throw Error("could not create primary surface",error);
    }
}


DirectXPrimary::~DirectXPrimary()
{
    // close
    close();
}




void DirectXPrimary::update()
{
    // flip surface
    if (m_front!=m_back) DirectXCheck(m_front->Flip(0,DDFLIP_WAIT));

    // sleep
    //Sleep(0);
}




void* DirectXPrimary::lock()
{
    // block
    block();

    // setup surface descriptor
    DDSURFACEDESC descriptor;
    descriptor.dwSize = sizeof descriptor;

    // check fullscreen
    if (m_fullscreen)
    {
        // lock surface
        DirectXCheck(m_back->Lock(0,&descriptor,DDLOCK_WAIT,0));

        // update locked pointer
        m_locked = descriptor.lpSurface;
    }
    else
    {
        // get origin of client area
        POINT point = { 0,0 };
        ClientToScreen(m_window->handle(),&point);

        // setup client area rectangle
        RECT rect = { point.x, point.y, point.x+m_width, point.y+m_height };

        // lock surface
        DirectXCheck(m_back->Lock(&rect,&descriptor,DDLOCK_WAIT,0));

        // update locked pointer
        m_locked = descriptor.lpSurface;
    }

    // locked pointer
    return m_locked;
}


void DirectXPrimary::unlock()
{
    // unlock surface
    DirectXCheck(m_back->Unlock(m_locked));
}




void DirectXPrimary::clear()
{
    // block
    block();

    // check if fullscreen
    if (m_fullscreen)
    {
        // clear surface
        DDBLTFX fx;
        fx.dwSize = sizeof(fx);
        fx.dwFillColor = 0;
        DirectXCheck(m_back->Blt(0,0,0,DDBLT_COLORFILL,&fx));
    }
    else
    {
        // todo: replace with hardware clear!

        // check console format
        if (format().direct())
        {
            // direct color
            clear(Color(0,0,0,0),m_area);
        }
        else
        {
            // indexed color
            clear(Color(0),m_area);
        }
    }
}


void DirectXPrimary::clear(const Color &color,const Area &area)
{
    // block
    block();

    // check if fullscreen
    if (m_fullscreen)
    {
        // clip clear area
        Area clipped = Clipper::clip(area,m_clip);

        // clear color
        DWORD clear_color = 0;
    
        // check color
        if (color.direct())
        {
            // pack clear color integer from direct color class
            clear_color = ( int32( color.a() * 255.0f ) << 24 ) |
                          ( int32( color.r() * 255.0f ) << 16 ) |
                          ( int32( color.g() * 255.0f ) <<  8 ) |
                          ( int32( color.b() * 255.0f ) <<  0 );
        }
        else
        {
            // clear color is an index
            clear_color = color.index();
        }

        // setup blt rect
        RECT rect;
        rect.left = clipped.left();
        rect.top = clipped.top();
        rect.right = clipped.right();
        rect.bottom = clipped.bottom();

        // clear surface
        DDBLTFX fx;
        fx.dwSize = sizeof(fx);
        fx.dwFillColor = clear_color;
        DirectXCheck(m_back->Blt(&rect,0,0,DDBLT_COLORFILL,&fx));
    }
    else
    {
        // todo: replace with accelerated clearing code!

        // lock console
        void *pixels = lock();

        try
        {
            // clip area
            Area clipped_area = Clipper::clip(area,this->clip());

            // request clear
            m_clear.request(format());

            // clear console area
            m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();

            // error message
            throw Error("failed to clear console area",error);
        }
    }
}




void DirectXPrimary::palette(const Palette &palette)
{
    // block
    block();

    // check format and palette
    if (!m_format.indexed()) return;

    // get palette data
    const int32 *data = palette.data();

    // convert palette data
    PALETTEENTRY temp[256];
    for (int i=0; i<256; i++)
    {
        temp[i].peRed   = (BYTE) ( (data[i] & 0x00FF0000) >> 16 );
        temp[i].peGreen = (BYTE) ( (data[i] & 0x0000FF00) >> 8  );
        temp[i].peBlue  = (BYTE) ( (data[i] & 0x000000FF) >> 0  );
        temp[i].peFlags = 0;
    }

    // directdraw palette interface
    LPDIRECTDRAWPALETTE lpDDP = 0;

    // get attached surface palette
    if (m_lpDDS->GetPalette(&lpDDP)!=DD_OK)
    {
        // create palette object
        DirectXCheck(m_lpDD2->CreatePalette(DDPCAPS_8BIT|DDPCAPS_ALLOW256|DDPCAPS_INITIALIZE,temp,&lpDDP,0));

        // attach palette to surface
        DirectXCheck(m_lpDDS->SetPalette(lpDDP));
    }
    else
    {
        // set palette entries
        DirectXCheck(lpDDP->SetEntries(0,0,256,temp));
    }

    // update palette
    m_palette = palette;
}


const Palette& DirectXPrimary::palette() const
{
    // get palette
    return m_palette;
}




void DirectXPrimary::clip(const Area &area)
{
    // set new clip area
    m_area = Clipper::clip(area,m_area);
}




int DirectXPrimary::width() const
{
    // get width
    return m_width;
}


int DirectXPrimary::height() const
{
    // get height
    return m_height;
}


int DirectXPrimary::pages() const
{
    // get pages
    return m_pages;
}


int DirectXPrimary::pitch() const
{
    // block
    block();

    // setup surface descriptor
    DDSURFACEDESC descriptor;
    descriptor.dwSize = sizeof(descriptor);

    // get surface descriptor
    DirectXCheck(m_back->GetSurfaceDesc(&descriptor));

    // return surface pitch
    return descriptor.lPitch;
}


const Area& DirectXPrimary::area() const
{
    // get area
    return m_area;
}


const Area& DirectXPrimary::clip() const
{
    // get clip
    return m_clip;
}


const Format& DirectXPrimary::format() const
{
    // get format
    return m_format;
}




LPDIRECTDRAWSURFACE DirectXPrimary::lpDDS() const
{
    // get directdraw surface interface
    return m_lpDDS;
}                      




void DirectXPrimary::activate()
{
    // activate console
    m_active = true;
}


void DirectXPrimary::deactivate()
{
    // deactivate console
    m_active = false;
}


bool DirectXPrimary::active() const
{
    // is active?
    return m_active;
}


void DirectXPrimary::block() const
{
    // check if active
    if (!active())
    {
        // block until active
        while (!active()) m_window->update();

        // update window
        m_window->update();

        // restore
        restore();
    }
}


void DirectXPrimary::save() const
{
    // todo: save contents of primary
}


void DirectXPrimary::restore() const
{
    // restore primary surface
    DirectXCheck(m_lpDDS->Restore());

    // todo: restore contents of primary
}




Format DirectXPrimary::translate(DDPIXELFORMAT const &ddpf)
{
    // pixel format
    Format format;

    // check rgb data
    if (ddpf.dwFlags & DDPF_PALETTEINDEXED8)
    {
        // indexed color
        format = Format(8);
    }
    else if (ddpf.dwFlags & DDPF_RGB)
    {
        // direct color
        format = Format(ddpf.dwRGBBitCount,ddpf.dwRBitMask,ddpf.dwGBitMask,ddpf.dwBBitMask);
    }
    else
    {
        // error message
        throw Error("invalid pixel format");
    }

    // note: i should add alpha component support

    // return format
    return format;
}




void DirectXPrimary::close()
{
    /*
    // check if fullscreen
    if (m_fullscreen)
    {
        // check back buffer
        if (m_back)
        {
            // clear all pages
            for (int i=0; i<m_pages+1; i++)
            {
                clear();
                update();
            }
        }
    }
    */

    // check m_lpDDS
    if (m_lpDDS)
    {
        // release interface
        m_lpDDS->Release();

        // clear pointer
        m_lpDDS = 0;
    }
}
