//
// DirectX Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

#ifndef __PTC_DIRECTX_CONSOLE_H
#define __PTC_DIRECTX_CONSOLE_H

// include files
#include "Core/Base.h"
#include "Core/Copy.h"
#include "Core/Mode.h"
#include "Core/Types.h"
#include "Core/Clear.h"
#include "Core/Format.h"
#include "Core/Config.h"
#include "Core/Palette.h"
#include "Win32/Window.h"
#include "Win32/Keyboard.h"
#include "DirectX/Hook.h"
#include "DirectX/Library.h"
#include "DirectX/Display.h"
#include "DirectX/Primary.h"




class DirectXConsole : public BaseConsole
{
    // friend classes
    friend class DirectXHook;

    public:

        // setup
        DirectXConsole();
        ~DirectXConsole();

        // console configuration
        virtual void PTCAPI configure(const char file[]);

        // console option string
        virtual bool PTCAPI option(const char option[]);

        // console modes
        virtual const Mode* PTCAPI modes();

        // console management
        virtual void PTCAPI open(const char title[],int pages = 0);
        virtual void PTCAPI open(const char title[],const Format &format,int pages = 0);
        virtual void PTCAPI open(const char title[],int width,int height,const Format &format,int pages = 0);
        virtual void PTCAPI open(const char title[],const Mode &mode,int pages = 0);
        virtual void PTCAPI close();
               
        // synchronization
        virtual void PTCAPI flush();
        virtual void PTCAPI finish();
        virtual void PTCAPI update();
        virtual void PTCAPI update(const Area &area);

        // keyboard input
        virtual bool PTCAPI key();
        virtual Key PTCAPI read();

        // copy to surface
        virtual void PTCAPI copy(BaseSurface &surface);
        virtual void PTCAPI copy(BaseSurface &surface,const Area &source,const Area &destination);

        // memory access
        virtual void* PTCAPI lock();
        virtual void PTCAPI unlock();

        // load pixels to console
        virtual void PTCAPI load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette);
        virtual void PTCAPI load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination);

        // save console pixels
        virtual void PTCAPI save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette);
        virtual void PTCAPI save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination);

        // clear console
        virtual void PTCAPI clear();
        virtual void PTCAPI clear(const Color &color);
        virtual void PTCAPI clear(const Color &color,const Area &area);

        // console palette
        virtual void PTCAPI palette(const Palette &palette);
        virtual const Palette& PTCAPI palette() const;

        // console clip area
        virtual void PTCAPI clip(const Area &area);

        // data access
        virtual int PTCAPI width() const;
        virtual int PTCAPI height() const;
        virtual int PTCAPI pages() const;
        virtual int PTCAPI pitch() const;
        virtual const Area& PTCAPI area() const;
        virtual const Area& PTCAPI clip() const;
        virtual const Format& PTCAPI format() const;
        virtual const char* PTCAPI name() const;
        virtual const char* PTCAPI title() const;
        virtual const char* PTCAPI information() const;

        // extension functions
        virtual HWND PTCAPI window() const;
        virtual LPDIRECTDRAW PTCAPI lpDD() const;
        virtual LPDIRECTDRAW2 PTCAPI lpDD2() const;
        virtual LPDIRECTDRAWSURFACE PTCAPI lpDDS() const;

    private:

        // internal console management routines
        void internal_pre_open_setup(const char title[]);
        void internal_open_fullscreen_start();
        void internal_open_fullscreen(int width,int height,const Format &format);
        void internal_open_fullscreen_finish(int pages);
        void internal_open_windowed(int width,int height,int pages);
        void internal_post_open_setup();
        void internal_reset();
        void internal_close();

        // disallow copy construction
        DirectXConsole(const DirectXConsole &console);

        // disallow console assignment
        DirectXConsole& operator =(const DirectXConsole &console);

        // console debug checks
        void check_open() const;
        void check_unlocked() const;

        // data
        Mode m_modes[256];
        char m_title[1024];
        char m_information[1024];

        // flags
        bool m_open;
        bool m_locked;
        bool m_active;

        // output type
        enum Output
        {
            DEFAULT,
            WINDOWED,
            FULLSCREEN
        };

        // option data
        int m_default_width;
        int m_default_height;
        int m_default_pages;
        Format m_default_format;
        Output m_output_mode;

        // objects
        Copy m_copy;

        // Win32 objects
        Win32Window *m_window;
        Win32Keyboard *m_keyboard;

        // DirectX objects
        DirectXHook *m_hook;
        DirectXLibrary m_library;
        DirectXDisplay m_display;
        DirectXPrimary *m_primary;
};




#endif
