//
// DirectX Console implementation for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "Core/Error.h"
#include "Core/Format.h"
#include "DirectX/Console.h"

// using directive
using namespace ptc::DirectX;

// using declarations
using ptc::Core::Error;
using ptc::Core::Format;
using ptc::Win32::Window;
using ptc::Win32::Keyboard;




Console::Console(const char title[],int width,int height,const ptc::Base::Format &format)
{
    // defaults
    window = 0;
    library = 0;
    display = 0;
    primary = 0;
    keyboard = 0;

    try
    {
        // open console
        open(title,width,height,format);
    }
    catch (Error &error)
    {
        // close
        close();

        // error message
        error.rethrow("could not create DirectX console");
    }
}


Console::~Console()
{
    // close
    close();
}




void Console::open(const char title[],int width,int height,const ptc::Base::Format &format)
{
    // close
    close();

    // create library
    library = new Library();

    // create display
    display = new Display(library->lpDD());

    // test display
    display->test(title,width,height,format);

    // create window
    window = new Window(title,0,WS_VISIBLE|WS_POPUP|WS_SYSMENU,SW_NORMAL,0,0,0,0);

    // open display
    display->open(window->handle(),title,width,height,format);

    try
    {
        // triple buffered video memory primary surface
        primary = new Primary(library->lpDD(),format.indexed(),3,true);
    }
    catch (Error&)
    {
        // double buffered primary surface
        primary = new Primary(library->lpDD(),format.indexed(),2,false);
    }

    // create keyboard
    keyboard = new Keyboard(window->handle(),window->thread());
}


void Console::close()
{
    // free keyboard
    delete keyboard;
    keyboard = 0;

    // free primary
    delete primary;
    primary = 0;

    // free display
    delete display;
    display = 0;

    // free window
    delete window;
    window = 0;

    // free library
    delete library;
    library = 0;
}




void Console::update()
{
    // update console
    primary->update();
}




void Console::write(void *pixels,int width,int height,const ptc::Base::Format &format,int32 palette[])
{
    // write pixels to primary surface
    primary->write(pixels,width,height,format,palette);
}




bool Console::key()
{
    // check for keypress
    return keyboard->key();
}


int Console::read()
{
    // read keypress
    return keyboard->read();
}




void Console::copy(ptc::Base::Surface &surface)
{
    // copy to surface
    throw Error("console to surface copy is not implemented");
}




void* Console::lock()
{
    // lock primary
    return primary->lock();
}


void Console::unlock()
{
    // unlock console
    primary->unlock();
}




void Console::palette(int32 palette[])
{
    // set palette
    primary->palette(palette);
}




int Console::width()
{
    // get width
    return primary->width();
}


int Console::height()
{
    // get height
    return primary->height();
}


int Console::pitch()
{
    // get pitch
    return primary->pitch();
}


const ptc::Base::Format& Console::format()
{
    // get format
    return primary->format();
}


const char* Console::name()
{
    // get name
    return "DirectX";
}
