//
// Copy class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include <math.h>
#include <memory.h>
#include "Copy/Copy.h"
#include "Core/Error.h"
#include "Core/Format.h"

// using declarations
using ptc::Core::Error;
using ptc::Core::Format;




Copy::Copy()
{
    // setup table
    table = buffer;
}


Copy::Copy(const ptc::Base::Format &source,const ptc::Base::Format &destination)
{
    // setup table
    table = buffer;

    // set pixel formats
    formats(source,destination);
}



Copy::~Copy()
{
    // destructor
}




void Copy::formats(const ptc::Base::Format &source,const ptc::Base::Format &destination)
{
    // check if pixel formats are changing
    if (source==s_format && destination==d_format) return;

    // setup formats
    s_format = source;
    d_format = destination;

    // setup bytes per pixel
    s_bytes = source.bits() / 8;
    d_bytes = destination.bits() / 8;

    // check for copy conversion
    if (source==destination) convert = convert_copy;
    else
    {
        // check format type
        if (source.direct())
        {
            // setup direct color
            setup(source,destination);

            // select conversion functions
            convert = function(source,destination);
            generate = 0;
        }
        else
        {
            // palette pixel format
            Format palette(32,0x00FF0000,0x0000FF00,0x000000FF,0xFF000000);

            // setup indexed color
            setup(palette,destination);

            // setup conversion functions
            convert = function(source,destination);
            generate = function(palette,destination);
        }
    }
}




void Copy::palette(int32 palette[])
{
    // generate palette table
    if (s_format.indexed() && d_format.direct()) generate(*this,palette,this->buffer,256);
}




void Copy::copy(void *s_pixels,int s_x,int s_y,int s_width,int s_height,int s_pitch,
                void *d_pixels,int d_x,int d_y,int d_width,int d_height,int d_pitch)
{
    // check width and height
    if (s_width!=d_width || s_height!=d_height) throw Error("stretching is not implemented");

    // get memory pointers
    char8 *s = ((char8*)s_pixels) + s_pitch*s_y + s_x*s_bytes;
    char8 *d = ((char8*)d_pixels) + d_pitch*d_y + d_x*d_bytes;

    // setup data
    int width  = s_width;
    int height = s_height;

    // copy a line at a time
    for (int y=0; y<height; y++)
    {
        // copy line
        convert(*this,s,d,width);

        // next line
        s += s_pitch;
        d += d_pitch;
    }
}




void Copy::convert_copy(Copy &copy,void *source,void *destination,unsigned count)
{
    // copy pixels
    int bytes = count * copy.s_bytes;
    memcpy(destination,source,bytes);
}


void Copy::convert_direct32_direct32_rgba(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    int32 *s = (int32*) source;
    int32 *d = (int32*) destination;

    // pixel loop
    while (count--)
    {
        // mask color components
        int32 r = *s & copy.info.r.mask;
        int32 g = *s & copy.info.g.mask;
        int32 b = *s & copy.info.b.mask;
        int32 a = *s & copy.info.a.mask;

        // shift pixel into place
        r >>= copy.info.r.shift;
        g >>= copy.info.g.shift;
        b >>= copy.info.b.shift;
        a >>= copy.info.a.shift;

        // store pixel
        *d = r | g | b | a;
        
        // next
        s++;
        d++;
    }
}


void Copy::convert_direct32_direct24_rgba(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    int32 *s = (int32*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // mask color components
        int32 r = *s & copy.info.r.mask;
        int32 g = *s & copy.info.g.mask;
        int32 b = *s & copy.info.b.mask;
        int32 a = *s & copy.info.a.mask;

        // shift pixel into place
        r >>= copy.info.r.shift;
        g >>= copy.info.g.shift;
        b >>= copy.info.b.shift;
        a >>= copy.info.a.shift;

        // combine components
        int32 color = r | g | b | a;
        
        #if defined __LITTLE_ENDIAN__

            // store color (little endian)
            *(d+0) = *(((char*)&color) + 3);
            *(d+1) = *(((char*)&color) + 2);
            *(d+2) = *(((char*)&color) + 1);

        #elif defined __BIG_ENDIAN__

            // store color (big endian)
            *(d+0) = *(((char*)&color) + 0);
            *(d+1) = *(((char*)&color) + 1);
            *(d+2) = *(((char*)&color) + 2);

        #else

            // compiler error message
            #error you must configure the processor endianness!

        #endif

        // next
        s++;
        d+=3;
    }
}


void Copy::convert_direct32_direct16_rgba(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    int32 *s = (int32*) source;
    short16 *d = (short16*) destination;

    // pixel loop
    while (count--)
    {
        // mask color components
        int32 r = *s & copy.info.r.mask;
        int32 g = *s & copy.info.g.mask;
        int32 b = *s & copy.info.b.mask;
        int32 a = *s & copy.info.a.mask;

        // shift pixel into place
        r >>= copy.info.r.shift;
        g >>= copy.info.g.shift;
        b >>= copy.info.b.shift;
        a >>= copy.info.a.shift;

        // store pixel
        *d = (short16) (r | g | b | a);
        
        // next
        s++;
        d++;
    }
}


void Copy::convert_direct32_direct8_rgba(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    int32 *s = (int32*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // mask color components
        int32 r = *s & copy.info.r.mask;
        int32 g = *s & copy.info.g.mask;
        int32 b = *s & copy.info.b.mask;
        int32 a = *s & copy.info.a.mask;

        // shift pixel into place
        r >>= copy.info.r.shift;
        g >>= copy.info.g.shift;
        b >>= copy.info.b.shift;
        a >>= copy.info.a.shift;

        // store pixel
        *d = (char8) (r | g | b | a);
        
        // next
        s++;
        d++;
    }
}


void Copy::convert_direct32_direct32_rgb(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    int32 *s = (int32*) source;
    int32 *d = (int32*) destination;

    // pixel loop
    while (count--)
    {
        // mask color components
        int32 r = *s & copy.info.r.mask;
        int32 g = *s & copy.info.g.mask;
        int32 b = *s & copy.info.b.mask;

        // shift pixel into place
        r >>= copy.info.r.shift;
        g >>= copy.info.g.shift;
        b >>= copy.info.b.shift;

        // store pixel
        *d = r | g | b;
        
        // next
        s++;
        d++;
    }
}


void Copy::convert_direct32_direct24_rgb(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    int32 *s = (int32*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // mask color components
        int32 r = *s & copy.info.r.mask;
        int32 g = *s & copy.info.g.mask;
        int32 b = *s & copy.info.b.mask;

        // shift pixel into place
        r >>= copy.info.r.shift;
        g >>= copy.info.g.shift;
        b >>= copy.info.b.shift;

        // combine components
        int32 color = r | g | b;
        
        #if defined __LITTLE_ENDIAN__

            // store color (little endian)
            *(d+0) = *(((char*)&color) + 3);
            *(d+1) = *(((char*)&color) + 2);
            *(d+2) = *(((char*)&color) + 1);

        #elif defined __BIG_ENDIAN__

            // store color (big endian)
            *(d+0) = *(((char*)&color) + 0);
            *(d+1) = *(((char*)&color) + 1);
            *(d+2) = *(((char*)&color) + 2);

        #else

            // compiler error message
            #error you must configure the processor endianness!

        #endif

        // next
        s++;
        d+=3;
    }
}


void Copy::convert_direct32_direct16_rgb(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    int32 *s = (int32*) source;
    short16 *d = (short16*) destination;

    // pixel loop
    while (count--)
    {
        // mask color components
        int32 r = *s & copy.info.r.mask;
        int32 g = *s & copy.info.g.mask;
        int32 b = *s & copy.info.b.mask;

        // shift pixel into place
        r >>= copy.info.r.shift;
        g >>= copy.info.g.shift;
        b >>= copy.info.b.shift;

        // store pixel
        *d = (short16) (r | g | b);
        
        // next
        s++;
        d++;
    }
}


void Copy::convert_direct32_direct8_rgb(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    int32 *s = (int32*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // mask color components
        int32 r = *s & copy.info.r.mask;
        int32 g = *s & copy.info.g.mask;
        int32 b = *s & copy.info.b.mask;

        // shift pixel into place
        r >>= copy.info.r.shift;
        g >>= copy.info.g.shift;
        b >>= copy.info.b.shift;

        // store pixel
        *d = (char8) (r | g | b);
        
        // next
        s++;
        d++;
    }
}


void Copy::convert_indexed8_direct32(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    char8 *s = (char8*) source;
    int32 *d = (int32*) destination;
    
    // setup lookup pointer
    int32 *lookup = (int32*) copy.table;

    // pixel loop
    while (count--)
    {
        // convert pixel
        *d = lookup[*s];
        
        // next
        s++;
        d++;
    }
}


void Copy::convert_indexed8_direct24(Copy &copy,void *source,void *destination,unsigned count)
{
    // nasty ...
}


void Copy::convert_indexed8_direct16(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    char8 *s = (char8*) source;
    short16 *d = (short16*) destination;
    
    // setup lookup pointer
    short16 *lookup = (short16*) copy.table;

    // pixel loop
    while (count--)
    {
        // convert pixel
        *d = lookup[*s];
        
        // next
        s++;
        d++;
    }
}


void Copy::convert_indexed8_direct8(Copy &copy,void *source,void *destination,unsigned count)
{
    // setup pointers
    char8 *s = (char8*) source;
    char8 *d = (char8*) destination;
    
    // setup lookup pointer
    char8 *lookup = (char8*) copy.table;

    // pixel loop
    while (count--)
    {
        // convert pixel
        *d = lookup[*s];
        
        // next
        s++;
        d++;
    }
}




int Copy::base(int32 mask)
{
    // get lsb of mask
    int32 lsb = mask ^ ( (mask<<1) & mask );

    // calculate shift to lsb from bit 0
    return (int) ( log(lsb) / log(2) );
}


void Copy::setup(const ptc::Base::Format &source,const ptc::Base::Format &destination)
{
    // setup red information
    info.r.shift = base(source.r()) - base(destination.r());
    info.r.mask  = source.r() & (destination.r() << info.r.shift);

    // setup green information
    info.g.shift = base(source.g()) - base(destination.g());
    info.g.mask  = source.g() & (destination.g() << info.g.shift);

    // setup blue information
    info.b.shift = base(source.b()) - base(destination.b());
    info.b.mask  = source.b() & (destination.b() << info.b.shift);

    // setup alpha information
    info.a.shift = base(source.a()) - base(destination.a());
    info.a.mask  = source.a() & (destination.a() << info.a.shift);
}


Copy::Function Copy::function(const ptc::Base::Format &source,const ptc::Base::Format &destination)
{
    // check format type
    if (source.direct() && destination.direct())
    {
        // direct color -> direct color
        if (source.a() && destination.a())
        {
            // rgba direct color
            switch (source.bits())
            {
                case 32:
                {
                    // 32bit -> X
                    switch (destination.bits())
                    {
                        case 32: return convert_direct32_direct32_rgba;
                        case 24: return convert_direct32_direct24_rgba;
                        case 16: return convert_direct32_direct16_rgba;
                        case 8:  return convert_direct32_direct8_rgba;
                    }
                }
                break;
            }
        }
        else
        {
            // rgb direct color
            switch (source.bits())
            {
                case 32:
                {
                    // 32bit -> X
                    switch (destination.bits())
                    {
                        case 32: return convert_direct32_direct32_rgb;
                        case 24: return convert_direct32_direct24_rgb;
                        case 16: return convert_direct32_direct16_rgb;
                        case 8:  return convert_direct32_direct8_rgb;
                    }
                }
                break;
            }
        }
    }
    else if (source.indexed())
    {
        // indexed color -> direct color
        switch (source.bits())
        {
            case 8:
            {
                // 8bit -> X
                switch (destination.bits())
                {
                    case 32: return convert_indexed8_direct32;
                    case 24: return convert_indexed8_direct24;
                    case 16: return convert_indexed8_direct16;
                    case 8:  return convert_indexed8_direct8;
                }
            }
            break;
        }
    }
    
    // error message
    throw Error("unsupported pixel format conversion");

    // failure
    return 0;
}
