//
// Image abstraction class for PTC 2.0 C++ API
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#include <X11/Xlib.h>
#include <iostream.h>
#include "ptconfig.h"
#include "Core/Error.h"
#include "Display.h"
#include "Image.h"


// It doesn't like the next two in any class. I hope the names are long
// enough to avoid clashes. I'd wrap them in a name space but the older
// gcc gives a warning.


X11Image::X11Image(Display *display,int screen,int width,int height,
		   Format &format) : m_width(width), m_height(height),
                   m_disp(display)
{

}


X11Image::~X11Image()
{
}


/* ---------------- Normal Xlib images ------------------ */

X11NormalImage::X11NormalImage(Display *display,int screen,int width,
  int height,Format &format) : X11Image(display,screen,width,height,format)
{ cerr << "Creating normal image" << endl << flush;

  int xpad=format.bits();
  if(format.bits()==24) xpad=32;

  int xpitch=width*format.bits()/8;
  xpitch=xpitch+3;
  xpitch&=~3;

  m_pixels=new char[xpitch*height];

  m_image=XCreateImage(display,DefaultVisual(display,screen),
		       DefaultDepth(display,screen),
		       ZPixmap,0,m_pixels,
		       width,height,xpad,0);
  if(!m_image)
  throw Error("cannot create XImage");
}


X11NormalImage::~X11NormalImage()
{ XDestroyImage(m_image);
}


void X11NormalImage::put(Window w,GC gc,int x,int y)
{ XPutImage(m_disp,w,gc,m_image,0,0,x,y,m_width,m_height);
  XSync(m_disp,0); 
}


void X11NormalImage::put(Window w,GC gc,int sx,int sy,int dx,int dy,
  int width,int height)
{ XPutImage(m_disp,w,gc,m_image,sx,sy,dx,dy,width,height);
  XSync(m_disp,0); 
}


void* X11NormalImage::lock()
{ return m_pixels;
}


int X11NormalImage::pitch()
{ return m_image->bytes_per_line;
}

/* -------------- Shared memory extension images ---------------- */

#ifdef HAVE_X11_EXTENSIONS_XSHM_H

#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>

static int (*m_shm_oldhandler)(Display *,XErrorEvent *);
static bool m_shm_error=false;

static int m_shm_errorhandler(Display *disp,XErrorEvent *xev)
{ if(xev->error_code==BadAccess)        
  { m_shm_error=true;
    return 0;
  }
  else
  return((*m_shm_oldhandler)(disp,xev));
}


X11SHMImage::X11SHMImage(Display *display,int screen,int width,int height,
  Format &format) : X11Image(display,screen,width,height,format)
{ cerr << "Creating SHM Image" << endl << flush;

  m_image=XShmCreateImage(display,DefaultVisual(display,screen), 
	 		  DefaultDepth(display,screen),
			  ZPixmap,NULL,&shminfo,width,height);
  if(m_image==NULL)
  throw Error("cannot create SHM Image");

  shminfo.shmid=shmget(IPC_PRIVATE,m_image->bytes_per_line*m_image->height,
		       IPC_CREAT|0777);
  if(shminfo.shmid==-1)
  throw Error("cannot get shared memory segment");
  
  shminfo.shmaddr = (char*)shmat(shminfo.shmid,0,0);
  shminfo.readOnly = False;
  m_image->data = shminfo.shmaddr;

  if(shminfo.shmaddr==(char*)-1)
  throw Error("cannot allocate shared memory");

  // Try and attach the segment to the server. Bugfix: Have to catch
  // bad access errors in case it runs over the net.
  
  m_shm_oldhandler=XSetErrorHandler(m_shm_errorhandler);

  if(!XShmAttach(display,&shminfo))
  throw Error("cannot attach shared memory segment to display");
  
  XSync(display,0);
  
  if(m_shm_error)
  throw Error("cannot attach shared memory segment to display");

  XSetErrorHandler(m_shm_oldhandler);
}


X11SHMImage::~X11SHMImage()
{ XShmDetach(m_disp,&shminfo);
  XSync(m_disp,0);

  XDestroyImage(m_image);

  shmdt(shminfo.shmaddr);
  shmctl(shminfo.shmid,IPC_RMID,0);
}


void X11SHMImage::put(Window w,GC gc,int x,int y)
{ XShmPutImage(m_disp,w,gc,m_image,0,0,x,y,m_width,m_height,0);
  XSync(m_disp,0); 
}


void X11SHMImage::put(Window w,GC gc,int sx,int sy,int dx,int dy,int width,
  int height)
{ XShmPutImage(m_disp,w,gc,m_image,sx,sy,dx,dy,width,height,0);
  XSync(m_disp,0); 
}


void* X11SHMImage::lock()
{ return shminfo.shmaddr;
}

int X11SHMImage::pitch()
{ return m_image->bytes_per_line;
}

#endif






