//
// X11 Display abstraction classes for PTC 2.0 C++ API
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#ifndef __PTC_X11DISPLAY_H
#define __PTC_X11DISPLAY_H

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "ptconfig.h"
#include "Core/Area.h"
#include "Core/Clear.h"
#include "Core/Copy.h"
#include "Core/Format.h"
#include "Core/Color.h"
#include "Core/Palette.h"
#include "Core/Key.h"
#include "Image.h"

class Console;

#ifdef HAVE_PTHREADS
#include <pthread.h>
#endif 


enum x11_id {PTC_WINDOW,PTC_DGA};

class X11Display {

public:
    X11Display();
    virtual ~X11Display();

    // Check if DGA can be initialised
    static bool checkDGA(int width,int height,Display *disp,int screen);
  

    virtual void open(const char title[],int width,int height,
		      const Format& format,Display *disp,int screen) = 0;

    // This will always return a windowed console. The first version
    // fills the whole window, the second one has a custom size
    virtual void open(Display *disp,int screen,Window w,
		      const Format& format) = 0;
    virtual void open(Display *disp,int screen,Window window,
		      const Format& format,int x,int y,int w,int h) = 0;

    virtual void close() = 0;


    // Update the console
    virtual void update() = 0;
    virtual void update(const Area &area) = 0;

    // Keyboard I/O
    virtual bool key() = 0;
    virtual Key read() = 0;
    
    // Lock / unlock the display memory
    virtual void* lock() = 0;
    virtual void unlock() = 0;
    
  
    // load pixels to console
    virtual void load(const void *pixels,int width,int height,
		      const Format &format,const Palette &palette);
    virtual void load(const void *pixels,int width,int height,
		      const Area &source,const Area &destination,
		      const Format &format,const Palette &palette);
    
    // save console pixels
    virtual void save(void *pixels,int width,int height,
		      const Format &format,const Palette &palette);
    virtual void save(void *pixels,int width,int height,
		      const Area &source,const Area &destination,
		      const Format &format,const Palette &palette);

    // clear surface
    virtual void clear(const Color &color);
    virtual void clear(const Color &color,const Area &area);
    
    // Console palette
    virtual void palette(const Palette &palette) = 0;
    virtual const Palette& palette();
    
    // Data access
    int width() const;
    int height() const;
    virtual int pitch() const = 0;
    const Format& format() const;

    // Set flags (only used internally now!)
    void flags(long flags) { m_flags=flags; }

    // X11 helper functions for your enjoyment

    // return the display we are using
    Display* getX11Display() const { return m_disp; }

    // return the screen we are using
    int getX11Screen() const { return m_screen; }

    // return our window (0 if DGA)
    virtual Window getX11Window() const = 0;

    int id() const { return m_id; }

protected:
    // Generate internal format from requested format
    Format getFormat(const Format &format);          

    // Initialise the keyboard mapping table
    void setKeyMapping();

    // Conversion object
    Copy m_copy;
    Clear m_clear;
    Palette m_palette;
    
    long m_flags;
    unsigned int m_width;
    unsigned int m_height;
    Format m_format;
    
    Display *m_disp;                                // X11 Display
    int m_screen;                                   // X11 Screen
    
    Colormap m_cmap;                                // Color map
    XColor *m_colours;                              // Color cells
    
    x11_id m_id;

    int *m_functionkeys;
    int *m_normalkeys;
    
#ifdef HAVE_PTHREADS
    pthread_t m_thread;
#endif

    friend class Console;
};


#endif
