//
// X11 console implementation for PTC 2.0 C++ API
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//


// include files
#include <string.h>
#include <X11/Xlib.h>
#include "ptconfig.h"
#include "Core/Error.h"
#include "Core/Base.h"
#include "Core/Area.h"
#include "Core/Copy.h"
#include "Console.h"
#include "Display.h"
#include "DGADisplay.h"
#include "WindowDisplay.h"



Console::Console()
{
    // defaults
    x11disp = 0;
    m_flags = 0;

#ifdef HAVE_PTHREADS
    if (!XInitThreads()) {
      throw Error("Cannot initialise X threads");
    }
#endif

    // get user configured options
    configure("ptc.cfg");
}



Console::~Console()
{
    // Close the console
    close();
}



bool Console::option(const char str[])
{ 
  if(!strcmp(str,"dga pedantic init"))
  { m_flags|=PTC_X11_PEDANTIC_DGA;
    return true;
  }

  if(!strcmp(str,"dga off"))
  { m_flags|=PTC_X11_NODGA;
    return true;
  }

  if(!strcmp(str,"leave window open"))
  { m_flags|=PTC_X11_LEAVE_WINDOW;
    return true;
  }

  if(!strcmp(str,"leave display open"))
  { m_flags|=PTC_X11_LEAVE_DISPLAY;
    return true;
  }

  // As a last resort, pass it to the copy object to handle
  if (x11disp)
  return x11disp->m_copy.option(str);

  return false;
}


// TODO: Find current pixel depth
void Console::open(const char title[])
{ open(title,Format(32,0xff0000,0xff00,0xff));
}


// TODO: Find best DGA mode
void Console::open(const char title[],const Format& format)
{ open(title,640,480,format);
}


void Console::open(const char title[],int width,int height,
  const Format& format)
{
    close();

    // Check if we can open an X display
    Display *disp=XOpenDisplay(0);
    if(disp==NULL) throw Error("Cannot open X display");


    // DefaultScreen should be fine
    int screen=DefaultScreen(disp);


    x11disp=0;

    if (!(m_flags&PTC_X11_NODGA)) {
      try {
        x11disp=new X11DGADisplay();

        x11disp->flags(m_flags|PTC_X11_LEAVE_DISPLAY);

        x11disp->open(title,width,height,format,disp,screen);
      }
      catch (Error e) 
      { delete x11disp;
        x11disp=0;
      }

      if (x11disp) x11disp->flags(m_flags);
    }
    
    if (!x11disp)
    { x11disp=new X11WindowDisplay();

      x11disp->flags(m_flags);

      x11disp->open(title,width,height,format,disp,screen);
    }

}



void Console::open(Display *disp,int screen,Window w,const Format& format)
{
  // We already have a display and a window, so we create a windowdisplay ;)
  x11disp=new X11WindowDisplay();

  x11disp->open(disp,screen,w,format);

  x11disp->flags(m_flags);
}



void Console::open(Display *disp,int screen,Window window,const Format &format,
  int x,int y,int w,int h)
{
  // We already have a display and a window, so we create a windowdisplay ;)
  x11disp=new X11WindowDisplay();
  
  x11disp->open(disp,screen,window,format,x,y,w,h);

  x11disp->flags(m_flags);
}



void Console::close()
{
    // close display
    if(x11disp)
    { 
      delete x11disp;
      x11disp=0;
    }
}



void Console::copy(BaseSurface &surface)
{ surface.load(x11disp->lock(),x11disp->width(),x11disp->height(),
               x11disp->format(),x11disp->palette());
}


void Console::copy(BaseSurface &surface,const Area &source,
  const Area &destination)
{
  surface.load(x11disp->lock(),x11disp->width(),x11disp->height(),
               source,destination,x11disp->format(),x11disp->palette());
}



bool Console::isWindow() const
{
  if(x11disp->id()==PTC_WINDOW) return true;
  else
  return false;
}



void Console::load(const void *pixels,int width,int height,
  const Format &format,const Palette &palette)
{ x11disp->load(pixels,width,height,format,palette);
}



void Console::load(const void *pixels,int width,int height,
  const Area &source,const Area &destination,const Format &format,
  const Palette &palette)
{ x11disp->load(pixels,width,height,source,destination,format,palette);
}



void Console::save(void *pixels,int width,int height,
  const Format &format,const Palette &palette)
{ x11disp->save(pixels,width,height,format,palette);
}



void Console::save(void *pixels,int width,int height,const Area &source,
  const Area &destination,const Format &format,const Palette &palette) 
{ x11disp->save(pixels,width,height,source,destination,format,palette);
}



void Console::clear()
{ static Color c_index(0);
  static Color c_direct(0,0,0,0);

  if (format().direct()) clear(c_direct);
  else clear(c_index);
}


 
const char* Console::name() const
{ return "X11";
}



void Console::configure(const char filename[])
{

}
