//
// DirectX Display class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// include files
#include "Core/Error.h"
#include "DirectX/Check.h"
#include "DirectX/Display.h"
#include "DirectX/Translate.h"




DirectXDisplay::DirectXDisplay()
{
    // setup data
    m_open = false;
    m_fullscreen = false;
    m_ddraw = 0;
    m_window = 0;
    m_foreground = 0;
}


DirectXDisplay::~DirectXDisplay()
{
    // close
    close();
}




void DirectXDisplay::setup(LPDIRECTDRAW2 ddraw)
{
    // setup ddraw
    m_ddraw = ddraw;

    // temporary platform dependent information fudge!
    wsprintf(m_information,"windows version x.xx.x\nddraw version x.xx\ndisplay driver name xxxxx\ndisplay driver vendor xxxxx\ncertified driver? x\n");

    // zero index
    m_index = 0;

    // enumerate all display modes
    DirectXCheck(m_ddraw->EnumDisplayModes(0,0,this,(LPDDENUMMODESCALLBACK)callback));

    // terminate the mode list
    m_modes[m_index] = Mode();

    // resolution index
    int index = 0;

    // setup the resolutions list
    for (int i=0; i<m_index; i++)
    {
        // match flag
        bool match = false;

        // check against existing resolutions
        for (int j=0; j<index; j++)
        {
            // compare resolution
            if (m_modes[i].width()==m_resolutions[j].width() &&
                m_modes[i].height()==m_resolutions[j].height())
            {
                // found match
                match = true;
            }
        }

        // check match
        if (!match)
        {
            // new resolution, add it to the list
            m_resolutions[index] = m_modes[i];

            // increase index
            index ++;
        }
    }
   
    // terminate resolutions list
    m_resolutions[index] = Mode();

    // this is a truly disgusting kludge sort... blerghhhhhh!

    // sort the resolutions list
    for (i=0; i<index; i++)
    {
        // compare against all other resolutions
        for (int j=0; j<index; j++)
        {
            // check orientation
            if (i<j)
            {
                // check if resolution is greater
                if (m_resolutions[i].width()>m_resolutions[j].width() ||
                    m_resolutions[i].height()>m_resolutions[j].height())
                {
                    // swap items
                    Mode temp(m_resolutions[i]);
                    m_resolutions[i] = m_resolutions[j];
                    m_resolutions[j] = temp;
                }
            }
            else if (i>j)
            {
                // check if resolution is smaller
                if (m_resolutions[i].width()<m_resolutions[j].width() ||
                    m_resolutions[i].height()<m_resolutions[j].height())
                {
                    // swap items
                    Mode temp(m_resolutions[i]);
                    m_resolutions[i] = m_resolutions[j];
                    m_resolutions[j] = temp;
                }
            }
        }
    }
}




const Mode* DirectXDisplay::modes()
{
    // get modes
    return m_modes;
}




bool DirectXDisplay::test(const Mode &mode,bool exact)
{
    // check mode list is valid
    if (m_index==0) return true;

    // check exact
    if (exact)
    {
        // search through mode list
        for (int i=0; i<m_index; i++)
        {
            // look for match
            if (mode==m_modes[i]) return true;
        }

        // failure
        return false;
    }
    else
    {
        // resolution index
        int index = 0;

        // check resolution list
        while (m_resolutions[index].valid())
        {
            // check for fuzzy match
            if (mode.width()<=m_resolutions[index].width() &&
                mode.height()<=m_resolutions[index].height()) return true;
        }

        // failure
        return false;
    }
}


void DirectXDisplay::cooperative(HWND window,bool fullscreen)
{
    // set cooperative level
    if (fullscreen)
    {
        // enter exclusive mode
        DirectXCheck(m_ddraw->SetCooperativeLevel(window,DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX));
    }
    else
    {
        // set normal cooperative level
        DirectXCheck(m_ddraw->SetCooperativeLevel(window,DDSCL_NORMAL));
    }

    // set window
    m_window = window;

    // set fullscreen flag
    m_fullscreen = fullscreen;
}


void DirectXDisplay::open()
{
    // update flags
    m_open = true;

    // clear mode
    m_mode = Mode();
}


void DirectXDisplay::open(const Mode &mode,bool exact,int frequency)
{
    // check exact
    if (exact)
    {
        // internal open exact mode
        internal_open(mode,true,frequency);
    }
    else
    {
        // internal open nearest mode
        internal_open_nearest(mode,false,frequency);
    }
}


void DirectXDisplay::close()
{
    // check if open
    if (m_open && m_ddraw)
    {
        // check if fullscreen
        if (m_fullscreen)
        {
            // restore display mode
            m_ddraw->RestoreDisplayMode();
        
            // leave exclusive mode
            m_ddraw->SetCooperativeLevel(m_window,DDSCL_NORMAL);
        }
    }

    // clear flags
    m_open = false;
    m_fullscreen = false;
}




void DirectXDisplay::save()
{
    // save foreground window handle
    m_foreground = GetForegroundWindow();

    // save foreground window rect
    GetWindowRect(m_foreground,&m_foreground_rect);

    // save foreground window placement
    m_foreground_placement.length = sizeof WINDOWPLACEMENT;
    GetWindowPlacement(m_foreground,&m_foreground_placement);
}


void DirectXDisplay::restore()
{
    // check foreground window is valid and not the console window
    if (m_foreground && IsWindow(m_foreground) && m_foreground!=m_window)
    {
        // check console window and fullscreen
        if (IsWindow(m_window) && m_fullscreen)
        {
            // minimize console window
            SetWindowPos(m_window,HWND_BOTTOM,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE|SWP_NOACTIVATE);
        }

        // restore foreground window
        SetForegroundWindow(m_foreground);

        // restore foreground window placement
        SetWindowPlacement(m_foreground,&m_foreground_placement);

        // restore old foreground window position
        SetWindowPos(m_foreground,HWND_TOP,m_foreground_rect.left,m_foreground_rect.top,m_foreground_rect.right-m_foreground_rect.left,m_foreground_rect.bottom-m_foreground_rect.top,SWP_FRAMECHANGED|SWP_NOCOPYBITS);
    
        // clear foreground
        m_foreground = 0;
    }
}




Mode DirectXDisplay::mode() const
{
    // get mode
    return m_mode;
}


bool DirectXDisplay::fullscreen() const
{
    // get fullscreen
    return m_fullscreen;
}


const char* DirectXDisplay::information() const
{
    // get information
    return m_information;
}




void DirectXDisplay::internal_open(const Mode &mode,bool exact,int frequency)
{
    // check exact
    if (exact)
    {
        // set exact display mode
        DirectXCheck(m_ddraw->SetDisplayMode(mode.width(),mode.height(),mode.format().bits(),frequency,0));
    }
    else
    {
        /*
        // debugging
        char message[1024];
        wsprintf(message,"width %d\nheight %d\nbits %d\nexact %d\nfrequency %d\n",mode.width(),mode.height(),mode.format().bits(),(int)exact,frequency);
        throw Error(message);
        */

        // set fuzzy display mode
        switch (mode.format().bits())
        {
            case 32: 
            {
                // set virtual 32 bit mode
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),32,frequency,0)==DD_OK) break;
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),24,frequency,0)==DD_OK) break;
                DirectXCheck(m_ddraw->SetDisplayMode(mode.width(),mode.height(),16,frequency,0));
            }
            break;

            case 24:
            {
                // set virtual 24 bit mode
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),24,frequency,0)==DD_OK) break;
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),32,frequency,0)==DD_OK) break;
                DirectXCheck(m_ddraw->SetDisplayMode(mode.width(),mode.height(),16,frequency,0));
            }
            break;

            case 16: 
            {
                // set virtual 16 bit mode
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),16,frequency,0)==DD_OK) break;
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),24,frequency,0)==DD_OK) break;
                DirectXCheck(m_ddraw->SetDisplayMode(mode.width(),mode.height(),32,frequency,0));
            }
            break;

            case 8: 
            {
                // set virtual 8 bit mode
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),8,frequency,0)==DD_OK)  break;
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),32,frequency,0)==DD_OK) break;
                if (m_ddraw->SetDisplayMode(mode.width(),mode.height(),24,frequency,0)==DD_OK) break;
                DirectXCheck(m_ddraw->SetDisplayMode(mode.width(),mode.height(),16,frequency,0));
            }
            break;

            default:
            {
                // error message
                throw Error("unsupported pixel format");
            }
        }
    }

    // update flags
    m_open = true;

    // setup mode
    m_mode = mode;
}


void DirectXDisplay::internal_open_nearest(const Mode &mode,bool exact,int frequency)
{
    // check resolutions list
    if (m_resolutions[0].valid())
    {
        // resolutions index
        int index = 0;

        // mode match
        Mode match;
        Mode match_exact;

        // search for closest matching resolution
        while (m_resolutions[index].valid())
        {
            // get resolution data
            const int width = m_resolutions[index].width();
            const int height = m_resolutions[index].height();

            // check for match
            if (mode.width()<=width && mode.height()<=height)
            {
                // check for exact match
                if (width==mode.width() && height==mode.height())
                {
                    // exact match
                    match_exact = Mode(width,height,mode.format());
                }

                // check if match is valid
                if (match.valid())
                {
                    // calculate difference between match and mode
                    const int dx1 = match.width() - mode.width();
                    const int dy1 = match.height() - mode.height();

                    // calculate difference between current resolution and mode
                    const int dx2 = width - mode.width();
                    const int dy2 = height - mode.height();

                    // compare differences
                    if (dx2<dx1 || dy2<dy1)
                    {
                        // update match
                        match = Mode(width,height,mode.format());
                    }
                }
                else
                {
                    // first match
                    match = Mode(width,height,mode.format());
                }

                // increase index
                index ++;
            }
            else
            {
                // stop
                break;
            }
        }

        // check exact match
        if (match_exact.valid())
        {
            try
            {
                // try exact match
                internal_open(match_exact,exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check match
        if (match.valid())
        {
            try
            {
                // try nearest match
                internal_open(match,exact,frequency);

                // success
                return;
            }
            catch (Error &)
            {
                // failure
            }
        }
    }

    // the mode list doesnt exist, or didnt help us find a mode... we will have to wing it!

    try
    {
        // try requested mode first
        internal_open(mode,exact,frequency);
    }
    catch (Error&)
    {
        // fallback to the nearest "standard" mode

        // check if 320x200 is a good match
        if (mode.width()<=320 && mode.height()<=200)
        {
            try
            {
                // open 320x200
                internal_open(Mode(320,200,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check if 320x240 is a good match
        if (mode.width()<=320 && mode.height()<=240)
        {
            try
            {
                // open 320x240
                internal_open(Mode(320,240,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check if 512x384 is a good match
        if (mode.width()<=512 && mode.height()<=384)
        {
            try
            {
                // open 512x384
                internal_open(Mode(512,384,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check if 640x400 is a good match
        if (mode.width()<=640 && mode.height()<=400)
        {
            try
            {
                // open 640x400
                internal_open(Mode(640,400,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check if 640x400 is a good match
        if (mode.width()<=640 && mode.height()<=480)
        {
            try
            {
                // open 640x480
                internal_open(Mode(640,480,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check if 800x600 is a good match
        if (mode.width()<=800 && mode.height()<=600)
        {
            try
            {
                // open 800x600
                internal_open(Mode(800,600,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check if 1024x768 is a good match
        if (mode.width()<=1024 && mode.height()<=768)
        {
            try
            {
                // open 1024x768
                internal_open(Mode(1024,768,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check if 1280x1024 is a good match
        if (mode.width()<=1280 && mode.height()<=1024)
        {
            try
            {
                // open 1280x1024
                internal_open(Mode(1280,1024,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }

        // check if 1600x1200 is a good match
        if (mode.width()<=1600 && mode.height()<=1200)
        {
            try
            {
                // open 1600x1200
                internal_open(Mode(1600,1200,mode.format()),exact,frequency);

                // success
                return;
            }
            catch (Error&)
            {
                // failure
            }
        }
    }

    // error message
    throw Error("could not set display mode");
}




HRESULT WINAPI DirectXDisplay::callback(LPDDSURFACEDESC descriptor,LPVOID context)
{
    // check pointers are valid (necessary?)
    if (!descriptor || !context) return DDENUMRET_CANCEL; 
    
    // cast context pointer to DirectXDisplay
    DirectXDisplay *display = (DirectXDisplay*)context;

    // check that the descriptor contains the data we want
    if (descriptor->dwFlags&DDSD_WIDTH && descriptor->dwFlags&DDSD_HEIGHT && descriptor->dwFlags&DDSD_PIXELFORMAT)
    {
        // setup display mode
        display->m_modes[display->m_index] = Mode(descriptor->dwWidth,descriptor->dwHeight,DirectXTranslate(descriptor->ddpfPixelFormat));

        // increase index
        display->m_index ++;
    }
    else
    {
        // we dont have the information required... hmppft!
    }

    // continue enumeration
    return DDENUMRET_OK;
}
