//
// Texture warp demo for OpenPTC 1.0 C++ API
// Copyright (c) 1998 Jonathan Matthew
// This source code is licensed under the GNU GPL
//

// include files
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "ptc.h"


// colour balance values.  change these if you don't like the colouring
// of the texture.

const int32 red_balance = 2;
const int32 green_balance = 3;
const int32 blue_balance = 1;

void blur(Surface &s)
{
    // lock surface
    char8 *d = (char8 *)s.lock();
    int pitch = s.pitch();
    int spack = (s.height()-1)*pitch;
    int spack2 = s.height()*pitch;
    int r;

    // first pixel
    for (r=0; r<4; r++) {
        d[r] = (d[pitch+r] + d[r+4] + d[spack+r] + d[pitch-4+r])/4;
    }

    // rest of first line
    for (; r<pitch; r++) {
        d[r] = (d[r+pitch] + d[r+4] + d[r-4] + d[spack+r])/4;
    }

    // rest of surface except last line
    for (; r<(s.height()-1)*pitch; r++) {
        d[r] = (d[r-pitch] + d[r+pitch] + d[r+4] + d[r-4])/4;
    }

    // last line except last pixel
    for (; r<(s.height()*pitch)-4; r++) {
        d[r] = (d[r-pitch] + d[r+4] + d[r-4] + d[r-spack])/4;
    }

    // last pixel
    for (; r<(s.height()*pitch); r++) {
        d[r] = (d[r-pitch] + d[r-4] + d[r-spack] + d[r+4-pitch])/4;
    }
    s.unlock();
}

void generate(Surface &surface)
{
    // draw random dots all over the surface
    int32 *dest = (int32 *)surface.lock();
    int i;
    for (i=0; i < surface.width() * surface.height(); i++) {
        int x = rand()%surface.width();
        int y = rand()%surface.height();
        int32 *d = dest + (y*surface.width()) + x;
        int32 cv = ((rand()%100)<<16) | ((rand()%100)<<8) | (rand()%100);
        *d++ = cv;
    }
    surface.unlock();

    // blur the surface
    for (i=0; i<5; i++) blur(surface);

    // multiply the color values
    dest = (int32 *)surface.lock();
    for (i=0; i<surface.width() * surface.height(); i++) {
        int32 cv = *dest;
        char8 r = (cv>>16)&255;
        char8 g = (cv>>8)&255;
        char8 b = cv&255;
        r *= red_balance;
        g *= green_balance;
        b *= blue_balance;
        if (r>255) r = 255;
        if (g>255) g = 255;
        if (b>255) b = 255;
        *dest++ = (r<<16)|(g<<8)|b;
    }
    surface.unlock();
}

void grid_map(int32 *grid, float xbase, float ybase, float xmove, float ymove, float amp)
{
    const float pi = 3.14159265358979323846;

    int x, y;
    float a,b,id;
    a = 0;

    for (y=0; y < 26; y++) {
        b = 0;
        for (x=0; x < 41; x++) {
            // it should be noted that there is no scientific basis for
            // the following three lines :)
            grid[0] = (int32)((xbase*14 + x*4 + xmove*sin(b)+sin(cos(a)*sin(amp))*15)*65536.0);
            grid[1] = (int32)((ybase*31 + y*3 + ymove*cos(b)*sin(sin(a)*cos(amp))*30)*65536.0);
            id = (cos(xbase)+sin(ybase)+cos(a*xmove*0.17)+sin(b*ymove*0.11)) * amp * 23;
            if (id < -127) {
                grid[2] = 0;
            } else if (id > 127) {
                grid[2] = 255<<16;
            } else {
                grid[2] = (128<<16) + (int)(id*65536.0);
            }
            grid += 3;
            b += pi/30;
        }
        a += pi/34;
    }
}

void make_light_table(char8 *lighttable)
{
    for (int i=0; i<256; i++) {
        for (int j=0; j < 256; j++) {
            // light table goes from 0 to i*2.
            int tv = (i*j)/128;
            if (tv > 255) tv = 255;
            lighttable[(j*256)+i] = tv;
        }
    }
}

// if you want to see how to do this properly, look at the tunnel3d demo.
// (not included in this distribution :)
void texture_warp(int32 *dest, int32 *grid, int32 *texture, char8 *lighttable)
{

    int utl,utr,ubl,ubr;
    int vtl,vtr,vbl,vbr;
    int itl,itr,ibl,ibr;
    int dudx,dvdx,didx,dudy,dvdy,didy,ddudy,ddvdy,ddidy;
    int dudx2,dvdx2,didx2;
    int bx,by,px,py;
    int uc,vc,ic,ucx,vcx,icx;

    int32 edi;
    int32 texel;
    int col;

    int32 *cbp;
    int32 *dp, dpix;

    cbp = grid;
    for (by=0; by<25; by++) {
        for (bx=0; bx<40; bx++) {
            utl = *cbp;
            vtl = *(cbp+1);
            itl = *(cbp+2);
            utr = *(cbp+(1*3));
            vtr = *(cbp+(1*3)+1);
            itr = *(cbp+(1*3)+2);
            ubl = *(cbp+(41*3));
            vbl = *(cbp+(41*3)+1);
            ibl = *(cbp+(41*3)+2);
            ubr = *(cbp+(42*3));
            vbr = *(cbp+(42*3)+1);
            ibr = *(cbp+(42*3)+2);
            dudx = (utr - utl) >> 3;
            dvdx = (vtr - vtl) >> 3;
            didx = (itr - itl) >> 3;
            dudx2 = (ubr - ubl) >> 3;
            dvdx2 = (vbr - vbl) >> 3;
            didx2 = (ibr - ibl) >> 3;
            dudy = (ubl - utl) >> 3;
            dvdy = (vbl - vtl) >> 3;
            didy = (ibl - itl) >> 3;
            ddudy = (dudx2 - dudx) >> 3;
            ddvdy = (dvdx2 - dvdx) >> 3;
            ddidy = (didx2 - didx) >> 3;
            uc = utl;
            vc = vtl;
            ic = itl;
            for (py=0; py<8; py++) {
                ucx = uc;
                vcx = vc;
                icx = ic;
                dp = dest + ( ((by*8+py)*320) + (bx*8) );
                for (px=0; px<8; px++) {

                    // get light table pointer for current intensity
                    char8 *ltp = lighttable + ((icx&0xff0000)>>8);

                    // get texel
                    edi = ((ucx&0xff0000)>>16)+((vcx&0xff0000)>>8);
                    texel = texture[edi];

                    // calculate actual colour
                    dpix = (int32)ltp[(texel>>16)&255];
                    dpix <<= 8;
                    dpix |= (int32)ltp[(texel>>8)&255];
                    dpix <<= 8;
                    dpix |= (int32)ltp[texel&255];

                    *dp++ = dpix;

                    ucx += dudx;
                    vcx += dvdx;
                    icx += didx;
                }

                uc += dudy;
                vc += dvdy;
                ic += didy;
                dudx += ddudy;
                dvdx += ddvdy;
                didx += ddidy;
            }
            cbp+=3;
        }
        cbp+=3;
    }
}

int main()
{
    try {
        // create format
        Format format(32, 0x00FF0000, 0x0000FF00, 0x000000FF);

        // create texture surface
        Surface texture(256,256,format);

        // create texture
        generate(texture);

        // create lighttable
        char8 *lighttable = new char8[256*256];
        make_light_table(lighttable);

        // create console
        Console console;

        // open console
        console.open("Warp demo",320,200,format);

        // create drawing surface
        Surface surface(320,200,format);

        // texture grid
        int32 grid[41*26*3];

        const float pi = 3.14159265358979323846;

        // control values
        float xbase = 0;
        float ybase = 0;
        float xmove = 0;
        float ymove = 0;
        float amp = 0;
        float dct = 0.024;
        float dxb = 0.031;
        float dyb = -0.019;
        float dxm = 0.015;
        float dym = -0.0083;

        // main loop
        while (!console.key()) {

            // create texture mapping grid
            grid_map(grid, xbase, ybase, xmove, ymove*3, amp);

            // map texture to drawing surface
            texture_warp((int32 *)surface.lock(), grid, (int32 *)texture.lock(), lighttable);
            texture.unlock();
            surface.unlock();

            // copy surface to console
            surface.copy(console);

            // update console
            console.update();

            // move control values (limit them so it doesn't go too far)
            xbase += dxb;
            if (xbase > pi) dxb = -dxb;
            if (xbase < -pi) dxb = -dxb;

            ybase += dyb;
            if (ybase > pi) dyb = -dyb;
            if (ybase < -pi) dyb = -dyb;

            xmove += dxm;
            if (xmove > 3.25) dxm = -dxm;
            if (xmove < -2.232) dxm = -dxm;

            ymove -= dym;
            if (ymove > 2.2352) dym = -dym;
            if (ymove < -3.574) dym = -dym;

            amp += dct;
            float sa = sin(amp);
            if (sa > -0.0001 && sa < 0.0001) {
                if (amp > 8.457547) dct = -dct;
                if (amp < -5.365735) dct = -dct;
            }
        }

        console.read();

    }
    catch (Error &e)
    {
        e.report();
    }
    return 0;
}
