//
// Generic C converter for the HERMES library
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Hermes.h"
#include "Base.h"
#include "Conv_GC.h"

static void (*Hermes_GC_Converter)(char8 *source,char8 *dest,
				   unsigned int count);
Hermes_Generic_Info Hermes_GC_GI;

BaseConverter Hermes_Generic_C=
{ NULL, 
  NULL,
  0,
  Hermes_GC_init,
  Hermes_GC_close,
  Hermes_GC_request,
  Hermes_GC_palette,
  Hermes_GC_copy
};


void Hermes_GC_init()
{ Hermes_Generic_C.m_source=(HermesFormat *)malloc(sizeof(HermesFormat)); 
  Hermes_Generic_C.m_dest=(HermesFormat *)malloc(sizeof(HermesFormat));
  Hermes_Generic_C.m_stretch=0;
}


void Hermes_GC_close()
{ if(Hermes_Generic_C.m_source)
  { free(Hermes_Generic_C.m_source);
    Hermes_Generic_C.m_source=0;
  }

  if(Hermes_Generic_C.m_dest)
  { free(Hermes_Generic_C.m_dest);
    Hermes_Generic_C.m_dest=0;
  }
}


char Hermes_GC_request(HermesFormat *source,HermesFormat *dest,char stretch)
{ char found=0;
  char generic=0;

  if(stretch) return 0;
  
  if(HermesFormatEquals(source,Hermes_Generic_C.m_source) &&
     HermesFormatEquals(dest,Hermes_Generic_C.m_dest) &&
     stretch==Hermes_Generic_C.m_stretch)
  return 1;


  // The monster block begins, but don't worry, normally this will only be
  // executed ONCE per program (at the first blit)
  // Apart from that, the most frequent modes will be found IMMEDIATELY,
  // because they form the first if.
  // And for the most UNUSUAL modes, this amounts to four comparisons and
  // four jumps in the assembler code.. And that ONCE per program, so it's
  // not too bad really.

  if(!source->indexed && !dest->indexed)
  switch(source->bpp)
  { 
    // From 32 bit to 32 bit   
    case 32: // Converters from the standard RGB888 format 
             if(source->mask_r==0xff0000 && source->mask_g==0xff00 &&
		source->mask_b==0xff)
             switch(dest->bpp)
	     { case 32: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
	                { Hermes_GC_Converter=Hermes_GC_32rgb888_32rgb888;
	                  found=1;
			  break;
			}
			else
			if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
			{ Hermes_GC_Converter=Hermes_GC_32rgb888_32bgr888;
			  found=1;
			  break;
			}
			else
	                { Hermes_GC_Converter=Hermes_GC_32rgb888_Generic32;
			  Hermes_Calculate_Generic_Info(24,16,8,
							dest,&Hermes_GC_GI);
			  generic=1;
			  found=1;
			}
			break;
			
	       case 24: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
		        { Hermes_GC_Converter=Hermes_GC_32rgb888_24rgb888;
		          found=1;
			  break;
			}
	                else
			if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
			{ Hermes_GC_Converter=Hermes_GC_32rgb888_24bgr888;
			  found=1;
			}
			break;

	       case 16: if(dest->mask_r==0xf800 && dest->mask_g==0x7e0 && 
			   dest->mask_b==0x1f)
	                { Hermes_GC_Converter=Hermes_GC_32rgb888_16rgb565;
			  found=1;
			  break;
			}
			else
			if(dest->mask_r==0x1f && dest->mask_g==0x7e0 && 
			   dest->mask_b==0xf800)
	                { Hermes_GC_Converter=Hermes_GC_32rgb888_16bgr565;
			  found=1;
			  break;
			}
	                else
			if(dest->mask_r==0x7c00 && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x1f)
			{ Hermes_GC_Converter=Hermes_GC_32rgb888_16rgb555;
			  found=1;
			  break;
			}
			else
			if(dest->mask_r==0x1f && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x7c00)
			{ Hermes_GC_Converter=Hermes_GC_32rgb888_16bgr555;
			  found=1;
			  break;
			}
	                else
			{ Hermes_GC_Converter=Hermes_GC_32rgb888_Generic16;
			  Hermes_Calculate_Generic_Info(24,16,8,
							dest,&Hermes_GC_GI);
			  generic=1;
			  found=1;
			}
	                break;

	        case 8: if(dest->mask_r==0xe0 && dest->mask_g==0x1c &&
			   dest->mask_b==0x3)
		        { Hermes_GC_Converter=Hermes_GC_32rgb888_8rgb332;
			  found=1;
			}
		        break;	     
	     }
	     else // Converters from muhmu format
	     if(source->mask_r==(0xff<<20) && source->mask_g==(0xff<<10) &&
		source->mask_b==0xff)
             switch(dest->bpp)
	     { case 32: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
	                { Hermes_GC_Converter=Hermes_GC_muhmu32_32rgb888;
	                  found=1;
			  break;
			}
			else
			if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
			{ Hermes_GC_Converter=Hermes_GC_muhmu32_32bgr888;
			  found=1;
			  break;
			}
			else
	                { Hermes_GC_Converter=Hermes_GC_muhmu32_Generic32;
			  Hermes_Calculate_Generic_Info(24,16,8,
							dest,&Hermes_GC_GI);
			  generic=1;
			  found=1;
			}
			break;
			
	       case 24: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
		        { Hermes_GC_Converter=Hermes_GC_muhmu32_24rgb888;
		          found=1;
			  break;
			}
	                else
			if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
			{ Hermes_GC_Converter=Hermes_GC_muhmu32_24bgr888;
			  found=1;
			}
			break;

	       case 16: if(dest->mask_r==0xf800 && dest->mask_g==0x7e0 && 
			   dest->mask_b==0x1f)
	                { Hermes_GC_Converter=Hermes_GC_muhmu32_16rgb565;
			  found=1;
			  break;
			}
			else
			if(dest->mask_r==0x1f && dest->mask_g==0x7e0 && 
			   dest->mask_b==0xf800)
	                { Hermes_GC_Converter=Hermes_GC_muhmu32_16bgr565;
			  found=1;
			  break;
			}
	                else
			if(dest->mask_r==0x7c00 && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x1f)
			{ Hermes_GC_Converter=Hermes_GC_muhmu32_16rgb555;
			  found=1;
			  break;
			}
			else
			if(dest->mask_r==0x1f && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x7c00)
			{ Hermes_GC_Converter=Hermes_GC_muhmu32_16bgr555;
			  found=1;
			  break;
			}
	                else
			{ Hermes_GC_Converter=Hermes_GC_muhmu32_Generic16;
			  Hermes_Calculate_Generic_Info(24,16,8,
							dest,&Hermes_GC_GI);
			  generic=1;
			  found=1;
			}
	                break;

	        case 8: if(dest->mask_r==0xe0 && dest->mask_g==0x1c &&
			   dest->mask_b==0x3)
		        { Hermes_GC_Converter=Hermes_GC_muhmu32_8rgb332;
			  found=1;
			}
		        break;
	     
	     }	     
	     else     // Convert from non-standard source format
	     { switch(dest->bpp)
	       { case 32: Hermes_GC_Converter=Hermes_GC_Generic32_Generic32;
              	          generic=1;
	                  found=1;
	                  break;
	     
	         case 24: Hermes_GC_Converter=Hermes_GC_Generic32_Generic24;
              	          generic=1;
	                  found=1;
	                  break;
	       
                 case 16: Hermes_GC_Converter=Hermes_GC_Generic32_Generic16;
              	          generic=1;
	                  found=1;
	                  break;
		 
	          case 8: Hermes_GC_Converter=Hermes_GC_Generic32_Generic8;
              	          generic=1;
	                  found=1;
	                  break;
	       }

               Hermes_Calculate_Generic_Info(Hermes_Topbit(source->mask_r),
					     Hermes_Topbit(source->mask_g),
					     Hermes_Topbit(source->mask_b),
					     dest,&Hermes_GC_GI);
	     }
	     break;


    // From 24 bit to *
    case 24: // From standard format rgb888
             if(source->mask_r==0xff0000 && source->mask_g==0xff00 && 
		source->mask_b==0xff)
             switch(dest->bpp)
	     { case 32: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
	                { Hermes_GC_Converter=Hermes_GC_24rgb888_32rgb888;
			  found=1;
			  break;
			}
			else
			{ Hermes_GC_Converter=Hermes_GC_24rgb888_Generic32;
	                  Hermes_Calculate_Generic_Info(24,16,8,
				  		        dest,&Hermes_GC_GI);
			  generic=1;
			  found=1;
			}
			break;

	       case 24: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
		        { Hermes_GC_Converter=Hermes_GC_24rgb888_24rgb888;
			  found=1;
			  break;
			} 
	                else
			if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
			{ Hermes_GC_Converter=Hermes_GC_24rgb888_24bgr888;
			  found=1;
			}
	                break;
	                
	       case 16: if(dest->mask_r==0xf800 && dest->mask_g==0x7e0 && 
			   dest->mask_b==0x1f)
		        { Hermes_GC_Converter=Hermes_GC_24rgb888_16rgb565;
			  found=1;
			  break;
			}
	                else
			if(dest->mask_r==0x7c00 && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x1f)
			{ Hermes_GC_Converter=Hermes_GC_24rgb888_16rgb555;
			  found=1;
			  break;
			}
	                else
			{ Hermes_GC_Converter=Hermes_GC_24rgb888_Generic16;
			  Hermes_Calculate_Generic_Info(24,16,8,dest,
							&Hermes_GC_GI);
			  generic=1;
			  found=1;
                        }
	                break;

	        case 8: if(dest->mask_r==0xe0 && dest->mask_g==0x1c &&
			   dest->mask_b==0x3)
		        { Hermes_GC_Converter=Hermes_GC_24rgb888_8rgb332;
			  found=1;
			}
		        break;
	     }
	     break;


    // From 16 bit to * 
    case 16: // From standard 16 bit rgb565
             if(source->mask_r==0xf800 && source->mask_g==0x7e0 && 
		source->mask_b==0x1f)
             switch(dest->bpp)
             { case 32: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
	                { Hermes_GC_Converter=Hermes_GC_16rgb565_32rgb888;
			  found=1;
			  break;
			}
			else
	                if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
	                { Hermes_GC_Converter=Hermes_GC_16rgb565_32bgr888;
			  found=1;
			  break;
			}
			else
			{ Hermes_GC_Converter=Hermes_GC_16rgb565_Generic32;
			  Hermes_Calculate_Generic_Info(16,11,5,dest,
							&Hermes_GC_GI);
			  generic=1;
			  found=1;
			}
			break;


	       case 24: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
		        { Hermes_GC_Converter=Hermes_GC_16rgb565_24rgb888;
			  found=1;
			  break;
			}
	                else
	                if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
		        { Hermes_GC_Converter=Hermes_GC_16rgb565_24bgr888;
			  found=1;
			  break;
			}
	                break;

	       case 16: if(dest->mask_r==0xf800 && dest->mask_g==0x7e0 &&
			   dest->mask_b==0x1f)
		        { Hermes_GC_Converter=Hermes_GC_16rgb565_16rgb565;
			  found=1;
			  break;
			} 
	                else
	                if(dest->mask_r==0x1f && dest->mask_g==0x7e0 &&
			   dest->mask_b==0xf800)
		        { Hermes_GC_Converter=Hermes_GC_16rgb565_16bgr565;
			  found=1;
			  break;
			}
	                else 
			if(dest->mask_r==0x7c00 && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x1f)
		        { Hermes_GC_Converter=Hermes_GC_16rgb565_16rgb555;
			  found=1;
			  break;
			} 
			else 
			if(dest->mask_r==0x1f && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x7c00)
		        { Hermes_GC_Converter=Hermes_GC_16rgb565_16bgr555;
			  found=1;
			  break;
			}
	                else
		        { Hermes_GC_Converter=Hermes_GC_16rgb565_Generic16;
			  Hermes_Calculate_Generic_Info(16,11,5,dest,
							&Hermes_GC_GI);
			  generic=1;
			  found=1;
			}
			break;


	        case 8: if(dest->mask_r==0xe0 && dest->mask_g==0x1c &&
			   dest->mask_b==0x3)
		        { Hermes_GC_Converter=Hermes_GC_16rgb565_8rgb332;
			  found=1;
			}
		        break;
	     }
	     else     // Convert from non-standard source format
	     { switch(dest->bpp)
	       { case 32: Hermes_GC_Converter=Hermes_GC_Generic16_Generic32;
              	          generic=1;
	                  found=1;
	                  break;
	     
	         case 24: Hermes_GC_Converter=Hermes_GC_Generic16_Generic24;
              	          generic=1;
	                  found=1;
	                  break;
	       
                 case 16: Hermes_GC_Converter=Hermes_GC_Generic16_Generic16;
              	          generic=1;
	                  found=1;
	                  break;
		 
	          case 8: Hermes_GC_Converter=Hermes_GC_Generic16_Generic8;
              	          generic=1;
	                  found=1;
	                  break;
	       }

               Hermes_Calculate_Generic_Info(Hermes_Topbit(source->mask_r),
					     Hermes_Topbit(source->mask_g),
					     Hermes_Topbit(source->mask_b),
					     dest,&Hermes_GC_GI);
	     }

             break;

  }
  else
  if(source->indexed && !dest->indexed && source->bpp==8 )
  { switch(dest->bpp)
    { case 32: Hermes_GC_Converter=Hermes_GC_index8_32;
               found=1;
               break;

      case 24: Hermes_GC_Converter=Hermes_GC_index8_24;
	       found=1;
	       break;

      case 16: Hermes_GC_Converter=Hermes_GC_index8_16;
               found=1;
               break;
    }

    // Set up shift information for lookup creator
    if(found)
    Hermes_Calculate_Generic_Info(24,16,8,dest,&Hermes_GC_GI);
  }
  else
  if(source->indexed && source->bpp==8 && dest->indexed && dest->bpp==8)
  { Hermes_GC_Converter=Hermes_GC_index8_i8;
    found=1;
  }

  if(found)
  { HermesFormatCopy(source,Hermes_Generic_C.m_source);
    HermesFormatCopy(dest,Hermes_Generic_C.m_dest);
    Hermes_Generic_C.m_stretch=stretch;

    printf("Hermes C Converter ");
    if(generic) printf("(Generic)");
    printf("\n");

    return 1;
  }

  // Couldn't find a converter
  return 0;
}


void Hermes_GC_palette(int32 *palette)
{ unsigned int i;
  int32 r,g,b;

  if(!Hermes_Generic_C.m_source->indexed)
  return;

  // If no left shifts are required, which is the most common case, we 
  // use a faster loop
  if(!Hermes_GC_GI.r_left && !Hermes_GC_GI.g_left && !Hermes_GC_GI.b_left)
  for(i=0;i<256;i++)
  { r=(palette[i]>>Hermes_GC_GI.r_right)&Hermes_Generic_C.m_dest->mask_r;
    g=(palette[i]>>Hermes_GC_GI.g_right)&Hermes_Generic_C.m_dest->mask_g;
    b=(palette[i]>>Hermes_GC_GI.b_right)&Hermes_Generic_C.m_dest->mask_b;
   
    Hermes_Lookup[i]=r|g|b;
  }   
  else
  for(i=0;i<256;i++)
  { r=((palette[i]>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((palette[i]>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((palette[i]>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
   
    Hermes_Lookup[i]=r|g|b;
  }  
}


void Hermes_GC_copy(void *s_pixels,int s_x,int s_y,int s_width,int s_height,
		   int s_pitch,void *d_pixels,int d_x,int d_y,int d_width,
		   int d_height,int d_pitch)
{ char8 *sp,*dp;

  if(s_width<=0 || d_width<=0) return;
  if(s_height<=0 || d_height<=0) return;

  // In case the destination gets clipped..
  if(d_width<s_width) s_width=d_width;
  if(d_height<s_height) s_height=d_height;

  sp=(char8 *)s_pixels+s_y*s_pitch+s_x*(Hermes_Generic_C.m_source->bpp>>3);
  dp=(char8 *)d_pixels+d_y*d_pitch+d_x*(Hermes_Generic_C.m_dest->bpp>>3);

  while(s_height--)
  { Hermes_GC_Converter(sp,dp,s_width);

    sp+=s_pitch;
    dp+=d_pitch;
  }  
}
