//
// Visual class for PTC 2.0 C++ API
// Copyright (c) 1998 Jarno Paananen (jpaana@s2.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "Core/Area.h"
#include "Core/Clip.h"
#include "Core/Error.h"
#include "Core/Color.h"
#include "Core/Palette.h"
#include "Base/Surface.h"
#include "GGI/Visual.h"

// using directive
using namespace ptc::GGI;

// using declarations
using ptc::Core::Area;
using ptc::Core::Clip;
using ptc::Core::Error;
using ptc::Core::Color;
using ptc::Core::Format;
using ptc::Core::Palette;




Visual::Visual(char* visual, int width, int height,
               const ptc::Base::Format &format, bool fullscreen)
{
    // defaults
    m_width   = 0;
    m_height  = 0;
    m_vis = NULL;

    ggiInit();
    
    try
    {
        m_vis = ggiOpen(visual);
        if (!m_vis)
            throw Error("Can't open visual");

        ggiAddFlags(m_vis, GGIFLAG_ASYNC);

        ggi_graphtype       type;

        switch( format.bits() )
        {
            case 32:
                type = GT_32BIT;
                break;
            case 24:
                type = GT_24BIT;
                break;
            case 16:
                type = GT_16BIT;
                break;
            case 15:
                type = GT_15BIT;
                break;
            case 8:
                type = GT_8BIT;
                break;
        }

        ggiCheckGraphMode(m_vis, width, height, width, height, type,
                          &m_mode);

        ggiSetMode(m_vis, &m_mode);

        m_width = m_mode.virt.x;
        m_height = m_mode.virt.y;

        m_format = translate(m_mode.graphtype);
    }
    catch (Error &error)
    {
        // close
        close();

        // error message
        error.rethrow("could not create visual");
    }
}


Visual::~Visual()
{
    // close
    close();
}




void Visual::update()
{
    ggiFlush(m_vis);
}




void Visual::copy(ptc::Base::Surface &surface)
{
    // lock visual
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console to surface");
    }
}


void Visual::copy(ptc::Base::Surface &surface,const ptc::Base::Area &source,const ptc::Base::Area &destination)
{
    // lock visual
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),source,destination,format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console area to surface area");
    }
}




void* Visual::lock()
{
    const ggi_directbuffer *db;
    
    db = ggiDBGetBuffer(m_vis, 0);
    return db->write;
}


void Visual::unlock()
{
}



void Visual::load(const void *pixels,int width,int height,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,0,0,width,height,width*format.bytes(),console_pixels,0,0,this->width(),this->height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to load pixels to console");
    }
}


void Visual::load(const void *pixels,int width,int height,const ptc::Base::Area &source,const ptc::Base::Area &destination,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
	    // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,width,height),source);
        Area clipped_destination = Clip::clip(Area(0,0,this->width(),this->height()),destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),width*format.bytes(),
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to load pixels to console area");
    }
}




void Visual::save(void *pixels,int width,int height,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(m_format,format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,width*format.bytes());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to save console pixels");
    }
}


void Visual::save(void *pixels,int width,int height,const ptc::Base::Area &source,const ptc::Base::Area &destination,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,this->width(),this->height()),source);
        Area clipped_destination = Clip::clip(Area(0,0,width,height),destination);

        // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),width*format.bytes());
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to save console area pixels");
    }
}




void Visual::clear(const ptc::Base::Color &color)
{
    // lock console
    void *pixels = lock();

    try
    {
        // request clear
        m_clear.request(format());

        // clear console
        m_clear.clear(pixels,0,0,width(),height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console");
    }
}


void Visual::clear(const ptc::Base::Color &color,const ptc::Base::Area &area)
{
    // lock console
    void *pixels = lock();

    try
    {
        // clip area
        Area clipped_area = Clip::clip(Area(0,0,width(),height()),area);

        // request clear
        m_clear.request(format());

        // clear console area
        m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console area");
    }
}




void Visual::palette(const ptc::Base::Palette &palette)
{
    // assign palette
    m_palette = palette;

    // get palette data
    const int32 *data = palette.data();

    // check format and palette
    if (!m_format.indexed()) return;

    // convert palette data
    ggi_color temp[256];
    for (int i=0; i<256; i++)
    {
        temp[i].r   = ( (data[i] & 0x00FF0000) >> 16 );
        temp[i].g   = ( (data[i] & 0x0000FF00) >> 8  );
        temp[i].b   = ( (data[i] & 0x000000FF) >> 0  );
        temp[i].a = 0xff;
    }

    // set palette entries
    ggiSetPalette(m_vis, 0, 256, temp);
}


const ptc::Base::Palette& Visual::palette() const
{
    // get palette
    return m_palette;
}




int Visual::width() const
{
    // get width
    return m_width;
}


int Visual::height() const
{
    // get height
    return m_height;
}


int Visual::pitch() const
{
    // return surface pitch
    return m_mode.virt.x * (m_format.bits()/8);
}


const ptc::Base::Format& Visual::format() const
{
    // get format
    return m_format;
}




Format Visual::translate(const ggi_graphtype &type)
{
    // pixel format
    Format format;

    switch ( GT_SIZE(type) )
    {
        case 32:
            format=Format(32, 0xff0000, 0xff00, 0xff);
            break;
        case 24:
            format=Format(24, 0xff0000, 0xff00, 0xff);
            break;
        case 16:
            format=Format(16, 0xf80, 0x7e0, 0x1f);
            break;
        case 15:
            format=Format(15, 0x7c0, 0x3e, 0x1f);
            break;
        case 8:
            format=Format(8);
            break;
        default: 
            throw Error("invalid pixel format");
            break;
    }

    // return format
    return format;
}




void Visual::close()
{
    if ( m_vis )
        ggiClose(m_vis);
    m_vis = NULL;
}

int Visual::key() const
{
    return ggiKbhit(m_vis);
}

int Visual::read() const
{
    return ggiGetc(m_vis);
}
