//
// Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "Key.h"
#include "Area.h"
#include "Color.h"
#include "Error.h"
#include "Config.h"
#include "Clipper.h"
#include "Console.h"
#include <string.h>
#include <fstream.h>




Console::Console()
{
    //
    // Console constructor
    // -------------------
    //
    // When a console is created it is initially closed and unlocked.
    //
    // Most functions of the console cannot be used until the
    // console has been opened with "Console::open".
    //
    // The default constructor automatically reads option strings 
    // from the "ptc.cfg" file in the working directory if it exists.
    //

    // defaults
    m_width  = 0;
    m_height = 0;
    m_pages  = 0;
    m_pitch  = 0;
    m_open   = false;
    m_locked = false;

    // clear strings
    m_title[0] = 0;
    m_information[0] = 0;

    // configure console
    configure("ptc.cfg");
}


Console::~Console()
{
    //
    // Console destructor
    // ------------------
    //    
    // The destructor automatically closes the console.
    //

    // close
    close();
}




void Console::configure(const char file[])
{
    //
    // Configure console
    // -----------------
    //
    // This function reads console options from the text file specified by
    // the 'file' parameter and pipes them through the "option" function. 
    // If the configuration file does not exist then this function fails 
    // silently.
    //
    // The console constructor calls this function with the file "ptc.cfg" 
    // so that the end user of a program developed with ptc can have some
    // level of control over its operation. This is necessary because in 
    // most cases the application developer does not usually think of
    // providing this level of flexibility to the user.
    //
    // This function is valid at all times, however the options specified
    // in the configuration file may not be. See Console::option for more
    // information on console options.
    //

    // open configuration file
    ifstream is(file,ios::in | ios::nocreate);
    
    // check stream
    if (!is) return;

    // read option strings
    while (!is.eof())
    {
        // option line
        char line[1024];

        // read line from stream
        is.getline(line,1024);

        // process option
        option(line);
    }
}




bool Console::option(const char option[])
{
    // 
    // Console option string
    // ---------------------
    //
    // Pass a case independent option string to the console.
    //
    // Option strings are defined by the implementer of the
    // distribution in order to give the user some control over
    // the console that cannot be obtained through the standard
    // platform independent console interface.
    //
    // For example, under DirectX it is possible to have both
    // fullscreen and windowed output, and there is no way for
    // the user to choose which one they want when they open a
    // console. However, by using Console::option the implementer
    // can trap option strings to let the user control whether
    // the console uses fullscreen or windowed output:
    //
    //     console.option("use windowed output");
    //     console.option("use fullscreen output");
    //     console.option("use default output");
    //
    // Currently there are no platform independent option strings
    // so the only option strings that need to be recognized are
    // the platform specific ones that you want to add.
    //
    // This function returns 'true' if the option string was
    // recognized, 'false' if the console did not recognize or
    // accept it.
    //
    // It is up to the implementer of the option to document when
    // the option string may be called. Some options need to be
    // called before the display is opened to have effect (such
    // as the example above) and others may need to be called
    // after the display is opened, others still can be set at 
    // any time without problems. Ideally an option string will
    // be valid at any time.
    //
    // Note that this implementation also passes the option
    // string to the copy object in order to provide control over
    // copying from the console to other surfaces.
    //
    // See "Copy::option" for more details on copy option strings.
    //

    // option handled flag
    bool recognized = false;

    // [platform dependent option string interpretation code]

    // pass the option along to the copy object as well
    if (m_copy.option(option)) recognized = true;

    // return recognized flag
    return recognized;
}




const Mode* Console::modes()
{
    //
    // Get console modes
    // -----------------
    //
    // This function returns a pointer to a const array of mode
    // objects representing all of the fullscreen display modes
    // that are supported by the console.
    //
    // If a console operates under a windowed environment, or the
    // platform does not provide access to a list of modes, then
    // this function can be left as it is. The implementation below
    // just returns an empty mode list.
    //
    // However, if the platform supports a list of fullscreen
    // display modes you should provide mode list support to the
    // console by initializing the mode objects in the 'm_modes'
    // array with all of the display modes that the platform
    // supports and terminating the array with a mode class created
    // with the default constructor (an 'invalid' mode).
    //
    // Here is some pseudocode to setup the mode list:
    //
    //     // mode index
    //     int index = 0;
    //
    //     // iterate through all modes
    //     while (there_are_more_modes())
    //     {
    //         // get mode information
    //         int width = get_current_mode_width();
    //         int height = get_current_mode_height();
    //         Format format = get_current_mode_format();
    //
    //         // setup mode in mode list
    //         m_modes[index] = Mode(width,height,format);
    //      
    //         // next display mode
    //         go_to_next_mode();
    //         index ++;
    //     }
    //
    //     // terminate the mode list
    //     m_modes[index] = Mode();
    //
    // This function is valid at all times.
    //

    // [platform dependent code to get a list of display modes]

    // return modes
    return m_modes;
}




void Console::open(const char title[],int pages)
{
    //
    // Open console
    // ------------
    //
    // This function opens a console using the default width, height
    // and pixel format. The user calls this open function perhaps
    // when they really do not mind what output width, height or
    // pixel format is supplied.
    //
    // This open function must always succeed. It should search for
    // the 'best' width, height and pixel format available on the
    // platform and open the console matching these parameters.
    //
    // The best console width and height is of course entirely platform
    // dependent, however the best console format is easily defined.
    // The best pixel format is a direct color format, with the highest
    // number of bits per pixel. For example, 32 bit RGB888 would be
    // chosen over 16bit RGB565. If direct color is not possible on the
    // display then an indexed color format should be used as a last
    // resort.
    //
    // See the open functions below for more details.
    //

    // default format
    Format format(32,0x00FF0000,0x0000FF0000,0x000000FF);

    // open console
    open(title,format,pages);
}


void Console::open(const char title[],const Format &format,int pages)
{
    //
    // Open console
    // ------------
    //
    // This function opens a console using the default width and height.
    // The user calls this open function when they do not care so much
    // about the actual resolution of the display being opened, but
    // still have a preference for a pixel format.
    //
    // It is up to the implementer to determine what width and height
    // should be the default for the platform. If in doubt, leave the
    // choice up to the window manager (if applicable), or get a list
    // of available display modes and make an intelligent decision
    // which is the best to use. If a mode list is not provided just
    // pick the most commonly supported display mode (eg. 320x200 on 
    // PC VGA or 640x480 under VESA and DirectX). 
    //
    // See the open function below for more details.
    //

    // default width and height
    int width  = 320;
    int height = 200;

    // open console
    open(title,width,height,format,pages);
}


void Console::open(const char title[],int width,int height,const Format &format,int pages)
{
    //
    // Open console
    // ------------
    //
    // This function opens the console mode which is the closest match
    // to the width, height and format requested.
    //
    // The width and height parameters are only hints. If a display mode 
    // exists that matches exactly then that display mode should always 
    // be set, however if no exact match is possible then a display mode, 
    // or display window can be opened with the 'best match' to the width
    // and height parameters. What the best match actually is, of course 
    // is entirely platform dependent. The only requirement for this match
    // is that it must be possible to copy a surface of size width,height
    // to the console. As long as this requirement is met, the implementer
    // is free to do whatever they want internally when the console is
    // opened.
    //
    // As with the width and height parameters, the 'format' parameter
    // is only a hint and the implementer is free to open a display with
    // any pixel format they want provided that it is possible to convert
    // from the 'format' specified to the pixel format of the console.
    //

    // [platform dependent code to find the closest matching console mode]
    
    // find closest matching console mode
    Mode mode = Mode(width,height,format);

    // open console
    open(title,mode,pages);
}


void Console::open(const char title[],const Mode &mode,int pages)
{
    //
    // Open console
    // ------------
    //
    // Opening a console means different things on different platforms.
    // Under window systems opening a console generally means opening
    // a window to display the console output, under fullscreen graphics
    // systems opening a console corresponds to setting a display mode.
    //
    // It is transparent to the user what actually happens when a console
    // is opened. The only thing that matters is that somehow the console
    // provides graphical output.
    //
    // If the platform supports it, the title of the window, application, 
    // or whatever the operating system uses to identify the title of the 
    // application should be set to the 'title' parameter. For example, 
    // under Win32 the title bar of the window would be set to 'title',
    // and under DOS the title parameter would be ignored
    //
    // The final parameter 'pages' is a hint. This hint is lets the user
    // control the number of console pages to be initialized. For example,
    // when pages is 3 the user is requesting triple buffering, and when
    // the pages parameter is set to 1 only a single page is requested.
    // When the pages parameter is 0, the user is indicating that they
    // wish to use the default number of console pages.
    //
    // The open function must always make sure that it closes any existing
    // display windows that have been opened in this class, and that it
    // can correctly manage multiple consoles being opened simultaneously
    // in one program. For example, its perfectly valid in a windowed 
    // environment to have multiple consoles in a program because each 
    // console open creates its own output window. However in a fullscreen
    // graphics it is not allowed because there is only one display mode.
    // Any attempt to open multiple consoles simultaneously where it does
    // not make sense to do so must fail by throwing an error.
    //
    // After a console is opened all console pixel buffers, surfaces, virtual
    // screens, video pages etc. must be cleared to black in direct color or
    // cleared to index zero in indexed color. The console palette must also 
    // be cleared to have all entries set to black.
    //

    // check that the mode is valid
    if (!mode.valid()) throw Error("invalid mode");

    // close
    close();

    // check pages
    if (pages==0)
    {
        //
        // Default number of console pages
        // -------------------------------
        //
        // The default number of console pages is platform dependent. 
        // However, if given the choice you should always use more 
        // than one page. Single buffering causes tearing and flicker
        // when the user renders directly to the visible page and this 
        // is unacceptable.
        //
        // Two or three pages (double or triple buffering) are good
        // default values for the number of pages to be used.
        //
        // For more information on console pages see "Console::pages".
        //
        
        // [platform dependent code to get the default number of console pages]

        // default
        pages = 2;
    }

    // [platform dependent code to open the console with a specific mode]

    // set open flag
    m_open = true;

    // [platform dependent code to clear all console buffers]

    // clear palette
    palette(Palette());

    // setup console data
    m_width  = mode.width();
    m_height = mode.height();
    m_format = mode.format();
    m_pitch  = mode.width() * mode.format().bytes();
    m_pages  = pages;

    //
    // On some platforms the pitch is different to what is coded above. In 
    // this case you must query the platform for the pitch of the display,
    // instead of just calculating it from the width.
    //
    // See Console::pitch for more information.
    //

    // setup console area
    m_area = Area(0,0,width(),height());

    // setup clip area
    m_clip = m_area;

    // check length of console title
    if (strlen(title)+1<=sizeof(m_title))
    {
        // update console title
        strcpy(m_title,title);
    }
    else
    {
        // update console title truncated
        memcpy(m_title,title,sizeof(m_title));
        m_title[sizeof(m_title)-1] = 0;
    }
    
    // [ platform dependent code to setup console information string ]

    // setup information
    m_information[0] = 0;
}


void Console::close()
{
    //
    // Close console
    // -------------
    //
    // Closing a console leaves any fullscreen display mode, closes any
    // display windows, and cleans up whatever resources that were created 
    // when the console was opened.
    //
    // The close function must be able to handle being called multiple times
    // without problems. For example, if close is called and the console is 
    // not open then close must return without doing anything.
    //
    // If the console is locked then the close function throws an error
    // exception. The user must unlock the console before calling close.
    //
    // Any key presses that remain in the key buffer since the last call to
    // "open" are flushed by close. This prevents garbage key strokes being
    // passed on from one open console to another or being passed back to the 
    // operating system after the program exits.
    //
    // This function is only valid when the console is unlocked.
    //

    // check if open
    if (m_open)
    {
        // the console must be unlocked when closed
        if (m_locked) throw Error("console is still locked");

        // flush all key presses
        while (key()) read();

        // [platform dependent code to close the console]

        // defaults
        m_width  = 0;
        m_height = 0;
        m_pages  = 0;
        m_pitch  = 0;
        m_format = Format();

        // clear strings
        m_title[0] = 0;
        m_information[0] = 0;

        // clear areas
        m_area = Area();
        m_clip = Area();

        // console is closed
        m_open = false;
    }
}




void Console::flush()
{
    //
    // Flush console operations
    // ------------------------
    //
    // Some console operations such as copying and clearing may be 
    // performed in parallel with the main system on accelerated graphics 
    // hardware. In cases where this is true, a call to flush ensures that 
    // all of these pending operations complete in a finite time. 
    //
    // This function does not guarantee that all operations have completed 
    // before it returns. Console::finish should be used instead if this is 
    // what is required.
    //
    // Additionally, this function makes sure that on windowed systems
    // all window events are processed. This advoids problems caused by
    // ignoring window events such as an unresponsive console window.
    //
    // No updating of the console occurs when this function is called.
    // If this is required, Console::update should be called instead
    // of Console::flush.
    //
    // On many systems this function performs no functionality, for
    // example on DOS Console::flush has no meaning because there is
    // no windowing system that needs window messages to be processed,
    // and under VGA/VBE there are no parallel console operations. If 
    // this is the case then Console::flush should just be left as an 
    // empty function.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug checks
    check_open();
    check_unlocked();

    // [platform dependent code to flush all console operations]

    // [platform dependent code to process all console window messages]
}


void Console::finish()
{
    //
    // Finish console operations
    // -------------------------
    //
    // This function operates in the same manner that Console::flush
    // does except that it does not return until all pending console
    // operations have been completed.
    //
    // On many systems this function performs no functionality, If this 
    // is the case then Console::finish should just be left as an empty
    // function.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();

    // [platform dependent code to finish all console operations]

    // [platform dependent code to process all console window messages]
}


void Console::update()
{
    //
    // Update console
    // --------------
    //
    // This function updates the screen with the contents of the console.
    //
    // On different platforms the exact implementation of Console::update
    // will vary widely. 
    //
    // The best case scenario is that where there are multiple video pages 
    // in memory and Console::update just swaps the pages and makes the page 
    // that was just rendered to the current visual page. If the platform
    // supports synchronizing page flips to the vertical retrace then
    // it should be used by default.
    //
    // Another situation is where you have only one video memory page (the
    // visual page). In this case you should typically render to buffer in
    // system memory, and on each call to update copy the system memory
    // surface to the display memory. Again if the system provides some way
    // to synchronize the console update with the vertical retrace it should
    // be used by default.
    //
    // The final scenario is where there is no direct access to video memory.
    // This is quite typical under windowed systems, and in this case update
    // would use native operating system calls to copy the console to the 
    // display. For example under Win32 GDI console operations would render
    // to an offscreen bitmap and use native Win32 GDI calls to draw the 
    // console bitmap in the output window.
    //
    // It is up to the implementer to determine what type of strategy they
    // will use when coding the console update for their platform.
    //
    // Additionally, Console::update makes sure that on windowed systems
    // all window events are processed. This advoids problems caused by
    // ignoring window events such as an unresponsive console window.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();

    // [platform dependent code to update the console]

    // [platform dependent code to process all console window messages]
}


void Console::update(const Area &area)
{
    //
    // Update console area
    // -------------------
    //
    // This function updates the contents of the console, hinting that only
    // the part of the console contained in the 'area' parameter needs to 
    // actually be updated.
    //
    // This can give quite a speedup under windowed systems where you would
    // otherwise have to tell the gui to draw the entire console bitmap each
    // call to update when only a small portion of the console has in fact
    // changed.
    //
    // The area parameter is only a hint. There is no guarantee that only
    // this area will be updated. For example, a page flipping system cannot
    // update only a small area of the console, it can only update all or 
    // nothing. In the case of page flipping systems, just ignore the area
    // as coded below and update the whole console.
    //
    // If you replace the code below with your own area update, please make
    // sure you copy the "debug check" code from the update function above
    // into your area update code, and if applicable, the window messages
    // processing code. You will also have to clip the update area with the
    // code below:
    //
    //     // clip update area against console area
    //     Area clipped = Clipper::clip(area,this->clip());
    //
    // This function is only valid if the console is open and unlocked.
    //

    // update
    update();
}




bool Console::key()
{
    //
    // Check for key press
    // -------------------
    //
    // This function should return 'true' if a key has been pressed
    // and is ready to be read from the keyboard buffer with 'read'.
    // 
    // The operation of this function is similar to the 'kbhit' 
    // function that PC programmers are familiar with under DOS and
    // Win32 console applications.
    //
    // Console::key returns immediately whether or not a key is waiting
    // in the key buffer.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // [platform dependent code to check if there are keys buffered]
    bool key_pressed = false;

    // return value
    return key_pressed;
}

                         
Key Console::read()
{
    //
    // Read key
    // --------
    //
    // This function reads a key press from the key buffer. If there
    // is no key presses stored in the key buffer then Console::read
    // blocks until a key is pressed.
    //
    // The operation of this function is similar to the 'getch'
    // function that PC programmers are familiar with under DOS and
    // Win32 console applications.
    // 
    // The return value of this function is a "Key" object. It is the
    // responsibility of the implementer to translate from the native
    // platform keyboard input to the format that PTC uses. See the
    // "Key" object for more information.
    //
    // The behavour of this function depends on a key buffering system.
    // If the native platform does not provide such a system then it
    // is necessary to implement one yourself. For example, under Win32
    // key presses are sent as WM_KEYDOWN messages which have to be
    // buffered manually.
    //
    // The Console::read function is designed for basic keyboard input
    // only. Because of this it does provide an indication when there
    // are multiple keys pressed, and cannot provide asynchronous key
    // press checking (ie. like the GetAsyncKeyState api under win32).
    // 
    // Such functionality is out of the scope of PTC and will need to
    // be implemented as platform dependent code by the end user, or
    // more ideally, as a portable input library supporting PTC.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();

    // [platform dependent code to read a key from the key buffer]

    // [platform dependent code to translate keyboard input to PTC "Key" code]
    int code = 0;

    // [platform dependent code to determine what modifier keys are pressed]
    bool alt = false;
    bool shift = false;
    bool control = false;

    // return key object
    return Key(code,alt,shift,control);
}




void Console::copy(BaseSurface &surface)
{
    //
    // Copy console to surface
    // -----------------------
    //
    // This function copies the contents of the console to the surface 
    // specified by the parameter 'surface'.
    //
    // If the width and height of the console does not match the width and 
    // height of the surface then the contents of the console will be 
    // stretched to fix the destination surface. If stretching is not 
    // possible, an exception should be thrown to indicate failure.
    //
    // If the pixel formats of the console and surface are not identical
    // then the console pixels will be converted to the destination pixels 
    // during the copy if possible. If a conversion is not possible an 
    // exception will be thrown to indicate failure.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();
    
    //
    // a) unlockable console
    //

    // [platform dependent code to copy console to surface]

    //
    // b) lockable console
    //

    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console to surface",error);
    }
}


void Console::copy(BaseSurface &surface,const Area &source,const Area &destination)
{
    //
    // Copy console area to surface area
    // ---------------------------------
    //
    // This function copies the area 'source' of the console to the
    // area 'destination' of the surface specified by the parameter 
    // 'surface'.
    //
    // If the width and height of the source and destination rectangles 
    // do not match then the source console pixels should be stretched 
    // to fit the destination surface area. If this is not possible an
    // exception should be thrown to indicate failure.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();
    
    //
    // a) unlockable console
    //

    // [platform dependent code to copy console area to surface area]

    //
    // b) lockable console
    //

    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette(),source,destination);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console area to surface area",error);
    }
}




void* Console::lock()
{
    //
    // Lock console
    // ------------
    //
    // This function locks the console pixels and returns a pointer to the 
    // console pixels with which the user can read and write directly to the 
    // console.
    //
    // Every successful lock (one that does not throw an exception) must be 
    // matched with a call to unlock. The time between lock and unlock 
    // should be kept to a minimum. Locking the console once and using 
    // that pointer for the rest of the program is not acceptable.
    //
    // Only one lock is allowed at a time on a surface. When the console is 
    // already locked and lock is called, an exception must be thrown to 
    // indicate this.
    //
    // If an attempt to close the console is made while the console is 
    // locked an exception will be thrown. You must first unlock the console
    // before closing it.
    //
    // Other console operations such as "update", "load", "save", "copy" and
    // "clear" may not be called while a console is locked, typically because
    // these functions depend on lock/unlock internally and cannot work when
    // the console is already locked because only one lock is allowed.
    //
    // While the console is locked it must remain static. No parameters such 
    // as width, height, pitch, format may change, and the pixels pointer 
    // must not change. While the console is not locked, all of the 
    // parameters may change under certain situations (ie. width and height 
    // may change if the user resizes a window etc.).
    //
    // Even the pointer to the console pixels may change while the console
    // is unlocked. This means that after Console::unlock is called, the 
    // pixels pointer returned by lock is invalid and cannot be used for
    // reading or writing.
    //
    // Having support for console locking is not manditory. On some 
    // platforms (usually windowed platforms) locking the display memory 
    // directly is not possible. These platforms typically load pixels to 
    // a bitmap area that does not allow direct read/write access to this 
    // bitmap (ie. GDI under Win32 with HBITMAPs).
    //
    // When locking is not supported by the platform just throw an error
    // as below to indicate failure. If locking is supported, remove the 
    // error throw in a) and implement the platform dependent code to lock 
    // the console pixels in b).
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    
    // fail if the console is already locked
    if (m_locked) throw Error("console is already locked");

    //
    // a) unlockable console
    //

    // console lock is not supported
    throw Error("console lock is not supported");

    //
    // b) lockable console
    //

    // [platform dependent code to get pointer to console pixels]

    // get pixels pointer
    void *pixels = 0;

    // we are locked
    m_locked = true;

    // return pixels
    return pixels;
}


void Console::unlock()
{
    //
    // Unlock console
    // --------------
    //
    // This function unlocks the console which was previously locked with 
    // a call to "Console::lock".
    //
    // See the lock function for more information on locking and unlocking 
    // consoles.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // fail if the console is not locked
    if (!m_locked) throw Error("console is not locked");

    //
    // a) unlockable console
    //

    // console unlock is not supported
    throw Error("console unlock is not supported");

    //
    // b) lockable console
    //

    // [platform dependent code to unlock console pixels]

    // we are unlocked
    m_locked = false;
}




void Console::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
    //
    // Load pixels to console
    // ----------------------
    //
    // This function loads pixels to the console.
    //
    // The parameters 'width', 'height', 'pitch' and 'format' describe 
    // the pixels to be loaded to the surface. If the width and height
    // do not match the size of the console, if possible the console 
    // should stretch the pixels when they are loaded. If stretching is
    // not supported an exception should be thrown.
    //
    // The 'palette' parameter is a reference to a Palette object that
    // internally is stored as a 256 entry array of 32bit color integers.
    // This palette is used to lookup from 8 bit indexed color pixels to
    // truecolor pixels. When loading 8 bit indexed pixels to a console
    // that is 8 bit indexed as well, the pixels are just copied without 
    // regard for the palette specified (ie. they use the console palette).
    //
    // If the console is lockable then you can use option b) below, and the
    // copy object will do all the work for you. However, if your console
    // implementation is not lockable (see Console::lock) then you will
    // have to load pixels to the console yourself using whatever method 
    // the platform supports.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();
    
    //
    // a) unlockable console
    //

    // [platform dependent code to load pixels to console]

    //
    // b) lockable console
    //

    // check clip area
    if (clip()==area())
    {
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(format,this->format());
    
            // set copy palettes
            m_copy.palette(palette,this->palette());

            // copy pixels to surface
            m_copy.copy(pixels,0,0,width,height,pitch,console_pixels,0,0,this->width(),this->height(),this->pitch());

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();

            // error message
            throw Error("failed to load pixels to console",error);
        }
    }
    else
    {
        // load explicit areas
        load(pixels,width,height,pitch,format,palette,Area(0,0,width,height),area());
    }
}


void Console::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
    //
    // Load pixels to console area
    // ---------------------------
    //
    // This function loads pixels to an area of the console.
    //
    // If the source an destination area width and height do not match
    // then the source pixels loaded should be stretched to fit the
    // destination console area. 
    //
    // See the previous Console::load function for more information on
    // loading pixels to the console.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();
    
    //
    // a) unlockable console
    //

    // [platform dependent code to load pixels to console area]

    //
    // b) lockable console
    //

    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,Area(0,0,width,height),clipped_source,destination,clip(),clipped_destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch,
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        throw Error("failed to load pixels to console area",error);
    }
}




void Console::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
    //
    // Save console pixels
    // -------------------
    //
    // This function saves the contents of the current console rendering
    // page to the pixel array pointed to by the 'pixels' parameter.
    //
    // The width and height parameters describe the dimensions of the 
    // destination pixel buffer. If the width and height do not match the 
    // width and height of the console then the console pixels should be 
    // stretched to fit if possible. If it is not possible to stretch the 
    // pixels then an exception should be thrown to indicate failure.
    //
    // The format parameter describes the pixel format of the destination 
    // pixels. If the pixel format does not match the format of the console 
    // then the console pixels will be converted to the destination pixels 
    // format if possible. If the conversion is not possible an exception 
    // should be thrown to indicate failure.
    //
    // The palette parameter passed describes the palette of the destination 
    // pixels. If the destination pixel format is not indexed color then 
    // this parameter is ignored.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();
    
    //
    // a) unlockable console
    //

    // [platform dependent code to save console pixels]

    //
    // b) lockable console
    //

    // check clip area
    if (clip()==area())
    {
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(this->format(),format);
    
            // set copy palettes
            m_copy.palette(this->palette(),palette);

            // copy console pixels to 'pixels' buffer
            m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,pitch);

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();
        
            // error message
            throw Error("failed to save console pixels",error);
        }
    }
    else
    {
        // save explicit areas
        save(pixels,width,height,pitch,format,palette,area(),Area(0,0,width,height));
    }
}


void Console::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
    //
    // Save console area pixels
    // ------------------------
    //
    // This function saves an area of the console to an area of the pixels 
    // buffer pointed to by the 'pixels' parameter.
    //
    // If the source area and the destination area do not match then the 
    // console pixels will be stretched to fit the destination rectangle
    // if possible. If it is not possible to stretch the pixels then an 
    // exception should be thrown to indicate failure.
    //
    // For information about the width, height, format and palette 
    // parameters see the previous save function.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();
    
    //
    // a) unlockable console
    //

    // [platform dependent code to save console pixel area]

    //
    // b) lockable console
    //

    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,clip(),clipped_source,destination,Area(0,0,width,height),clipped_destination);

        // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),this->pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch);
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to save console area pixels",error);
    }
}




void Console::clear()
{
    //
    // Clear console
    // -------------
    //
    // This function clears the entire console to black in direct color,
    // and to index zero in indexed color.
    //
    // If the console clip area is set to something different than the
    // whole console area then only this area will be cleared.
    //
    // This function is only valid if the console is open and unlocked.
    //
    
    // check console format
    if (format().direct())
    {
        // direct color
        clear(Color(0,0,0,0));
    }
    else
    {
        // indexed color
        clear(Color(0));
    }
}


void Console::clear(const Color &color)
{
    //
    // Clear console
    // -------------
    //
    // This function clears the entire console to the specified color.
    //
    // If the console clip area is set to something different than the
    // whole console area then only this area will be cleared.
    //
    // This function is only valid if the console is open and unlocked.
    // 

    // clear console
    clear(color,area());
}


void Console::clear(const Color &color,const Area &area)
{
    //
    // Clear console area
    // ------------------
    //
    // This function clears an area of the console to the specified color.
    //
    // This function may be implemented in one of two ways. Firstly,
    // if the console is lockable you can use code case b) and the clear
    // will be already implemented for you. However, if your console is 
    // not lockable you will need to provide specific code in case a) to 
    // clear the console.
    //
    // If the platform provides access to hardware accelerated clearing
    // operations then you should use the hardware clearing instead of the 
    // manual lock and clear case presented below.
    //
    // The clear area must be clipped to the console clip area.
    //
    // This function is only valid if the console is open and unlocked.
    //

    // debug check
    check_open();
    check_unlocked();
    
    //
    // a) unlockable console
    //

    // [platform dependent code to clear console area]

    //
    // b) lockable console
    //

    // lock console
    void *pixels = lock();

    try
    {
        // clip area
        Area clipped_area = Clipper::clip(area,this->clip());

        // request clear
        m_clear.request(format());

        // clear console area
        m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to clear console area",error);
    }
}




void Console::palette(const Palette &palette)
{
    //
    // Set console palette
    // -------------------
    //
    // This function sets the palette of the console.
    //
    // Even though the console palette exists only for indexed pixel 
    // formats, a call to Console::palette should fail silently when called 
    // while the display is in a direct color format. This is because it is
    // possible to request an indexed color format in open but retrieve a 
    // direct color format.
    //
    // The 'palette' parameter is a reference to a palette object that 
    // internally stores the palette as an array of 256 color values packed 
    // into a 32 bit unsigned integer:
    //
    // [a][r][g][b]
    //
    // To get a pointer to the palette data use the function Palette::data()
    //
    // Each component takes up 8 bits. 0 is is black and 255 is maximum
    // intensity.
    //
    // The color values can be extracted from the color integer as follows:
    //
    // char8 a = (char8) ( (color & 0xFF000000) >> 24 );
    // char8 r = (char8) ( (color & 0x00FF0000) >> 16 );
    // char8 g = (char8) ( (color & 0x0000FF00) >> 8  );
    // char8 b = (char8) ( (color & 0x000000FF) >> 0  );
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // fail silently if not indexed color
    if (!format().indexed()) return;
    
    // update cached palette
    m_palette = palette;

    // get palette data
    const void *data = palette.data();

    // [platform dependent code to set the console palette]
}


const Palette& Console::palette() const
{
    //
    // Get console palette
    // -------------------
    //
    // This function gets the palette of the console and returns a palette 
    // object containing the palette data.
    //
    // The implementation below depends on the 'm_palette' object being 
    // updated each time the console palette is set. As long as m_palette 
    // remains current there is no need for platform dependent code to get
    // the console palette.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // get palette
    return m_palette;
}




void Console::clip(const Area &area)
{
    //
    // Set clip area
    // -------------
    //
    // This function sets the clip area used to clip in the "load", "save"
    // "copy", "clear" and "update" functions.
    //
    // The clip area requested is always clipped against the console area,
    // this ensures that a clip area can never be set to a value that
    // would cause clipping to malfunction.
    // 
    // By default the clipping area is set to the entire console area.
    //
    // The "lock" and "unlock" functions totally ignore the clip area.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();

    // set new clip area
    m_clip = Clipper::clip(area,this->area());
}




int Console::width() const
{
    //
    // Get width of console
    // --------------------
    //
    // This function returns the width of the console in pixels.
    //
    // Currently this function is implemented by just returning a stored 
    // width value at Console::open. If the width of the console can possibly 
    // change (ie. user resizes the console window under a gui) then the 
    // width of the console can be updated to match the size of the window
    // if desired.
    //
    // If the dimensions of the console can change then it must do so in a 
    // controlled fashion. All of the console dimensions (ie. width, height, 
    // pages, pitch, and format) are guaranteed to remain constant in between
    // calls to 'Console::flush', 'Console::finish' and 'Console::update', 
    // and to remain constant while the console is locked.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // get width
    return m_width;
}


int Console::height() const
{
    // 
    // Get height of console
    // ---------------------
    //
    // This function returns the height of the console in pixels.
    //
    // See the notes in Console::width for more information about reporting
    // the dimensions of the console.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // get height
    return m_height;
}


int Console::pages() const
{
    // 
    // Get number of console pages
    // ---------------------------
    //
    // This function returns the number of console pages.
    //
    // For example, if the console is triple buffering the number of
    // pages will be 3 (one visible page plus two offscreen), double
    // buffering will return 2, and single buffering (one visible page
    // only) will return 1.
    //
    // If the platform does not support the concept of console pages
    // then you should return 0 here to indicate that the number of
    // console pages is unknown.
    //
    // See the notes in Console::width for more information about reporting
    // the dimensions of the console.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // get height
    return m_pages;
}


int Console::pitch() const
{
    //
    // Get pitch of console
    // --------------------
    //
    // This function returns the pitch of the console in bytes.
    //
    // 'pitch' is defined as the number of bytes to add to a char8* (byte) 
    // pointer to go from pixel [x][y] to pixel [x][y+1].
    //
    // The pitch can be any value, even negative!
    //
    // On most fullscreen platforms pitch is easily obtainable. Use
    // this value (usually something like "bytes per scanline" etc)
    // instead of just hardcoding pitch in Console.open to be equal
    // to width*format.bytes().
    //
    // On some platforms the pitch of the console can change while the 
    // console is open. In this case the pitch should be queried each time 
    // this function is called, instead of being stored in the m_pitch 
    // variable in Console::open.
    //
    // See the notes in Console::width for more information about reporting
    // the dimensions of the console.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // get pitch
    return m_pitch;
}


const Area& Console::area() const
{
    //
    // Get area of console
    // -------------------
    //
    // This function returns the area of the console.
    //
    // The area rectangle is always defined as:
    //
    //     Area area(0,0,width(),height());
    //
    // You must make sure that the console area is always up to date
    // with the width and height of the console. Whenever they change
    // it must be changed to match.
    //
    // See the notes in Console::width for more information about reporting
    // the dimensions of the console.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();

    // get area
    return m_area;
}


const Area& Console::clip() const
{
    //
    // Get clip area of console
    // ------------------------
    //
    // This function returns the clip of the console.
    //
    // By default the clip area is set to the entire console area,
    // however the user can set the clip area to any area within
    // the console.
    //
    // You must make sure that the clip area is always up to date
    // with the width and height of the console. Whenever they change
    // it must scaled accordingly. For example, if the clip area was
    // set an area covering top 20% of the console and the window is
    // resized then the clip area must be updated to be 20% of the new
    // window size.
    //
    // See the notes in Console::width for more information about reporting
    // the dimensions of the console.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();

    // get area
    return m_area;
}


const Format& Console::format() const
{
    //
    // Get format of console
    // ---------------------
    //
    // This function returns the pixel format of the console.
    //
    // The pixel format can change while a console is open in some cases. 
    // One example would be, windowed DirectX output under Win32, and the 
    // user changes the display settings to another mode with a different 
    // pixel format. However, in the vast majority of cases the format will 
    // remain constant while the console is open.
    //
    // See the notes in Console::width for more information about reporting
    // the dimensions of the console.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // get format
    return m_format;
}


const char* Console::name() const
{
    //
    // Get name of console
    // -------------------
    //
    // This function returns a string representing the name of the console.
    //
    // The console name is used to identify the platform. For example, a 
    // DirectX implementation of PTC would return "DirectX" as the console
    // name, a VGA implementation would return "VGA", and so on.
    //
    // Use the standard capitalization of the platform name, and if the
    // platform name is an acronym (ie. VGA, VBE, GGI) use all upper case.
    //
    // This function is valid whether the console is opened or closed.
    //

    // get name
    return "Porting Kit";
}


const char* Console::title() const
{
    //
    // Get console title
    // -----------------
    //
    // This function returns the console title string specified when the
    // console was last opened.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();
    
    // get title
    return m_title;
}


const char* Console::information() const
{
    //
    // Get console information
    // -----------------------
    //
    // This function returns a const pointer to a platform dependent
    // information string.
    //
    // You can use this string to output any useful information about
    // the console platform. This will prove to be extremely useful
    // in debugging and testing your OpenPTC implementation as users
    // can just run the "Info" example and dump all the console information
    // to a file and send it to you to assist with debugging.
    //
    // Typical things that could be put in the information string include:
    //
    //     - cpu identification data
    //     - operating system versions
    //     - driver versions/vendor strings
    //     - platform capability information (eg. capability bits etc).
    //
    // The exact formatting of the information string is up to the implementer.
    //
    // Because this function is const you cannot generate the information
    // string here. You should generate the string in Console::open instead.
    //
    // This function is only valid if the console is open.
    //

    // debug check
    check_open();

    // get information
    return m_information;
}




void Console::check_open() const
{
    //
    // Check if console is open
    // ------------------------
    //
    // This function is used in Console member functions that are only 
    // defined while the console is open.
    //
    // If __DEBUG__ is defined, then it throws an exception to indicate 
    // that the console operation is not valid while the console is closed.
    //
    // If __DEBUG__ is not defined, it performs no checking whether or not 
    // the console is open. This is for efficiency, nobody wants a check on 
    // every console.width/console.height in release build!
    //

    #ifdef __DEBUG__

        // check that console is open
        if (!m_open) throw Error("console is not open");

    #else

        // no checking in release build

    #endif
}


void Console::check_unlocked() const
{
    //
    // Check if console is unlocked
    // ----------------------------
    //
    // This function is used in Console member functions that are only 
    // defined while the console is unlocked.
    //
    // If __DEBUG__ is defined, then it throws an exception to indicate that 
    // the console operation is not valid while the console not unlocked.
    //
    // If __DEBUG__ is not defined, it performs no checking whether or not
    // the console is open. This is for efficiency, nobody wants a check on 
    // every console.width/console.height in release build!
    //

    #ifdef __DEBUG__

        // check that console is unlocked
        if (m_locked) throw Error("console is not unlocked");

    #else

        // no checking in release build

    #endif
}
