//
// Main interface routines of the HERMES library
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#include <string.h>
#include "Hermes.h"
#include "Base.h"
#include "Conv_GC.h"
#include "Conv_MMX.h"
#include "Conv_x86.h"

static BaseConverter *HermesConverters[3];  // Array of pointers to all conv.
static BaseConverter *HermesConv;           // The converter to use
static int HermesNumConverters=2;

void HermesInit()
{ int i;

//HermesConverters[0]=&Hermes_MMX;
  HermesConverters[0]=&Hermes_x86;
  HermesConverters[1]=&Hermes_Generic_C;
  
  for(i=0;i<HermesNumConverters;i++)
  { HermesConverters[i]->init();
  }  
}


void HermesClose()
{ int i;

  for(i=0;i<HermesNumConverters;i++)
  HermesConverters[i]->close();

}


char HermesRequest(HermesFormat* source, HermesFormat* dest, char stretch)
{ int i;

  HermesConv=0;

  for(i=0;i<HermesNumConverters;i++)
  { if(HermesConverters[i]->request(source,dest,stretch))
    { HermesConv=HermesConverters[i];
      break;
    }
  }

  if(HermesConv==0) return 0;
  else return 1;
}



void HermesPalette(int32 *palette)
{ HermesConv->palette(palette);
}


void HermesCopy(void *s_pixels,int s_x,int s_y,int s_width,int s_height,
	       int s_pitch,void *d_pixels,int d_x,int d_y,int d_width,
	       int d_height,int d_pitch)
{ HermesConv->copy(s_pixels,s_x,s_y,s_width,s_height,s_pitch,
		  d_pixels,d_x,d_y,d_width,d_height,d_pitch);
}


// Utility routines

void HermesFormatCopy(HermesFormat* source,HermesFormat *dest)
{ memcpy(dest,source,sizeof(HermesFormat));
}


char HermesFormatEquals(HermesFormat* source,HermesFormat *dest)
{ 
  if(source->bpp != dest->bpp) return 0;
  if(source->mask_r != dest->mask_r) return 0;
  if(source->mask_g != dest->mask_g) return 0;
  if(source->mask_b != dest->mask_b) return 0;
  if(source->indexed != dest->indexed) return 0;

  return 1;
}





// utility functions

void Hermes_Calculate_Generic_Info(int32 s_r,int32 s_g,int32 s_b,
  HermesFormat *format,Hermes_Generic_Info *info)
{ int r_right,g_right,b_right;
  
  // Calculate right shift amount for red. If it's <0, then set it to null
  // and calculate left shift amount
  r_right=s_r-Hermes_Topbit(format->mask_r);
  if(r_right<0)
  { info->r_left=-r_right;
    info->r_right=0;
  }
  else
  { info->r_left=0;
    info->r_right=r_right;
  }
   
  g_right=s_g-Hermes_Topbit(format->mask_g);
  if(g_right<0)
  { info->g_left=-g_right;
    info->g_right=0;
  }
  else
  { info->g_left=0;
    info->g_right=g_right;
  }

  b_right=s_b-Hermes_Topbit(format->mask_b);
  if(b_right<0)
  { info->b_left=-b_right;
    info->b_right=0;
  }
  else
  { info->b_left=0;
    info->b_right=b_right;
  }
}


int Hermes_Topbit(int32 mask)
{ int i;

  if(!mask) return 0;

  i=0;

  while(!(mask&1))
  { mask>>=1;
    i++;
  }

  while(mask&1)
  { mask>>=1;
    i++;
  }

  return i;
}
