//
// Generic C converter for the HERMES library
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Hermes.h"
#include "Base.h"
#include "Conv_GC.h"

static void (*Hermes_GC_Converter)(char *source,char *dest,unsigned int count);
static int32 *Hermes_GC_Lookup;
static Hermes_Generic_Info Hermes_GC_GI;

void Hermes_GC_init()
{ Hermes_Generic_C.m_source=(HermesFormat *)malloc(sizeof(HermesFormat));
  Hermes_Generic_C.m_dest=(HermesFormat *)malloc(sizeof(HermesFormat));
  Hermes_Generic_C.m_stretch=0;

  Hermes_GC_Lookup=(int32 *)malloc(256*4);
}


void Hermes_GC_close()
{ free(Hermes_Generic_C.m_source);
  free(Hermes_Generic_C.m_dest);
  free(Hermes_GC_Lookup);
}


char Hermes_GC_request(HermesFormat *source,HermesFormat *dest,char stretch)
{ char found=0;

  if(stretch) return 0;
  
  if(HermesFormatEquals(source,Hermes_Generic_C.m_source) &&
     HermesFormatEquals(dest,Hermes_Generic_C.m_dest) &&
     stretch==Hermes_Generic_C.m_stretch)
  return 1;


  // The monster block begins, but don't worry, normally this will only be
  // executed ONCE per program (at the first blit)
  // Apart from that, the most frequent modes will be found IMMEDIATELY,
  // because they form the first if.

  if(!source->indexed && !dest->indexed)
  switch(source->bpp)
  { case 32: 
             switch(dest->bpp)
	     { case 32: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
	                { Hermes_GC_Converter=Hermes_GC_32rgb888_32rgb888;
	                  found=1;
			  break;
			}
			else
	                { Hermes_GC_Converter=Hermes_GC_32rgb888_Generic32;
			  Hermes_Calculate_Generic_Info(24,16,8,
							dest,&Hermes_GC_GI);
			  found=1;
			}
			break;
			
	       case 24: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
		        { Hermes_GC_Converter=Hermes_GC_32rgb888_24rgb888;
		          found=1;
			  break;
			}
	                else
			if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
			{ Hermes_GC_Converter=Hermes_GC_32rgb888_24bgr888;
			  found=1;
			}
			break;

	       case 16: if(dest->mask_r==0xf800 && dest->mask_g==0x7e0 && 
			   dest->mask_b==0x1f)
	                { Hermes_GC_Converter=Hermes_GC_32rgb888_16rgb565;
			  found=1;
			  break;
			}
			else
			if(dest->mask_r==0x7c00 && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x1f)
			{ Hermes_GC_Converter=Hermes_GC_32rgb888_16rgb555;
			  found=1;
			  break;
			}
	                else
			{ Hermes_GC_Converter=Hermes_GC_32rgb888_Generic16;
			  Hermes_Calculate_Generic_Info(24,16,8,
							dest,&Hermes_GC_GI);
			  found=1;
			}
	                break;

	        case 8: if(dest->mask_r==0xe0 && dest->mask_g==0x1c &&
			   dest->mask_b==0x3)
		        { Hermes_GC_Converter=Hermes_GC_32rgb888_8rgb332;
			  found=1;
			}
		        break;
	     
	     }
             break;


    case 24:
             switch(dest->bpp)
	     { case 32: Hermes_GC_Converter=Hermes_GC_24rgb888_Generic32;
	                Hermes_Calculate_Generic_Info(24,16,8,
						      dest,&Hermes_GC_GI);
			found=1;
			break;

	       case 16: Hermes_GC_Converter=Hermes_GC_24rgb888_Generic16;
		 		        Hermes_Calculate_Generic_Info(24,16,8,
	       			        dest,&Hermes_GC_GI);
			found=1;
			break;

	        case 8: if(dest->mask_r==0xe0 && dest->mask_g==0x1c &&
			   dest->mask_b==0x3)
		        { Hermes_GC_Converter=Hermes_GC_24rgb888_8rgb332;
			  found=1;
			}
		        break;
	     }
	     break;

  }
  else
  if(source->indexed && !dest->indexed && source->bpp==8 )
  { switch(dest->bpp)
    { case 32: Hermes_GC_Converter=Hermes_GC_index8_32;
               found=1;
               break;

      case 24: Hermes_GC_Converter=Hermes_GC_index8_24;
	       found=1;
	       break;

      case 16: Hermes_GC_Converter=Hermes_GC_index8_16;
               found=1;
               break;
    }

    // Set up shift information for lookup creator
    if(found)
    Hermes_Calculate_Generic_Info(24,16,8,dest,&Hermes_GC_GI);
  }
  else
  if(source->indexed && source->bpp==8 && dest->indexed && dest->bpp==8)
  { Hermes_GC_Converter=Hermes_GC_index8_i8;
    found=1;
  }

  if(found)
  { HermesFormatCopy(source,Hermes_Generic_C.m_source);
    HermesFormatCopy(dest,Hermes_Generic_C.m_dest);
    Hermes_Generic_C.m_stretch=stretch;

    printf("Hermes C Converter\n"); 
    return 1;
  }

  // Couldn't find a converter
  return 0;
}


void Hermes_GC_palette(int32 *palette)
{ int i;
  int32 r,g,b;

  if(!Hermes_Generic_C.m_source->indexed)
  return;

  // If no left shifts are required, which is the most common case, we 
  // use a faster loop
  if(!Hermes_GC_GI.r_left && !Hermes_GC_GI.g_left && !Hermes_GC_GI.b_left)
  for(i=0;i<256;i++)
  { r=(palette[i]>>Hermes_GC_GI.r_right)&Hermes_Generic_C.m_dest->mask_r;
    g=(palette[i]>>Hermes_GC_GI.g_right)&Hermes_Generic_C.m_dest->mask_g;
    b=(palette[i]>>Hermes_GC_GI.b_right)&Hermes_Generic_C.m_dest->mask_b;
   
    Hermes_GC_Lookup[i]=r|g|b;
  }   
  else
  for(i=0;i<256;i++)
  { r=((palette[i]>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((palette[i]>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((palette[i]>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
   
    Hermes_GC_Lookup[i]=r|g|b;
  }  
}


void Hermes_GC_copy(void *s_pixels,int s_x,int s_y,int s_width,int s_height,
		   int s_pitch,void *d_pixels,int d_x,int d_y,int d_width,
		   int d_height,int d_pitch)
{ char *sp,*dp;
  int i;

  sp=(char *)s_pixels+s_y*s_pitch+s_x*Hermes_Generic_C.m_source->bpp/8;
  dp=(char *)d_pixels+d_y*d_pitch+d_x*Hermes_Generic_C.m_dest->bpp/8;

  for(i=0;i<s_height;i++)
  { Hermes_GC_Converter(sp,dp,s_width);

    sp+=s_pitch;
    dp+=d_pitch;
  }  
}



// ------------------------- Converters ------------------------------------


#define READ32(adr) *((int32 *)adr)
#define WRITE32(adr,s) *((int32 *)adr)=s

#define WRITE16(adr,s) *((short16 *)adr)=s


// FROM 32 BIT RGB ROUTINES

void Hermes_GC_32rgb888_32rgb888(char *source,char *dest,unsigned int count)
{
  while(count--)
  { WRITE32(dest,READ32(source));
    dest+=4;
    source+=4;
  }
}

void Hermes_GC_32rgb888_24rgb888(char *source,char *dest,unsigned int count)
{ int32 s_pixel;
  char8 *s_point=(char8 *)&s_pixel;

  while(count--)
  { s_pixel=READ32(source);

    #if defined __LITTLE_ENDIAN__
    
    *((char8 *)dest+0)=*(s_point+0);
    *((char8 *)dest+1)=*(s_point+1);
    *((char8 *)dest+2)=*(s_point+2);

    #elif defined __BIG_ENDIAN__
 
    *((char8 *)dest+0)=*(s_point+3);
    *((char8 *)dest+1)=*(s_point+2);
    *((char8 *)dest+2)=*(s_point+1);
    
    #else
       #error Processor endian-ness not defined!
    #endif
  
    source+=4;
    dest+=3;
  }
}


void Hermes_GC_32rgb888_24bgr888(char *source,char *dest,unsigned int count)
{ int32 s_pixel;
  char8 *s_point=(char8 *)&s_pixel;

  while(count--)
  { s_pixel=READ32(source);

    #if defined __LITTLE_ENDIAN__
    
    *((char8 *)dest+2)=*(s_point+0);
    *((char8 *)dest+1)=*(s_point+1);
    *((char8 *)dest+0)=*(s_point+2);

    #elif defined __BIG_ENDIAN__
 
    *((char8 *)dest+2)=*(s_point+3);
    *((char8 *)dest+1)=*(s_point+2);
    *((char8 *)dest+0)=*(s_point+1);
    
    #else
       #error Processor endian-ness not defined!
    #endif
  
    source+=4;
    dest+=3;
  }
}


void Hermes_GC_32rgb888_16rgb565(char *source,char *dest,unsigned int count)
{ int i,r,g,b;
  int32 s_pixel,d_pixelblock;
  short16 d_pixel;
  
  for(i=0;i<count>>1;i++,source+=8, dest+=4)
  { s_pixel=READ32(source);

    d_pixelblock=((s_pixel>>8)&0xf800)|
                 ((s_pixel>>5)&0x7e0)|
                 ((s_pixel>>3)&0x1f);

    s_pixel=READ32(source+1);

    d_pixelblock|=(((s_pixel>>8)&0xf800)|
                  ((s_pixel>>5)&0x7e0)|
                  ((s_pixel>>3)&0x1f))<<16;
   
    WRITE32(dest,d_pixelblock);
  }

  if(count&1)
  { s_pixel=READ32(source);

    r=(s_pixel>>8)&0xf800;
    g=(s_pixel>>5)&0x7e0;
    b=(s_pixel>>3)&0x1f;

    d_pixel=(short16)(r|g|b);

    WRITE16(dest,d_pixel);
  }
}



void Hermes_GC_32rgb888_16rgb555(char *source,char *dest,unsigned int count)
{ int r,g,b;
  int32 s_pixel,d_pixelblock;
  short16 d_pixel;
  unsigned int i;

  for(i=0;i<count>>1;i++,source+=8,dest+=4)
  { s_pixel=READ32(source);

    d_pixelblock=((s_pixel>>9)&0x7c00)|
                 ((s_pixel>>6)&0x3e0)|
                 ((s_pixel>>3)&0x1f);

    s_pixel=READ32(source+1);

    d_pixelblock|=(((s_pixel>>9)&0x7c00)|
                  ((s_pixel>>6)&0x3e0)|
                  ((s_pixel>>3)&0x1f))<<16;
   
    WRITE32(dest,d_pixelblock);
  }

  if(count&1)
  { s_pixel=READ32(source);

    r=(s_pixel>>9)&0xfc00;
    g=(s_pixel>>6)&0x7e0;
    b=(s_pixel>>3)&0x1f;

    d_pixel=(short16)(r|g|b);

    WRITE16(dest,d_pixel);
  }
}


void Hermes_GC_32rgb888_8rgb332(char *source,char *dest,unsigned int count)
{ int i;
  int32 s_pixel;
  char8 d_pixel;

  for(i=0;i<count;i++,source+=4,dest++)
  { s_pixel=READ32(source);

    d_pixel=((s_pixel>>16)&0xe0)|
            ((s_pixel>>11)&0x1c)|
            ((s_pixel>>6)&0x3);

    *((char8 *)dest)=d_pixel;
  }
}

// GENERIC (slow) routines from 32 bit to *. Note that these routines will
// hopefully NEVER be called as all cases should be catered for by special
// routines
void Hermes_GC_32rgb888_Generic32(char *source,char *dest,unsigned int count)
{ int i;
  int32 s_pixel,r,g,b;

  for(i=0;i<count;i++, source+=4, dest+=4)
  { s_pixel=READ32(source);

    r=((s_pixel>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((s_pixel>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((s_pixel>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    
    WRITE32(dest,(r|g|b));
    
  }
}


void Hermes_GC_32rgb888_Generic16(char *source,char *dest,unsigned int count)
{ int i;
  int32 s_pixel,r,g,b;
  int32 d_block;

  // This is a slower routine, but we will still write dwords, not words

  for(i=0;i<count>>1;i++,source+=8, dest+=4)
  { s_pixel=READ32(source);
  
    r=((s_pixel>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((s_pixel>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((s_pixel>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    d_block=(int32)(r|g|b);

    s_pixel=READ32(source+1);

    r=((s_pixel>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((s_pixel>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((s_pixel>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    d_block=d_block|((int32)(r|g|b)<<16);

    WRITE32(dest,d_block);
  }

  if(count&1)
  { s_pixel=READ32(source);
  
    r=((s_pixel>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((s_pixel>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((s_pixel>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;

    WRITE16(dest,(short16)(r|g|b));
  }
}




// FROM 24 BIT ROUTINES

void Hermes_GC_24rgb888_Generic32(char *source,char *dest,unsigned int count)
{ int32 r,g,b;

  while(count--)
  { 
    #if defined __LITTLE_ENDIAN__

    r=(int32)*(source+3)<<16;
    g=(int32)*(source+2)<<8;
    b=(int32)*(source+1);

    #elif defined __BIG_ENDIAN__

    r=(int32)*(source+0)<<16;
    g=(int32)*(source+1)<<8;
    b=(int32)*(source+2);

    #endif

    r=((r>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((g>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((b>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;

    WRITE32(dest,(r|g|b));

    source+=3;
    dest+=4;
  }
}


// optimise me
void Hermes_GC_24rgb888_Generic16(char *source,char *dest,unsigned int count)
{ int32 r,g,b;
  int i;

  for(i=0;i<count;i++,source+=3,dest+=2)
  { 
    #if defined __LITTLE_ENDIAN__

    r=(int32)*(source+3)<<16;
    g=(int32)*(source+2)<<8;
    b=(int32)*(source+1);

    #elif defined __BIG_ENDIAN__

    r=(int32)*(source+0)<<16;
    g=(int32)*(source+1)<<8;
    b=(int32)*(source+2);

    #endif
    
    r=((r>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((g>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((b>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    
    WRITE16(dest,(short16)(r|g|b));
  }
}


// optimise me
void Hermes_GC_24rgb888_8rgb332(char *source,char *dest,unsigned int count)
{ int32 r,g,b;
  int i;

  for(i=0;i<count;i++,source+=3,dest++)
  { 
    #if defined __LITTLE_ENDIAN__

    r=((int32)*(source+3))&0xe0;
    g=(((int32)*(source+2))>>3)&0x1c;
    b=(((int32)*(source+1))>>6)&0x3;

    #elif defined __BIG_ENDIAN__

    r=((int32)*(source+0))&0xe0;
    g=(((int32)*(source+1))>>3)&0x1c;
    b=(((int32)*(source+2))>>6)&0x3;

    #endif

    *((char8 *)dest)=(char8)(r|g|b);
  }
}




// FROM INDEXED 8 BIT ROUTINES

void Hermes_GC_index8_32(char *source,char *dest,unsigned int count)
{ int i;
  char8 s_pixel;
  int32 d_pixel;
  
  for(i=0;i<count;i++,source++,dest+=4)
  { s_pixel=(char8)*(source);

    d_pixel=Hermes_GC_Lookup[s_pixel];
    WRITE32(dest,d_pixel);
  }
}


void Hermes_GC_index8_24(char *source,char *dest,unsigned int count)
{ int i;
  char8 s_pixel;
  int32 d_pixel;
  char8 *d_ptr=(char8 *)&d_pixel;
  
  for(i=0;i<count;i++,source++,dest+=3)
  { s_pixel=(char8)*(source);

    d_pixel=Hermes_GC_Lookup[s_pixel];

    #if defined __LITTLE_ENDIAN__

    *((char8 *)dest+0)=*(d_ptr+0);
    *((char8 *)dest+1)=*(d_ptr+1);
    *((char8 *)dest+2)=*(d_ptr+2);
    
    #elif defined __BIG_ENDIAN__

    *((char8 *)dest+0)=*(d_ptr+3);
    *((char8 *)dest+1)=*(d_ptr+2);
    *((char8 *)dest+2)=*(d_ptr+1); 

    #endif

    WRITE32(dest,d_pixel);
  }
}



void Hermes_GC_index8_16(char *source,char *dest,unsigned int count)
{ int i;
  char8 s_pixel;
  int32 d_pixel;
  
  if(((int)dest&0x3)!=0)
  { s_pixel=(char8)*(source);
    WRITE16(dest,Hermes_GC_Lookup[s_pixel]);

    source++; 
    dest+=2;
    count--;
  }

  for(i=0;i<count>>1;i++,source+=2,dest+=4)
  { 
    d_pixel=Hermes_GC_Lookup[(char8)*(source)];
    d_pixel|=Hermes_GC_Lookup[(char8)*(source+1)]<<16;

    WRITE32(dest,d_pixel);
  }

  if(count&1)
  { s_pixel=(char8)*(source);
   
    WRITE16(dest,Hermes_GC_Lookup[s_pixel]);
  }
}


void Hermes_GC_index8_i8(char *source,char *dest,unsigned int count)
{ int i;
  int32 s_block;
  char8 s_pixel;
  
  for(i=0;i<count>>2;i++,source+=4,dest+=4)
  { s_block=READ32(source);

    WRITE32(dest,s_block);
  }

  for(i=0;i<(count&3);i++,source++,dest++)
  { s_pixel=*((char8 *)source);

    *((char8 *)dest)=s_pixel;
  }
}


