//
// X11 windowed mode driver class for OpenPTC 1.0 C++ API
// Copyright (c) 1998/99 Christian Nentwich (brn@eleet.mcb.at)
// The OpenPTC 1.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING contained in the distribution for
// licensing conditions
//

#include <X11/Xlib.h>
#include <stdlib.h>
#include "ptconfig.h"
#include "../Core/Error.h"
#include "../Core/Clipper.h"
#include "Console.h"
#include "Display.h"
#include "WindowDisplay.h"
#include "Image.h"





X11WindowDisplay::~X11WindowDisplay()
{ close();
}


X11WindowDisplay::X11WindowDisplay()
{ m_primary=0;
  m_window=0;
  m_colours=0;
  m_id=PTC_WINDOW;
  m_keypressed=false;
}


void X11WindowDisplay::open(const char title[],int width,int height,
  const Format& format,Display *disp,int screen) 
{ 
  m_disp=disp;
  m_screen=DefaultScreen(disp);
  m_height=height;
  m_width=width;
  m_destx=0;
  m_desty=0;

  // Check if we have that colour depth available.. Easy as there is no 
  // format conversion yet

  m_format=getFormat(format);

  // Create a window

  m_window=XCreateSimpleWindow(m_disp,DefaultRootWindow(m_disp),0,0,width,
			       height,0,0,0);
  

  // Register the delete atom
  
  m_atom_close=XInternAtom(m_disp,"WM_DELETE_WINDOW",False);
  XSetWMProtocols(m_disp,m_window,&m_atom_close,1);



  // Get graphics context

  XGCValues xgcv;
  
  xgcv.graphics_exposures=false;
  m_gc=XCreateGC(m_disp,m_window,GCGraphicsExposures,&xgcv);
  if(!m_gc)
  throw Error("can't create graphics context");



  // Set window title

  XTextProperty textprop;
  char *title_ptr[]={const_cast<char*>(title)};

  XStringListToTextProperty(title_ptr,1,&textprop);
  XSetWMName(m_disp,m_window,&textprop);
  XFlush(m_disp);
  XFree(textprop.value);


  // Map the window and wait for success

  XSelectInput(m_disp,m_window,StructureNotifyMask);

  XMapRaised(m_disp,m_window);

  for(;;)
  { XEvent e;

    XNextEvent(disp,&e);
    if(e.type==MapNotify) break;
  }


  // Get keyboard input and sync

  XSelectInput(m_disp,m_window,KeyPressMask|KeyReleaseMask|
	                       StructureNotifyMask|
	                       ButtonPressMask|ButtonReleaseMask|
	                       PointerMotionMask);
  XSync(m_disp,0);

 
  // Create XImage using factory method

  m_primary=createImage(m_disp,m_screen,m_width,m_height,m_format);

  bool found=false;                                // Stupid loop. The key
  do                                               // events were causing
  { XEvent e;                                      // problems.. 
     
    found=XCheckMaskEvent(m_disp,KeyPressMask|KeyReleaseMask,&e);
  } while(found);

  

  // Turn on backing storage for window (automatic redrawing)
  
  XSetWindowAttributes attr;

  attr.backing_store=Always;

  XChangeWindowAttributes(m_disp,m_window,CWBackingStore,&attr);

  // Set clipping area

  m_clip=Area(0,0,m_width,m_height);
  
  // Installs the right colour map for 8 bit modes

  createColormap();


#ifdef HAVE_PTHREADS
  if (pthread_create(&m_thread,NULL,X11WindowDisplay::eventHandler,this)) 
  throw Error("Cannot create event handling thread");
#endif
}



void X11WindowDisplay::open(Display *disp,int screen,Window w,
  const Format& format)
{ m_disp=disp;
  m_screen=screen;
  m_window=w;
  m_destx=0;
  m_desty=0;

  // Get the window's size

  { Window wtmp;
    unsigned int i;
    int j;

    XGetGeometry(m_disp,m_window,&wtmp,&j,&j,&m_width,&m_height,
	         &i,&i);
  }

  m_format=getFormat(format); 

  // Create the graphics context

  XGCValues xgcv;
  xgcv.graphics_exposures=false;
  m_gc=XCreateGC(m_disp,m_window,GCGraphicsExposures,&xgcv);
  if(!m_gc)
  throw Error("can't create graphics context");

  // Create XImage using factory method

  m_primary=createImage(m_disp,m_screen,m_width,m_height,m_format);

  // Set clipping area

  m_clip=Area(0,0,m_width,m_height);

  createColormap();
}




void X11WindowDisplay::open(Display *disp,int screen,Window window,
  const Format& format,int x,int y,int w,int h)
{ m_disp=disp;
  m_screen=screen;
  m_window=window;
  m_destx=x;
  m_desty=y;
  m_width=w;
  m_height=h;

  m_format=getFormat(format); 

  // Create the graphics context

  XGCValues xgcv;
  xgcv.graphics_exposures=false;
  m_gc=XCreateGC(m_disp,m_window,GCGraphicsExposures,&xgcv);
  if(!m_gc)
  throw Error("can't create graphics context");

  // Create XImage using factory method

  m_primary=createImage(m_disp,m_screen,m_width,m_height,m_format);

  // Set clipping area

  m_clip=Area(0,0,m_width,m_height);

  createColormap();
}


#include <iostream>

void X11WindowDisplay::close()
{ 

#ifdef HAVE_PTHREADS

  //  pthread_cancel(m_thread);
  //pthread_join(m_thread,NULL);
  //cout << "hi" << flush;
#endif
  

  if (m_cmap)
  { XFreeColormap(m_disp,m_cmap);
    m_cmap=0;
  }


  // Destroy XImage and buffer
  
  if(m_primary)
  { delete m_primary;
    m_primary=0;
  }

  if(m_colours)
  { delete [] m_colours;
    m_colours=0;
  }

  // Hide and destroy window

  if(m_window && !(m_flags&PTC_X11_LEAVE_WINDOW))
  { XUnmapWindow(m_disp,m_window);
    XSync(m_disp,0);

    XDestroyWindow(m_disp,m_window);
  }
}



void X11WindowDisplay::createColormap()
{
  // If we are in indexed mode, create the colour map 
  if(m_format.bits()==8)
  { 
    m_colours=new XColor[256];
    if(!m_colours) throw Error("Cannot allocated colour map cells");

    m_cmap=XCreateColormap(m_disp,RootWindow(m_disp,m_screen),
		           DefaultVisual(m_disp,m_screen),AllocAll);
    if(!m_cmap) throw Error("Cannot create colour map");

    XInstallColormap(m_disp,m_cmap);
    XSetWindowColormap(m_disp,m_window,m_cmap);
  }
  else m_cmap=0;


  // Set 332 palette, for now
  if(m_format.bits()==8 && m_format.direct())
  { 
    // Taken from PTC 0.72, i hope it's fine
    for (int i=0;i<256;i++)
    {
      float r = (float)((i&0xE0)>>5) * (float)255.0 / (float)7.0;
      float g = (float)((i&0x1C)>>2) * (float)255.0 / (float)7.0;
      float b = (float) (i&0x03)     * (float)255.0 / (float)3.0;
	
      m_colours[i].pixel=i;
	
      m_colours[i].red=((int)r)<<8;
      m_colours[i].green=((int)g)<<8;
      m_colours[i].blue=((int)b)<<8;

      m_colours[i].flags=DoRed|DoGreen|DoBlue;
    }
   
    XStoreColors(m_disp,m_cmap,m_colours,256);
  }
}


void X11WindowDisplay::update() 
{ m_primary->put(m_window,m_gc,m_destx,m_desty);

  // If we don't have pthreads, see if the close icon has been pressed
#ifndef HAVE_PTHREADS
  XEvent e;
  
  if(XCheckTypedEvent(m_disp,ClientMessage,&e))
  { if(e.xclient.format==32 && (Atom)e.xclient.data.l[0]==m_atom_close)
    exit(0);  
  }

#endif

}


void X11WindowDisplay::update(const Area &area) 
{ 
  Area updatearea=Clipper::clip(Area(0,0,m_width,m_height),area);
  m_primary->put(m_window,m_gc,updatearea.left(),updatearea.top(),
		 m_destx+updatearea.left(),m_desty+updatearea.top(),
		 updatearea.width(),updatearea.height());

  // If we don't have pthreads, see if the close icon has been pressed
#ifndef HAVE_PTHREADS
  XEvent e;
  
  if(XCheckTypedEvent(m_disp,ClientMessage,&e))
  { if(e.xclient.format==32 && (Atom)e.xclient.data.l[0]==m_atom_close)
    exit(0);  
  }
#endif

}

 
void* X11WindowDisplay::lock()
{ return m_primary->lock();
}


void X11WindowDisplay::unlock()
{ return;
}


bool X11WindowDisplay::key()
{ 
#ifdef HAVE_PTHREADS

  return m_keypressed;

#else
  XEvent e;
  
  if (XCheckMaskEvent(m_disp,KeyPressMask,&e))
  { 
    XPutBackEvent(m_disp,&e);             // Simulate "normal" kbhit behaviour
    return true;                          // i.e. leave the buffer intact
  }
  else
  return false;
#endif
}


Key X11WindowDisplay::read()
{ 
#ifdef HAVE_PTHREADS
  while (!m_keypressed);
  
  m_keypressed=false;
  return m_keylast;
  
#else

  XEvent e;

  XMaskEvent(m_disp,KeyPressMask,&e);         // Blocks and waits
 
  KeySym sym=XLookupKeysym(&e.xkey,0);

  switch (sym>>8) {
    case 0x00: return Key(m_normalkeys[sym&0xff]); 
    case 0xff: return Key(m_functionkeys[sym&0xff]);
      default: return Key();
  }


#endif
}



void X11WindowDisplay::palette(const Palette &palette)
{ const int32 *pal=palette.data();

  if(!m_format.indexed()) return;

  for(int i=0;i<256;i++)
  { m_colours[i].pixel=i;

    m_colours[i].red=((pal[i]>>16)&0xff)<<8;
    m_colours[i].green=((pal[i]>>8)&0xff)<<8;
    m_colours[i].blue=(pal[i]&0xff)<<8;

    m_colours[i].flags=DoRed|DoGreen|DoBlue;
  }
  XStoreColors(m_disp,m_cmap,m_colours,256);

}



int X11WindowDisplay::pitch() const
{ return m_primary->pitch();
}



X11Image* X11WindowDisplay::createImage(Display *display,int screen, int width,
  int height,Format &format)
{
  
#ifdef HAVE_X11_EXTENSIONS_XSHM_H
  if(XShmQueryExtension(display))      // Look for SHM extension
  { X11Image *tmp;

    try {
      tmp=new X11SHMImage(display,screen,width,height,format);
    }
    catch(Error e)                     // Fall back to normal ximage routines
    { tmp=new X11NormalImage(display,screen,width,height,format);
    }

    return tmp;
  }
  else
#endif
  return new X11NormalImage(display,screen,width,height,format);
}




#ifdef HAVE_PTHREADS

void* X11WindowDisplay::eventHandler(void *data)
{
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
  pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);

  // Create pointer to window display from data

  X11WindowDisplay *wd=(X11WindowDisplay *)data; 

  // Handler loop

  for(;;)
  {
    static XEvent e;

    XNextEvent(wd->m_disp,&e);

    pthread_testcancel();

    // Handle window exposure
  
    switch (e.type) {
    case ConfigureNotify: // Can handle resize here
                          break;

    case KeyPress: 
                   if (wd->m_keypressed) XPutBackEvent(wd->m_disp,&e);
                   else {
                     wd->m_keypressed=true;

		     KeySym sym=XLookupKeysym(&e.xkey,0);

		     switch (sym>>8) {
		       case 0x00: wd->m_keylast=Key(wd->m_normalkeys[sym&0xff]); 
		       case 0xff: wd->m_keylast=Key(wd->m_functionkeys[sym&0xff]);
		       default: wd->m_keylast=Key();
		     }
		   }
                   break;

    case ClientMessage: if(e.xclient.format==32 && 
			   e.xclient.data.l[0]==(long)wd->m_atom_close) {
                          exit(0);
                        }
                        break;
    }

  }

  pthread_exit(0);
}

#else

void* X11WindowDisplay::eventHandler(void *data) { return 0; }

#endif

