//
// X11 Common display driver class for OpenPTC 1.0 C++ API
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// The OpenPTC 1.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#include <X11/Xlib.h>
#include <X11/keysym.h>
#ifdef __SVR4
#include <X11/Sunkeysym.h>
#endif
#include <string.h>
#include "../Core/Error.h"
#include "../Core/Format.h"
#include "../Core/Surface.h"
#include "../Core/Clipper.h"
#include "Console.h"
#include "Display.h"
#include "Image.h"




X11Display::X11Display()
{ 
  m_flags=0;
  m_width=0;
  m_height=0;

  setKeyMapping();
}


X11Display::~X11Display()
{ 
  // Just close the display, everything else is done by the subclasses

  if (m_disp && !(m_flags&PTC_X11_LEAVE_DISPLAY))
  { XCloseDisplay(m_disp);
    m_disp=0;
  }

  if (m_normalkeys) delete [] m_normalkeys;
  if (m_functionkeys) delete [] m_functionkeys;

  m_normalkeys=0;
  m_functionkeys=0;
}


const Palette& X11Display::palette()
{ int32 *p=m_palette.lock();

  for(int i=0;i<256;i++)
  p[i]=(((int32)m_colours[i].red)<<8)|((int32)m_colours[i].green)|
       (((int32)m_colours[i].blue)>>8);

  m_palette.unlock();

  return m_palette;
}
 

int X11Display::width() const
{ return m_width;
}


int X11Display::height() const
{ return m_height;
}


const Format& X11Display::format() const
{ return m_format;
}



void X11Display::load(const void *pixels,int width,int height,int ppitch,
  const Format& format,const Palette &palette)
{ 
  if (clip()==area()) {
    try {
      m_copy.request(format,m_format);

      m_copy.palette(palette,m_palette);

      // Lock and copy (lock() is hidden somewhere in that line, watch out! :)
      m_copy.copy(pixels,0,0,width,height,width*format.bits()/8,
		  lock(),0,0,m_width,m_height,pitch());
      
      unlock();
    }
    catch(Error &e)
    { throw Error("Could not load console",e);
    }
  }
  else
  load(pixels,width,height,ppitch,format,palette,Area(0,0,width,height),area());
}


void X11Display::load(const void *pixels,int width,int height,int ppitch,
  const Format &format,const Palette &palette,
  const Area &source,const Area &destination)
{ try {

    // Clip source and destination areas
    Area clipped_source, clipped_destination;
    Clipper::clip(source,Area(0,0,width,height),clipped_source,destination,clip(),clipped_destination);

    // set copy formats
    m_copy.request(format,m_format);
    
    // set copy palette
    m_copy.palette(palette,m_palette);

    // copy pixels to primary
    m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),
	        clipped_source.width(),clipped_source.height(),
	        width*format.bits()/8,
	        lock(),clipped_destination.left(),clipped_destination.top(),
	        clipped_destination.width(),clipped_destination.height(),pitch());

    // unlock
    unlock();
  }
  catch(Error &e)
  { throw Error("Could not load console",e);
  }
}


void X11Display::save(void *pixels,int width,int height,int ppitch,
  const Format &format,const Palette &palette)
{
  if (clip()==area()) {
    
    try {
      // save primary pixels
      m_copy.request(m_format,format);

      m_copy.palette(m_palette,palette);

      // Lock and copy (lock() is hidden somewhere in that line, watch out! :)
      m_copy.copy(lock(),0,0,m_width,m_height,pitch(),
		  pixels,0,0,width,height,width*format.bits()/8);
	     
      unlock();
    }
    catch (Error &error) {
      // unlock
      unlock();

      // error message
      throw Error("failed to save console pixels",error);
    }
  }
  else
  save(pixels,width,height,ppitch,format,palette,area(),Area(0,0,width,height));
}


void X11Display::save(void *pixels,int width,int height,int ppitch,
  const Format &format,const Palette &palette,
  const Area &source,const Area &destination)
{
  // save primary pixels
  
  // Clip source and destination areas
  Area clipped_source,clipped_destination;
  Clipper::clip(source,clip(),clipped_source,destination,Area(0,0,width,height),clipped_destination);

  // set copy formats
  m_copy.request(m_format,format);
    
  // set copy palette
  m_copy.palette(m_palette,palette);

  // copy pixels to primary
  m_copy.copy(lock(),clipped_source.left(),clipped_source.top(),
	      clipped_source.width(),clipped_source.height(),pitch(),
	      pixels,clipped_destination.left(),clipped_destination.top(),
	      clipped_destination.width(),clipped_destination.height(),
	      width*format.bits()/8);

  // unlock
  unlock();
}



void X11Display::clear(const Color &color)
{
  // request clear
  m_clear.request(m_format);

  // clear surface
  m_clear.clear(lock(),0,0,m_width,m_height,pitch(),color);

  unlock();
}



void X11Display::clear(const Color &color,const Area &area)
{   // clip area
    Area clipped_area = Clipper::clip(Area(0,0,m_width,m_height),area);

    // request clear
    m_clear.request(m_format);

    // clear surface area
    m_clear.clear(lock(),clipped_area.left(),clipped_area.top(),
                  clipped_area.width(),clipped_area.height(),pitch(),color);
    unlock();
}





Format X11Display::getFormat(const Format &format)
{
  Format tmpformat;

  // Check if our screen has the same format available. I hate how X
  // keeps bits_per_pixel and depth different

  int tmp_depth=DefaultDepth(m_disp,m_screen);
  int numfound;

  XPixmapFormatValues *pfv=XListPixmapFormats(m_disp,&numfound);
 
  for(int i=0;i<numfound;i++)
  { if(pfv[i].depth==tmp_depth)
    { tmp_depth=pfv[i].bits_per_pixel;
      break;
    }
  }

  XFree(pfv);

  
  if(tmp_depth==8 && format.indexed())
  tmpformat=Format(8);
  else
  if(tmp_depth==8 && format.direct())
  tmpformat=Format(8,0xE0,0x1C,0x3);
  else
  tmpformat=Format(tmp_depth,
		   DefaultVisual(m_disp,m_screen)->red_mask,
		   DefaultVisual(m_disp,m_screen)->green_mask,
		   DefaultVisual(m_disp,m_screen)->blue_mask);

  return tmpformat;
}


const Area& X11Display::area() const
{ static Area disparea;
  disparea=Area(0,0,m_width,m_height);
  return disparea;
}
void X11Display::setKeyMapping()
{ 
  m_functionkeys=new int[256];
  m_normalkeys=new int[256];

  for (int i=0;i<256;i++)
  { m_functionkeys[i]=Key::UNDEFINED;
    m_normalkeys[i]=Key::UNDEFINED;
  }

#define ASSIGN_F(x,p) m_functionkeys[x&0xff]=p        // Assign to function key
#define ASSIGN_N(x,p) m_normalkeys[x&0xff]=p          // Assign to normal key


  // Assign function key indices from X definitions

  ASSIGN_F(XK_BackSpace,Key::BACKSPACE);
  ASSIGN_F(XK_Tab,Key::TAB);
  ASSIGN_F(XK_Clear,Key::CLEAR);
  ASSIGN_F(XK_Return,Key::ENTER);
  ASSIGN_F(XK_Pause,Key::PAUSE);
  ASSIGN_F(XK_Scroll_Lock,Key::SCROLLLOCK);
  ASSIGN_F(XK_Escape,Key::ESCAPE);
  ASSIGN_F(XK_Delete,Key::DELETE);

  ASSIGN_F(XK_Kanji,Key::KANJI);

  ASSIGN_F(XK_Home,Key::HOME);
  ASSIGN_F(XK_Left,Key::LEFT);
  ASSIGN_F(XK_Up,Key::UP);
  ASSIGN_F(XK_Right,Key::RIGHT);
  ASSIGN_F(XK_Down,Key::DOWN);
  ASSIGN_F(XK_Page_Up,Key::PAGEUP);
  ASSIGN_F(XK_Page_Down,Key::PAGEDOWN);
  ASSIGN_F(XK_End,Key::END);

  ASSIGN_F(XK_Print,Key::PRINTSCREEN);
  ASSIGN_F(XK_Insert,Key::INSERT);
  ASSIGN_F(XK_Num_Lock,Key::NUMLOCK); 

  ASSIGN_F(XK_KP_0,Key::NUMPAD0); 
  ASSIGN_F(XK_KP_1,Key::NUMPAD1); 
  ASSIGN_F(XK_KP_2,Key::NUMPAD2); 
  ASSIGN_F(XK_KP_3,Key::NUMPAD3); 
  ASSIGN_F(XK_KP_4,Key::NUMPAD4); 
  ASSIGN_F(XK_KP_5,Key::NUMPAD5); 
  ASSIGN_F(XK_KP_6,Key::NUMPAD6); 
  ASSIGN_F(XK_KP_7,Key::NUMPAD7); 
  ASSIGN_F(XK_KP_8,Key::NUMPAD8); 
  ASSIGN_F(XK_KP_9,Key::NUMPAD9); 

  ASSIGN_F(XK_F1,Key::F1); 
  ASSIGN_F(XK_F2,Key::F2); 
  ASSIGN_F(XK_F3,Key::F3); 
  ASSIGN_F(XK_F4,Key::F4); 
  ASSIGN_F(XK_F5,Key::F5); 
  ASSIGN_F(XK_F6,Key::F6); 
  ASSIGN_F(XK_F7,Key::F7); 
  ASSIGN_F(XK_F8,Key::F8); 
  ASSIGN_F(XK_F9,Key::F9); 
  ASSIGN_F(XK_F10,Key::F10); 
  ASSIGN_F(XK_F11,Key::F11); 
  ASSIGN_F(XK_F12,Key::F12); 

  ASSIGN_F(XK_Shift_L,Key::SHIFT);
  ASSIGN_F(XK_Shift_R,Key::SHIFT);
  ASSIGN_F(XK_Control_L,Key::CONTROL);   
  ASSIGN_F(XK_Control_R,Key::CONTROL);
  ASSIGN_F(XK_Caps_Lock,Key::CAPSLOCK);
  ASSIGN_F(XK_Meta_L,Key::META);  
  ASSIGN_F(XK_Meta_R,Key::META);
  ASSIGN_F(XK_Alt_L,Key::ALT);
  ASSIGN_F(XK_Alt_R,Key::ALT);



  // Assign normal key indices

  ASSIGN_N(XK_space,Key::SPACE);
  ASSIGN_N(XK_comma,Key::COMMA);
  ASSIGN_N(XK_minus,Key::SUBTRACT);
  ASSIGN_N(XK_period,Key::PERIOD);
  ASSIGN_N(XK_slash,Key::SLASH);
  ASSIGN_N(XK_0,Key::ZERO);
  ASSIGN_N(XK_1,Key::ONE);
  ASSIGN_N(XK_2,Key::TWO);
  ASSIGN_N(XK_3,Key::THREE);
  ASSIGN_N(XK_4,Key::FOUR);
  ASSIGN_N(XK_5,Key::FIVE);
  ASSIGN_N(XK_6,Key::SIX);
  ASSIGN_N(XK_7,Key::SEVEN);
  ASSIGN_N(XK_8,Key::EIGHT);
  ASSIGN_N(XK_9,Key::NINE);
  ASSIGN_N(XK_semicolon,Key::SEMICOLON);
  ASSIGN_N(XK_equal,Key::EQUALS);

  ASSIGN_N(XK_bracketleft,Key::OPENBRACKET);
  ASSIGN_N(XK_backslash,Key::BACKSLASH);
  ASSIGN_N(XK_bracketright,Key::CLOSEBRACKET);

  ASSIGN_N(XK_a,Key::A);
  ASSIGN_N(XK_b,Key::B); 
  ASSIGN_N(XK_c,Key::C); 
  ASSIGN_N(XK_d,Key::D);
  ASSIGN_N(XK_e,Key::E);
  ASSIGN_N(XK_f,Key::F);
  ASSIGN_N(XK_g,Key::G);
  ASSIGN_N(XK_h,Key::H);
  ASSIGN_N(XK_i,Key::I);
  ASSIGN_N(XK_j,Key::J);
  ASSIGN_N(XK_k,Key::K);
  ASSIGN_N(XK_l,Key::L);
  ASSIGN_N(XK_m,Key::M);
  ASSIGN_N(XK_n,Key::N);
  ASSIGN_N(XK_o,Key::O);
  ASSIGN_N(XK_p,Key::P);
  ASSIGN_N(XK_q,Key::Q);
  ASSIGN_N(XK_r,Key::R);
  ASSIGN_N(XK_s,Key::S);
  ASSIGN_N(XK_t,Key::T);
  ASSIGN_N(XK_u,Key::U);
  ASSIGN_N(XK_v,Key::V);
  ASSIGN_N(XK_w,Key::W);
  ASSIGN_N(XK_x,Key::X);
  ASSIGN_N(XK_y,Key::Y);
  ASSIGN_N(XK_z,Key::Z);

}



