//
// DirectX Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// include files
#include <stdio.h>
#include <string.h>
#include <fstream.h>
#include "Core/Key.h"
#include "Core/Area.h"
#include "Core/Color.h"
#include "Core/Error.h"
#include "Core/Config.h"
#include "Core/Clipper.h"
#include "Core/Console.h"
#include "DirectX/Hook.h"

// define defaults
#define DEFAULT_WIDTH 320;
#define DEFAULT_HEIGHT 200;
#define DEFAULT_FORMAT Format(32,0x00FF0000,0x0000FF00,0x000000FF);
#ifdef __DEBUG__
#define DEFAULT_OUTPUT WINDOWED
#else
#define DEFAULT_OUTPUT DEFAULT
#endif


PTCAPI DirectXConsole::DirectXConsole()
{
    // defaults
    m_open   = false;
    m_locked = false;

    // clear strings
    m_title[0] = 0;

    // clear objects
    m_hook = 0;
    m_window = 0;
    m_keyboard = 0;

    // default option data
    m_frequency = 0;
    m_default_width = DEFAULT_WIDTH;
    m_default_height = DEFAULT_HEIGHT;
    m_default_format = DEFAULT_FORMAT;
    m_center_window = false;
    m_synchronized_update = true;
    m_output_mode = DEFAULT_OUTPUT;
    m_window_mode = RESIZABLE;
    m_primary_mode_windowed = SECONDARY;
    m_primary_mode_fullscreen = DIRECT;
    m_nearest_mode = NEAREST_DEFAULT;
    m_cursor_mode = CURSOR_DEFAULT;

    // configure console
    configure("ptc.cfg");

    // setup display object
    m_display.setup(m_library.lpDD2());
}


PTCAPI DirectXConsole::~DirectXConsole()
{
    // close
    close();
}




void PTCAPI DirectXConsole::configure(const char file[])
{
    // open configuration file
    ifstream is(file,ios::in | ios::nocreate);
    
    // check stream
    if (!is) return;

    // read option strings
    while (!is.eof())
    {
        // option line
        char line[1024];

        // read line from stream
        is.getline(line,1024);

        // process option
        option(line);
    }
}




bool PTCAPI DirectXConsole::option(const char option[])
{
    // check for "default output" option
    if (stricmp(option,"default output")==0)
    {
        // default output mode
        m_output_mode = DEFAULT;

        // recognized
        return true;
    }

    // check for "windowed output" option
    if (stricmp(option,"windowed output")==0)
    {
        // windowed output mode
        m_output_mode = WINDOWED;

        // recognized
        return true;
    }

    // check for "fullscreen output" option
    if (stricmp(option,"fullscreen output")==0)
    {
        // fullscreen output mode
        m_output_mode = FULLSCREEN;
       
        // recognized
        return true;
    }

    // check for "default width" option
    if (strnicmp(option,"default width",13)==0)
    {
        // look for parameter
        if (option[13]!=0)
        {
            // we have a parameter
            m_default_width = atoi(&option[13]);
        }
        else
        {
            // no parameter
            m_default_width = DEFAULT_WIDTH;
        }

        // recognized
        return true;
    }

    // check for "default height" option
    if (strnicmp(option,"default height",14)==0)
    {                            
        // look for parameter
        if (option[14]!=0)
        {
            // we have a parameter
            m_default_height = atoi(&option[14]);
        }
        else
        {
            // no parameter
            m_default_height = DEFAULT_HEIGHT;
        }

        // recognized
        return true;
    }

    // check for "default bits" option
    if (strnicmp(option,"default bits",12)==0)
    {                            
        // look for parameter
        if (option[12]!=0)
        {
            // we have a parameter
            int bits = atoi(&option[12]);
            
            // setup format
            switch (bits)
            {
                case 8:  m_default_format = Format(8); break;
                case 16: m_default_format = Format(16,0xF800,0x07E0,0x001F); break;
                case 24: m_default_format = Format(24,0x00FF0000,0x0000FF00,0x000000FF); break;
                case 32: m_default_format = Format(32,0x00FF0000,0x0000FF00,0x000000FF); break;
                default: return false;
            }
        }
        else
        {
            // no parameter
            m_default_format = DEFAULT_FORMAT;
        }

        // recognized
        return true;
    }

    // check for "resizable window" option
    if (stricmp(option,"resizable window")==0)
    {
        // set window mode to resizable
        m_window_mode = RESIZABLE;

        // recognized
        return true;
    }

    // check for "fixed window" option
    if (stricmp(option,"fixed window")==0)
    {
        // set window mode to fixed
        m_window_mode = FIXED;

        // recognized
        return true;
    }

    // check for "windowed primary direct" option
    if (stricmp(option,"windowed primary direct")==0)
    {
        // set windowed primary mode to direct
        m_primary_mode_windowed = DIRECT;

        // recognized
        return true;
    }

    // check for "windowed primary secondary" option
    if (stricmp(option,"windowed primary secondary")==0)
    {
        // set windowed primary mode to secondary
        m_primary_mode_windowed = SECONDARY;

        // recognized
        return true;
    }

    // check for "fullscreen primary direct" option
    if (stricmp(option,"fullscreen primary direct")==0)
    {
        // set fullscreen primary mode to direct
        m_primary_mode_fullscreen = DIRECT;

        // recognized
        return true;
    }

    // check for "fullscreen primary secondary" option
    if (stricmp(option,"fullscreen primary secondary")==0)
    {
        // set fullscreen primary mode to secondary
        m_primary_mode_fullscreen = SECONDARY;

        // recognized
        return true;
    }

    // check for "center window" option
    if (stricmp(option,"center window")==0)
    {
        // set window centering on
        m_center_window = true;

        // recognized
        return true;
    }

    // check for "default window position" option
    if (stricmp(option,"default window position")==0)
    {
        // set window centering off
        m_center_window = false;

        // recognized
        return true;
    }

    // check for "synchronized update" option
    if (stricmp(option,"synchronized update")==0)
    {
        // set synchronized update to true
        m_synchronized_update = true;

        // recognized
        return true;
    }

    // check for "unsynchronized update" option
    if (stricmp(option,"unsynchronized update")==0)
    {
        // set synchronized update to false
        m_synchronized_update = false;

        // recognized
        return true;
    }

    // check for "default nearest" option
    if (stricmp(option,"default nearest")==0)
    {
        // set nearest mode to default
        m_nearest_mode = NEAREST_DEFAULT;

        // recognized
        return true;
    }

    // check for "center nearest" option
    if (stricmp(option,"center nearest")==0)
    {
        // set nearest mode to center
        m_nearest_mode = NEAREST_CENTERING;

        // recognized
        return true;
    }

    // check for "default stretch" option
    if (stricmp(option,"default stretch")==0)
    {
        // set nearest mode to default
        m_nearest_mode = NEAREST_STRETCHING;

        // recognized
        return true;
    }

    // check for "default cursor" option
    if (stricmp(option,"default cursor")==0)
    {
        // set cursor mode to default
        m_cursor_mode = CURSOR_DEFAULT;

        // recognized
        return true;
    }

    // check for "show cursor" option
    if (stricmp(option,"show cursor")==0)
    {
        // set cursor mode to show
        m_cursor_mode = CURSOR_SHOW;

        // recognized
        return true;
    }

    // check for "hide cursor" option
    if (stricmp(option,"hide cursor")==0)
    {
        // set cursor mode to hide
        m_cursor_mode = CURSOR_HIDE;

        // recognized
        return true;
    }

    // check for "frequency [n]" option
    if (strnicmp(option,"frequency",9)==0)
    {                            
        // look for parameter
        if (option[9]!=0)
        {
            // we have a parameter
            m_frequency = atoi(&option[9]);
        }
        else
        {
            // no parameter
            m_frequency = 0;
        }

        // recognized
        return true;
    }

    // pass the option to the copy object
    return m_copy.option(option);
}




const Mode* PTCAPI DirectXConsole::modes()
{
    // get display modes
    return m_display.modes();
}




void PTCAPI DirectXConsole::open(const char title[],int pages)
{
    // open console
    open(title,m_default_format,pages);
}


void PTCAPI DirectXConsole::open(const char title[],const Format &format,int pages)
{
    // open console
    open(title,m_default_width,m_default_height,m_default_format,pages);
}


void PTCAPI DirectXConsole::open(const char title[],int width,int height,const Format &format,int pages)
{
    // internal open nearest mode
    internal_open(title,0,Mode(width,height,format),pages,false);
}


void PTCAPI DirectXConsole::open(const char title[],const Mode &mode,int pages)
{
    // internal open exact mode
    internal_open(title,0,mode,pages,true);
}


void PTCAPI DirectXConsole::open(HWND window,int pages)
{
    // open console
    open(window,m_default_format,pages);
}


void PTCAPI DirectXConsole::open(HWND window,const Format &format,int pages)
{
    // open console
    open(window,m_default_width,m_default_height,m_default_format,pages);
}


void PTCAPI DirectXConsole::open(HWND window,int width,int height,const Format &format,int pages)
{
    // internal open nearest mode
    internal_open("",window,Mode(width,height,format),pages,false);
}


void PTCAPI DirectXConsole::open(HWND window,const Mode &mode,int pages)
{
    // internal open exact mode
    internal_open("",window,mode,pages,true);
}


void PTCAPI DirectXConsole::close()
{
    // check if open
    if (m_open)
    {
        // the console must be unlocked when closed
        if (m_locked) throw Error("console is still locked");

        // flush all key presses
        while (key()) read();
    }

    // close console
    internal_close();
}





void PTCAPI DirectXConsole::flush()
{
    // debug checks
    check_open();
    check_unlocked();

    // [platform dependent code to flush all console operations]

    // update window
    m_window->update();
}


void PTCAPI DirectXConsole::finish()
{
    // debug check
    check_open();
    check_unlocked();

    // [platform dependent code to finish all console operations]

    // update window
    m_window->update();
}


void PTCAPI DirectXConsole::update()
{
    // debug check
    check_open();
    check_unlocked();

    // update primary surface
    m_primary.update();

    // update window
    m_window->update();
}


void PTCAPI DirectXConsole::update(const Area &area)
{
    // update
    update();
}




bool PTCAPI DirectXConsole::key()
{
    // debug check
    check_open();
    
    // check for key press
    return m_keyboard->key();
}

                         
Key PTCAPI DirectXConsole::read()
{
    // debug check
    check_open();

    // read key press from keyboard
    return m_keyboard->read(*m_window);
}




void PTCAPI DirectXConsole::copy(BaseSurface &surface)
{
    // debug check
    check_open();
    check_unlocked();
    
    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console to surface",error);
    }
}


void PTCAPI DirectXConsole::copy(BaseSurface &surface,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();
    
    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette(),source,destination);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console area to surface area",error);
    }
}




void* PTCAPI DirectXConsole::lock()
{
    // debug check
    check_open();
    
    // fail if the console is already locked
    if (m_locked) throw Error("console is already locked");

    // lock primary surface
    void *pixels = m_primary.lock();

    // surface is locked
    m_locked = true;

    // return pixels
    return pixels;
}


void PTCAPI DirectXConsole::unlock()
{
    // debug check
    check_open();
    
    // fail if the console is not locked
    if (!m_locked) throw Error("console is not locked");

    // unlock primary surface
    m_primary.unlock();

    // we are unlocked
    m_locked = false;
}




void PTCAPI DirectXConsole::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
    // check clip area
    if (clip()==area())
    {
        // debug check
        check_open();
        check_unlocked();
    
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(format,this->format());
    
            // set copy palettes
            m_copy.palette(palette,this->palette());

            // copy pixels to surface
            m_copy.copy(pixels,0,0,width,height,pitch,console_pixels,0,0,this->width(),this->height(),this->pitch());

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();

            // error message
            throw Error("failed to load pixels to console",error);
        }
    }
    else
    {
        // load explicit areas
        load(pixels,width,height,pitch,format,palette,Area(0,0,width,height),area());
    }
}


void PTCAPI DirectXConsole::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();
    
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,Area(0,0,width,height),clipped_source,destination,clip(),clipped_destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch,
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        throw Error("failed to load pixels to console area",error);
    }
}




void PTCAPI DirectXConsole::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
    // debug check
    check_open();
    check_unlocked();
    
    // check clip area
    if (clip()==area())
    {
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(this->format(),format);
    
            // set copy palettes
            m_copy.palette(this->palette(),palette);

            // copy console pixels to 'pixels' buffer
            m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,pitch);

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();
        
            // error message
            throw Error("failed to save console pixels",error);
        }
    }
    else
    {
        // save explicit areas
        save(pixels,width,height,pitch,format,palette,area(),Area(0,0,width,height));
    }
}


void PTCAPI DirectXConsole::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();
    
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,clip(),clipped_source,destination,Area(0,0,width,height),clipped_destination);

        // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),this->pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch);
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to save console area pixels",error);
    }
}




void PTCAPI DirectXConsole::clear()
{
    // debug check
    check_open();
    check_unlocked();
    
    // check console format
    if (format().direct())
    {
        // direct color
        clear(Color(0,0,0,0));
    }
    else
    {
        // indexed color
        clear(Color(0));
    }
}


void PTCAPI DirectXConsole::clear(const Color &color)
{
    // debug check
    check_open();
    check_unlocked();

    // clear console
    clear(color,area());
}


void PTCAPI DirectXConsole::clear(const Color &color,const Area &area)
{
    // debug check
    check_open();
    check_unlocked();

    // clear primary surface
    m_primary.clear(color,area);
}




void PTCAPI DirectXConsole::palette(const Palette &palette)
{
    // debug check
    check_open();
    
    // set primary surface palette
    m_primary.palette(palette);
}


const Palette& PTCAPI DirectXConsole::palette() const
{
    // debug check
    check_open();
    
    // get primary surface palette
    return m_primary.palette();
}




void PTCAPI DirectXConsole::clip(const Area &area)
{
    // debug check
    check_open();

    // set clip area
    m_primary.clip(area);
}




int PTCAPI DirectXConsole::width() const
{
    // debug check
    check_open();
    
    // get primary width
    return m_primary.width();
}


int PTCAPI DirectXConsole::height() const
{
    // debug check
    check_open();
    
    // get primary height
    return m_primary.height();
}


int PTCAPI DirectXConsole::pages() const
{
    // debug check
    check_open();
    
    // get primary pages
    return m_primary.pages();
}


int PTCAPI DirectXConsole::pitch() const
{
    // debug check
    check_open();
    
    // get primary pitch
    return m_primary.pitch();
}


const Area& PTCAPI DirectXConsole::area() const
{
    // debug check
    check_open();
    
    // get primary area
    return m_primary.area();
}


const Area& PTCAPI DirectXConsole::clip() const
{
    // debug check
    check_open();
    
    // get primary clip
    return m_primary.clip();
}


const Format& PTCAPI DirectXConsole::format() const
{
    // debug check
    check_open();
    
    // get primary format
    return m_primary.format();
}


const char* PTCAPI DirectXConsole::name() const
{
    // get name
    return "DirectX";
}


const char* PTCAPI DirectXConsole::title() const
{
    // debug check
    check_open();
    
    // get title
    return m_title;
}


const char* PTCAPI DirectXConsole::information() const
{
    // debug check
    check_open();
    
    // get information
    return m_display.information();
}




HWND PTCAPI DirectXConsole::window() const
{
    // debug check
    check_open();
    
    // get window handle
    return m_window->handle();
}


LPDIRECTDRAW PTCAPI DirectXConsole::lpDD() const
{
    // debug check
    check_open();

    // get directdraw interface
    return m_library.lpDD();
}



LPDIRECTDRAW2 PTCAPI DirectXConsole::lpDD2() const
{
    // debug check
    check_open();

    // get directdraw 2 interface
    return m_library.lpDD2();
}



LPDIRECTDRAWSURFACE PTCAPI DirectXConsole::lpDDS() const
{
    // debug check
    check_open();

    // get directdraw surface interface
    return m_primary.lpDDS();
}


LPDIRECTDRAWSURFACE PTCAPI DirectXConsole::lpDDS_primary() const
{
    // debug check
    check_open();

    // get directdraw primary surface interface
    return m_primary.lpDDS_primary();
}


LPDIRECTDRAWSURFACE PTCAPI DirectXConsole::lpDDS_secondary() const
{
    // debug check
    check_open();

    // get directdraw secondary surface interface
    return m_primary.lpDDS_secondary();
}




void DirectXConsole::internal_open(const char title[],HWND window,const Mode &mode,int pages,bool exact)
{
    // check that the mode is valid
    if (!mode.valid()) throw Error("invalid mode");

    // get mode information
    int width = mode.width();
    int height = mode.height();
    Format format = mode.format();

    // setup before open
    internal_pre_open_setup(title,window);

    // check output mode
    switch (m_output_mode)
    {
        case DEFAULT:
        {
            try
            {
                // start internal fullscreen open
                internal_open_fullscreen_start(window,mode,exact);
                
                // internal open fullscreen mode
                internal_open_fullscreen(mode,exact);

                // finish internal fullscreen open
                internal_open_fullscreen_finish(mode,pages);
            }
            catch (Error&)
            {
                // internal reset
                internal_reset();

                // internal open windowed
                internal_open_windowed(window,mode,pages);
            }
        }
        break;

        case WINDOWED:
        {
            // internal open windowed
            internal_open_windowed(window,mode,pages);
        }
        break;

        case FULLSCREEN:
        {
            // start internal fullscreen open
            internal_open_fullscreen_start(window,mode,exact);
    
            // internal open fullscreen mode
            internal_open_fullscreen(mode,exact);

            // finish internal fullscreen open
            internal_open_fullscreen_finish(mode,pages);
        }
        break;
    }

    // setup after open
    internal_post_open_setup();
}


void DirectXConsole::internal_close()
{
    // clear flags
    m_open = false;

    // delete keyboard
    delete m_keyboard;

    // delete hook
    delete m_hook;

    // close primary
    m_primary.close();

    // close display
    m_display.close();

    // delete window
    delete m_window;

    // restore display
    m_display.restore();

    // clear pointers
    m_keyboard = 0;
    m_window = 0;
    m_hook = 0;
}




void DirectXConsole::internal_pre_open_setup(const char title[],HWND window)
{
    // close down
    internal_close();
    
    // check window
    if (!window)
    {
        // check length of console title
        if (strlen(title)+1<=sizeof(m_title))
        {
            // update console title
            strcpy(m_title,title);
        }
        else
        {
            // update console title truncated
            memcpy(m_title,title,sizeof(m_title));
            m_title[sizeof(m_title)-1] = 0;
        }
    }
    else
    {
        // get existing window title
        GetWindowText(window,m_title,sizeof m_title);
    }
}


void DirectXConsole::internal_open_fullscreen_start(HWND window,const Mode &mode,bool exact)
{
    // test display mode exists...
    m_display.test(mode,exact);

    // save display
    m_display.save();

    // cursor flag
    bool cursor = false;

    // handle cursor show mode
    if (m_cursor_mode==CURSOR_SHOW) cursor = true;

    // check window
    if (!window)
    {
        // create fullscreen window
        m_window = new Win32Window("PTC_DIRECTX_FULLSCREEN",m_title,WS_EX_TOPMOST,WS_POPUP|WS_SYSMENU|WS_VISIBLE,SW_NORMAL,0,0,GetSystemMetrics(SM_CXSCREEN),GetSystemMetrics(SM_CYSCREEN),cursor,false,false);
    }
    else
    {
        // use existing window
        m_window = new Win32Window(window,cursor);
    }
    
    // set cooperative level
    m_display.cooperative(m_window->handle(),true);
}


void DirectXConsole::internal_open_fullscreen(const Mode &mode,bool exact)
{
    // open display mode
    m_display.open(mode,exact,m_frequency);
}


void DirectXConsole::internal_open_fullscreen_finish(const Mode &mode,int pages)
{
    // determine if we need to create a secondary surface
    const bool secondary = m_primary_mode_fullscreen==SECONDARY || mode!=m_display.mode();

    // initialize primary surface
    m_primary.initialize(*m_window,m_library.lpDD2());

    try
    {
        // n buffered video memory
        m_primary.primary(pages,true,true);
    }
    catch (Error&)
    {
        try
        {
            // triple buffered video memory
            m_primary.primary(3,true,true);
        }
        catch (Error&)
        {
            try
            {
                // double buffered video memory
                m_primary.primary(2,true,true);
            }
            catch (Error&)
            {
                // check secondary
                if (!secondary)
                {
                    // double buffered video or system memory
                    m_primary.primary(2,false,true);
                }
                else
                {
                    // single buffered video or system memory
                    m_primary.primary(1,false,true);
                }
            }
        }
    }

    // check secondary
    if (secondary)
    {
        // create secondary surface
        m_primary.secondary(mode.width(),mode.height());
    }

    // check nearest mode
    if (m_nearest_mode==NEAREST_CENTERING) m_primary.centering(true);
    else if (m_nearest_mode==NEAREST_STRETCHING) m_primary.centering(false);

    // cursor flag
    bool cursor = false;

    // handle cursor show mode
    if (m_cursor_mode==CURSOR_SHOW) cursor = true;

    // create hook on window
    m_hook = new DirectXHook(*this,m_window->handle(),GetCurrentThreadId(),cursor,m_window->managed());
}


void DirectXConsole::internal_open_windowed(HWND window,const Mode &mode,int pages)
{
    // get mode width and height
    const int width  = mode.width();
    const int height = mode.height();

    // cursor flag
    bool cursor = true;

    // handle cursor hide mode
    if (m_cursor_mode==CURSOR_HIDE) cursor = false;

    // check window
    if (window)
    {
        // use existing window
        m_window = new Win32Window(window,cursor);
    }
    else
    {    
        // extended style
        int extended = 0;

        // make the window topmost if direct primary writing is setup
        if (m_primary_mode_windowed==DIRECT) extended = WS_EX_TOPMOST;

        // check window mode
        switch (m_window_mode)
        {
            case RESIZABLE:
            {
                // create resizable window
                m_window = new Win32Window("PTC_DIRECTX_WINDOWED_RESIZABLE",m_title,extended,WS_OVERLAPPEDWINDOW|WS_VISIBLE,SW_NORMAL,CW_USEDEFAULT,CW_USEDEFAULT,width,height,cursor,m_center_window,false);
            }
            break;

            case FIXED:
            {
                // create fixed window
                m_window = new Win32Window("PTC_DIRECTX_WINDOWED_FIXED",m_title,extended,WS_VISIBLE|WS_SYSMENU|WS_CAPTION|WS_MINIMIZE,SW_NORMAL,CW_USEDEFAULT,CW_USEDEFAULT,width,height,cursor,m_center_window,false);
            }
            break;
        }
    }

    // create hook on window
    m_hook = new DirectXHook(*this,m_window->handle(),GetCurrentThreadId(),cursor,m_window->managed());

    // set cooperative level for windowed output
    m_display.cooperative(m_window->handle(),false);

    // open windowed display
    m_display.open();

    // initialize primary surface
    m_primary.initialize(*m_window,m_library.lpDD2());

    // single paged primary surface
    m_primary.primary(1,false,false);

    // check for secondary primary mode
    if (m_primary_mode_windowed==SECONDARY)
    {
        // setup secondary surface
        m_primary.secondary(width,height);
    }
}


void DirectXConsole::internal_post_open_setup()
{
    // setup primary syncronization
    m_primary.synchronize(m_synchronized_update);

    // create win32 keyboard
    m_keyboard = new Win32Keyboard(m_window->handle(),m_window->thread(),false);

    // update window
    m_window->update();

    // set open flag
    m_open = true;
}


void DirectXConsole::internal_reset()
{
    // delete keyboard
    delete m_keyboard;

    // delete hook
    delete m_hook;

    // close primary
    m_primary.close();

    // delete window
    delete m_window;

    // clear pointers
    m_keyboard = 0;
    m_window = 0;
    m_hook = 0;
}




void DirectXConsole::paint()
{
    // no painting when locked or not open
    if (m_locked || !m_open) return;

    // paint primary
    m_primary.paint();
}




void DirectXConsole::check_open() const
{
    #ifdef __DEBUG__

        // check that console is open
        if (!m_open) throw Error("console is not open (dx)");

    #else

        // no checking in release build

    #endif
}


void DirectXConsole::check_unlocked() const
{
    #ifdef __DEBUG__

        // check that console is unlocked
        if (m_locked) throw Error("console is not unlocked");

    #else

        // no checking in release build

    #endif
}
