//
// Clear class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// include files
#include <string.h>
#include "Core/Clear.h"
#include "Core/Color.h"
#include "Core/Error.h"
#include "Core/Config.h"
#include "Core/Format.h"




DLLAPI PTCAPI Clear::Clear()
{ 
    // defaults
    m_maximum_r = 0;
    m_maximum_g = 0;
    m_maximum_b = 0;
    m_maximum_a = 0;
    m_maximum_index = 0;
    
    // initialize hermes
    if (!Hermes_Init()) throw Error("could not initialize hermes");

    // static default format
    static Format default_format;

    // default current format
    m_format = &default_format;

    // create hermes clearer instance
    m_handle = Hermes_ClearerInstance();

    // check hermes clearer instance
    if (!m_handle) throw Error("could not create hermes clearer instance"); 
}


DLLAPI PTCAPI Clear::~Clear()
{ 
    // return the clearer instance
    Hermes_ClearerReturn(m_handle);

    // free hermes
    Hermes_Done();
}




DLLAPI void PTCAPI Clear::request(const Format &format)
{ 
    // check format type
    if (format.direct())
    {
        // direct color
        m_maximum_r = maximum(format.r());
        m_maximum_g = maximum(format.g());
        m_maximum_b = maximum(format.b());
        m_maximum_a = maximum(format.a());
        m_maximum_index = 0;
    }
    else
    {
        // indexed color
        m_maximum_r = 0;
        m_maximum_r = 0;
        m_maximum_r = 0;
        m_maximum_r = 0;
        m_maximum_index = (1<<format.bits()) - 1;
    }

#ifdef __COMPILER_SUPPORTS_CONST_CAST__

    // cast from const format to hermes format pointer
    HermesFormat *hermes_format = const_cast<HermesFormat*>(&format.m_format);

#else

    // cast from const format to hermes format pointer
    HermesFormat *hermes_format = (HermesFormat*)&format.m_format;

#endif

    // request surface clear for this format
    if (!Hermes_ClearerRequest(m_handle,hermes_format)) throw Error("unsupported clear format");

    // update current format
    *m_format = format;
}




DLLAPI void PTCAPI Clear::clear(void *pixels,int x,int y,int width,int height,int pitch,const Color &color)
{ 
    #ifdef __DEBUG__

        //
        // This checking is performed only when __DEBUG__ is defined,
        // and can be used to track down errors early caused by passing
        // null pointers to the clear function.
        //
        // Even though technically clear should never recieve a null
        // pointer, we provide a check here to assist in debugging
        // just in case it ever does!
        //

        // check pixels pointer
        if (!pixels) throw Error("null pixels pointer in clear");

    #else

        // In release build no checking is performed for the sake of efficiency.

    #endif

    /*
    // setup clear value
    int value = 0;
    if (color.direct() && (color.r() || color.g() || color.b())) value = 0xFFFFFFFF;
    else if (color.direct() && color.index()) value = 0xFFFFFFFF;

    // get memory pointers
    char8 *s = ((char8*)pixels) + pitch*y + x*4;    // kludge: assumes 4 bytes per pixel

    // clear a line at a time
    for (int i=0; i<height; i++)
    {
        // clear line
        memset(s,value,width*4);

        // next line
        s += pitch;
    }
    */
        
    // check format type
    if (m_format->direct())
    {
        // check color type
        if (!color.direct()) throw Error("direct pixel formats can only be cleared with direct color");

        // setup clear color
        int32 r = (int32) ( color.r() * m_maximum_r );
        int32 g = (int32) ( color.g() * m_maximum_g );
        int32 b = (int32) ( color.b() * m_maximum_b );
        int32 a = (int32) ( color.a() * m_maximum_a );

        // clamp red
        if (r>m_maximum_r) r = m_maximum_r;
        else if (r<0) r = 0;

        // clamp green
        if (g>m_maximum_g) g = m_maximum_g;
        else if (g<0) g = 0;

        // clamp blue
        if (b>m_maximum_b) b = m_maximum_b;
        else if (b<0) b = 0;

        // clamp alpha
        if (a>m_maximum_a) a = m_maximum_a;
        else if (a<0) a = 0;

        // perform the clearing
        Hermes_ClearerClear(m_handle,pixels,x,y,width,height,pitch,r,g,b,a);
    }
    else
    {
        // check color type
        if (!color.indexed()) throw Error("indexed pixel formats can only be cleared with indexed color");

        // setup clear index
        int32 index = (int32) color.index();

        // clamp color index
        if (index>m_maximum_index) index = m_maximum_index;
        else if (index<0) index = 0;

        // perform the clearing
        Hermes_ClearerClear(m_handle,pixels,x,y,width,height,pitch,0,0,0,index);
    }
}




int32 Clear::maximum(int32 mask)
{
    // check mask
    if (!mask) return 0;

    // find the first bit set in the mask
    while (!(mask&1)) mask>>=1;

    // size of mask
    int32 size = 1;
    
    // iterate through the block of consecutive bits
    while (mask&1)
    {
        // increase size
        size<<=1;

        // next bit
        mask>>=1;
    }

    // return maximum
    return size - 1;
}
