//
// VGA Interface class for PTC 2.0 C++ API
// Copyright (c) 1998 Jonathan Matthew (jmatthew@uq.net.au)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

#include <conio.h>

#include "dpmi.h"
#include "vga.h"

void vgamode();
#pragma aux vgamode = "mov ax, 13h" "int 10h" modify [ax];

void biostextmode();
#pragma aux biostextmode = "mov ax, 3" "int 10h" modify [ax];


#ifndef NO_ASSEMBLER

// Assembler fakemode conversion
extern "C" {
int32 *PlaneBlt2_RBG(char *src, int32 *dest, int rows);
#pragma aux PlaneBlt2_RBG "_*" parm [ESI] [EDI] [EDX] value [EDI] modify [EAX EBX ECX EDX ESI];
}

#else

// C++ fakemode conversion
static int32 *PlaneBlt2_RBG(char *src, int32 *dest, int rows);

#endif

VGA::VGA(int xres, int yres, int modetype)
{

    // check dimensions
    if ((xres > 320) || (yres > 200)) {
        throw Error("no vga mode available");
    }

    // set up variables
    m_pages = 1;
    m_width = 320;
    m_height = 200;
    m_mode_type = modetype;

    // set mode
    vgamode();
    m_ptr = (char *)0xA0000;            // DJGPP: adjust low mem ptr

    // set up format
    if (modetype == RGB332) {
        m_pitch = 320;
        m_format = Format(8, 0xE0, 0x1C, 0x3);
    } else if (modetype == INDEX8) {
        m_pitch = 320;
        m_format = Format(8);
    } else if (modetype == FAKEMODE2A) {
        m_pitch = 640;
        m_format = Format(16, 0xF800, 0x7E0, 0x1F);

        // while we're here, tweak to 320x400
        wait_retrace();
        outp(0x3D4,0x11); outp(0x3D5,inp(0x3D5) & 0x7F);
        outp(0x3C2,0x63);
        outp(0x3D4,0x0);  outp(0x3D5,0x5F);
        outp(0x3D4,0x1);  outp(0x3D5,0x4F);
        outp(0x3D4,0x2);  outp(0x3D5,0x50);
        outp(0x3D4,0x3);  outp(0x3D5,0x82);
        outp(0x3D4,0x4);  outp(0x3D5,0x54);
        outp(0x3D4,0x5);  outp(0x3D5,0x80);
        outp(0x3D4,0x6);  outp(0x3D5,0xBF);
        outp(0x3D4,0x7);  outp(0x3D5,0x1F);
        outp(0x3D4,0x8);  outp(0x3D5,0x00);
        outp(0x3D4,0x9);  outp(0x3D5,0x40);
        outp(0x3D4,0x10); outp(0x3D5,0x9C);
        outp(0x3D4,0x11); outp(0x3D5,0x8E);
        outp(0x3D4,0x12); outp(0x3D5,0x8F);
        outp(0x3D4,0x13); outp(0x3D5,0x28);
        outp(0x3D4,0x14); outp(0x3D5,0x00);
        outp(0x3D4,0x15); outp(0x3D5,0x96);
        outp(0x3D4,0x16); outp(0x3D5,0xB9);
        outp(0x3D4,0x17); outp(0x3D5,0xE3);
        outp(0x3C4,0x01); outp(0x3C5,0x01);
        outp(0x3C4,0x04); outp(0x3C5,0x06);
        outp(0x3CE,0x05); outp(0x3CF,0x40);
        outp(0x3CE,0x06); outp(0x3CF,0x05);
        outp(0x3CE,0x06); outp(0x3CF,0x05);
        clear_memory();
    } else {
        // something weird is going on
        throw Error("what the fuck am i doing?");
    }

    // set palette for RGB332 and FAKEMODE2A modes
    if (modetype == RGB332) {
        int pal[256];
        for (int i=0; i<256; i++) {
            int r = (int)(((i&0xE0)>>5) * (255.0 / 7.0));
            int g = (int)(((i&0x1C)>>2) * (255.0 / 7.0));
            int b = (int)(((i&0x03)   ) * (255.0 / 3.0));
            pal[i] = (r<<16)|(g<<8)|(b);
        }
        palette((const int32 *)pal);
    } else if (modetype == FAKEMODE2A) {
        int pal[256];
        for (int i=0; i<256; i++) {
            // taken from PTC 0.73
            if (!(i&0x80)) {
                // bit 7 = 0 (top section)
                // red (4 bits)
                int r = ((i&0x78)>>3)*(255.0/15.0);
                // blue (3 bits)
                int b = (i&0x07) * (255.0 / 7.0);
                pal[i] = (r<<16) | (b);
            } else {
                // bit 7 = 1 (bottom section)
                // green
                int g = (i&0x1F) * (255.0 / 31.0);
                pal[i] = g<<8;
            }
        }
        palette((const int32 *)pal);
    }
}

VGA::~VGA() {
    if (m_init) {
        biostextmode();         // DJGPP: replace
        m_init = 0;
    }
}

void VGA::palette(const int32 *data)
{
    outp(0x3c8, 0);
    for (int i=0; i<256; i++) {
        int c = (data[i] >> 2) & 0x003F3F3F;
        outp(0x3c9, c>>16);
        outp(0x3c9, c>>8);
        outp(0x3c9, c);
    }
}

void VGA::wait_retrace()
{
    while ( inp(0x3DA) & 8) {}
    while (!inp(0x3DA) & 8) {}
}

void VGA::clear_memory()
{
    wait_retrace();

    char *dest = (char *)0xA0000;       // DJGPP: adjust
    for (int strip=0; strip<32; strip++) {
        outpw(0x3C4, 0x102);
        memset(dest, 0, 2048);
        outpw(0x3C4, 0x202);
        memset(dest, 0, 2048);
        outpw(0x3C4, 0x402);
        memset(dest, 0, 2048);
        outpw(0x3C4, 0x802);
        memset(dest, 0, 2048);
        dest+=2048;
    }
}


void VGA::fakemode_load(char *src, int wvr)
{
    // DJGPP: adjust
    int32 *dest = (int32 *)0xA0000;

    for (int w=0; w<25; w++) {
        // plane 0
        outpw(0x3C4, 0x102);
        PlaneBlt2_RBG(src+0, dest, 8);

        // plane 1
        outpw(0x3C4, 0x202);
        PlaneBlt2_RBG(src+2, dest, 8);

        // plane 2
        outpw(0x3C4, 0x402);
        PlaneBlt2_RBG(src+4, dest, 8);

        // plane 3
        outpw(0x3C4, 0x802);
        dest = PlaneBlt2_RBG(src+6, dest, 8);
        src += 320*4*4;
    }
    if (wvr) wait_retrace();
}

#ifdef NO_ASSEMBLER

static int32 *PlaneBlt2_RBG(char *src, int32 *dest, int rows)
{
    for (int row=0; row<rows; row++) {
        for (int col=0; col<20; col++) {
            unsigned int gl = ((src[ 0])<< 0) | ((src[ 8])<< 8) |
                              ((src[16])<<16) | ((src[24])<<24);
            unsigned int b = gl;
            gl &= 0xC0C0C0C0;
            gl >>= 6;

            unsigned int gh = ((src[ 1])<< 0) | ((src[ 9])<< 8) |
                              ((src[17])<<16) | ((src[25])<<24);
            unsigned int r = gh;
            gh &= 0x07070707;
            gh <<= 2;

            b &=  0x1C1C1C1C;
            b >>= 2;

            r &=  0xF0F0F0F0;
            r >>= 1;

            r += b;
            gl += gh;
            gl |= 0x80808080;

            dest[0] = r;
            dest[20] = gl;

            src += 4*4*2;
            dest++;
        }
        dest += 20;
    }
    return dest;
}
#endif

void *VGA::lock()
{
    // DJGPP: enable near ptrs if necessary
    return m_ptr;
}

void VGA::unlock()
{
    // DJGPP: disable near ptrs if necessary
}

Format &VGA::format()
{
    return m_format;
}
