//
// VBE Interface class for PTC 2.0 C++ API
// Copyright (c) 1998 Jonathan Matthew (jmatthew@uq.net.au)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

#include <stdio.h>
#include <conio.h>

#include "Types.h"
#include "Error.h"
#include "Format.h"
#include "tables.h"
#include "vbe.h"
#include "dpmi.h"

VBE::VBE(int xres, int yres, int format, int flags)
{
    int mode;
    if ((format < 0) || (format > 3))
        throw Error("illegal source format");

// DJGPP: enable near pointers
    vesa_init();
    find_virtual_mode(&mode, format, xres, yres, flags);
    m_lfb = 0;

    set_vid_mode(mode, xres, yres, flags);

    page_set_init();
// DJGPP: disable near pointers
    m_mode = mode;
    m_init = 1;
    m_locked = 0;
}

VBE::~VBE()
{
    if (m_init) {
        set_vid_mode(3);        // text mode
        dpmi.dos_free(&m_dosmem);
        m_init = 0;
    }
}

int VBE::vesa_error(int eax)
{
    int result = (eax & 0xFFFF);
    if (result != 0x4F) return 1;
    return 0;
}

int VBE::identify_pixel_format()
{
    if (m_mib->bits_per_pixel == 8) {
        m_format = 10;
        return 0;
    } else {
        for (int i=0; i<10; i++) {
            if ((m_mib->red_pos == pixel_format_table[i][0]) &&
                (m_mib->red_mask_size == pixel_format_table[i][1]) &&
                (m_mib->green_pos == pixel_format_table[i][2]) &&
                (m_mib->green_mask_size == pixel_format_table[i][3]) &&
                (m_mib->blue_pos == pixel_format_table[i][4]) &&
                (m_mib->blue_mask_size == pixel_format_table[i][5]) &&
                (m_mib->rsvd_pos == pixel_format_table[i][6]) &&
                (m_mib->rsvd_mask_size == pixel_format_table[i][7])) {
                    m_format = i;
                    return 0;
            }
        }
    }
    return 1;
}

int VBE::find_vid_mode(int *mode, int mt, int xres, int yres, int flags)
{
    DPMI::dpmi_regs dr;
    short *ml = m_modelist;
    int index = mt & 127;
    int bxres, byres;
    int mbpp;

    // set up best x,y res according to flags
    if (flags&1) {
        // flexible
        bxres = 0xffffff;
        byres = 0xffffff;
    } else {
        // not flexible
        bxres = xres;
        byres = yres;
    }

    *mode = 0;
    if ((mt < 0) || (index > 4) || (mt == 3)) return 1;
    while ((*ml != -1) && (*ml)) {
        // modeinfo call
        memset(&dr, 0, sizeof(DPMI::dpmi_regs));
        dr.l.eax = 0x4F01;
        dr.l.ecx = *ml;
        dr.l.es = m_dosmem.seg;
        dr.l.edi = 512;
        dpmi.dos_int(0x10, &dr, &dr);
        if (vesa_error(dr.l.eax)) {
            ml++;
            continue;
        }

        // check mode
        if (!identify_pixel_format()) {
            mbpp = pixel_format_table[m_format][8];

            if ((m_mib->x_res >= xres) && (m_mib->y_res >= yres) &&
                (m_mib->x_res <=bxres) && (m_mib->y_res <=byres) &&
                (m_mib->mem_model == vid_mode_model[index]) &&
                ((m_mib->mode_attr & 17) == 17) &&
                (mbpp == vid_mode_rbpp[index])) {
                if (mt & 128) {
                    if ((m_mib->mode_attr & 128) == 128) {
                        *mode = *ml;
                        bxres = m_mib->x_res;
                        byres = m_mib->y_res;
                    }
                } else {
                        // check for banked mode availability
                    if ((m_mib->mode_attr & 64) == 0) {
                        *mode = *ml;
                        bxres = m_mib->x_res;
                        byres = m_mib->y_res;
                    }
                }
            }
        }
        ml++;
    }

    if (!*mode) return 1;

    // get modeinfo again
    memset(&dr, 0, sizeof(DPMI::dpmi_regs));
    dr.l.eax = 0x4F01;
    dr.l.ecx = *mode;
    dr.l.es = m_dosmem.seg;
    dr.l.edi = 512;
    dpmi.dos_int(0x10, &dr, &dr);
    if (vesa_error(dr.l.eax)) {
        *mode = 0;
        return 1;
    }
    identify_pixel_format();
    return 0;
}

void VBE::find_virtual_mode(int *mode, int type, int xres, int yres, int flags)
{
    int i,mt;
    *mode = 0;

    // try for exact match first
    for (i=0; i<9; i++) {
        mt = vid_mode_search_order[type][i];
        if (!find_vid_mode(mode, mt, xres, yres, 0)) break;
    }

    // if that failed, try flexible if it's allowed
    if (!(*mode) && (flags&1)) {
        for (i=0; i<9; i++) {
            mt = vid_mode_search_order[type][i];
            if (!find_vid_mode(mode, mt, xres, yres, 1)) break;
        }
    }

    // set things up if a mode has been found
    if (*mode) {
        m_width = m_mib->x_res;
        m_height = m_mib->y_res;
        if (mt & 128) {
            m_mode_type = 0;
            *mode |= 0x4000;  // lfb flag
        } else m_mode_type = 1;
        return;
    }
    throw Error("unable to find a video mode");
}

void VBE::vesa_init()
{
    DPMI::dpmi_regs dr;
    int *vpr;

    // get dos memory
    dpmi.dos_alloc(2048, &m_dosmem);
    memset(m_dosmem.ptr, 0, 1024);
    vpr = (int *)m_dosmem.ptr;
    *vpr = '2EBV';              // Intel I hate you.

    // get vbe info
    memset(&dr, 0, sizeof(DPMI::dpmi_regs));
    dr.l.eax = 0x4F00;
    dr.l.es = m_dosmem.seg;
    dr.l.edi = 0;
    dpmi.dos_int(0x10, &dr, &dr);
    if (vesa_error(dr.l.eax)) {
        dpmi.dos_free(&m_dosmem);
        throw Error("unable to initialise vbe");
    }
    m_vib = (vbe_info_block *)m_dosmem.ptr;
    m_mib = (mode_info_block *)((int)m_dosmem.ptr + 512);

    // check 'VESA'
    if (m_vib->vbe_sig != 'ASEV') {     // Intel I hate you.
        dpmi.dos_free(&m_dosmem);
        throw Error("unable to initialise vbe");
    }

    // get version
    m_vbe_ver = 0;
    if (m_vib->vbe_maj_ver > 1) m_vbe_ver = 1;

    // get modelist pointer
    m_modelist = (short *)(((m_vib->vid_mode_ptr&0xFFFF0000)>>12)+
        (m_vib->vid_mode_ptr&0xFFFF));          // DJGPP: adjust
}

void biostextmode();            // DJGPP: add equivalent function
#pragma aux biostextmode = "mov ax, 3" "int 10h" modify [ax];

void VBE::set_vid_mode(int mode, int xres, int yres, int flags)
{
    DPMI::dpmi_regs dr;

    // unmap lfb if mapped
    if (m_lfb) {
        dpmi.unlock_mem((int)m_screen, m_vib->total_mem << 16);
        dpmi.unmap_linear_mem((int)m_screen);
        m_lfb = 0;
    }

    // if mode==3, just set text mode
    if (mode == 3) {
        biostextmode();
        return;
    }

    // set the mode
    memset(&dr, 0, sizeof(DPMI::dpmi_regs));
    dr.l.eax = 0x4F02;
    dr.l.ebx = mode & 0x7FFF;
    dpmi.dos_int(0x10, &dr, &dr);
    if (vesa_error(dr.l.eax)) {
        biostextmode();
        throw Error("unable to set video mode");
    }

    // get mode pitch
    memset(&dr, 0, sizeof(DPMI::dpmi_regs));
    dr.l.eax = 0x4F06;
    dr.l.ebx = 1;
    dpmi.dos_int(0x10, &dr, &dr);
    m_pitch = dr.w.bx;
    m_bpp = (pixel_format_table[m_format][8]+7)/8;

    // create ptc Format object
    if (m_bpp == 1) m_ptcformat = Format(8);
    else {
        m_ptcformat = Format(m_bpp*8,
            0xFFFFFFFF & (((1 << m_mib->red_mask_size)-1) << m_mib->red_pos),
            0xFFFFFFFF & (((1 << m_mib->green_mask_size)-1) << m_mib->green_pos),
            0xFFFFFFFF & (((1 << m_mib->blue_mask_size)-1) << m_mib->blue_pos));
    }

    // calculate number of pages available (max 3, anyway)
    int pn = ((m_mib->y_res * m_pitch) + 0xFFFF) & (~0xFFFF);
    int totalmem = m_vib->total_mem << 16;
    if (pn * 3 < totalmem) m_pages = 3;
    else if (pn * 2 < totalmem) m_pages = 2;
    else if (pn < totalmem) m_pages = 1;
    else {
        // not enough memory for one page.
        biostextmode();
        throw Error("not enough video memory");
    }

    // map lfb if necessary
    if (mode & 0x4000) {
        if (dpmi.map_linear_mem(m_mib->phys_base_ptr, m_vib->total_mem<<16,
            (int *)&m_screen)) {
            biostextmode();
            throw Error("unable to map lfb");
        }
        dpmi.lock_mem((int)m_screen, m_vib->total_mem << 16);

        // calculate offset necessary to centre image on screen
        int dispoffset = ((m_mib->y_res/2 - yres/2) * m_pitch) +
                          ((m_mib->x_res/2 - xres/2) * m_bpp);
        m_page_addr[0] = m_screen + dispoffset;
        m_page_addr[1] = m_screen + pn + dispoffset;
        m_page_addr[2] = m_screen + (2*pn) + dispoffset;
                // DJGPP: adjust pointers

        // clear pages (seems it doesn't always automatically happen)
        memset(m_screen, 0, m_mib->y_res * m_pitch);
        if (m_pages > 1) {
            memset(m_screen + pn, 0, m_mib->y_res * m_pitch);
            if (m_pages > 2) {
                memset(m_screen + (2*pn), 0, m_mib->y_res * m_pitch);
            }
        }

        m_lfb = 1;
    } else {
        m_screen = (char *)((m_mib->win_a_seg&0xFFFF) << 4);
        // DJGPP: adjust

        // calculate starting banks for each page
        int page_bank_size = (m_mib->y_res * m_pitch) / (m_mib->win_gran * 1024);
        int pixels_per_bank = (m_mib->win_gran * 1024) / m_bpp;
        pn /= (m_mib->win_gran * 1024);
        m_page_bank[0] = 0;
        m_page_bank[1] = pn;
        m_page_bank[2] = pn * 2;
        m_lfb = 0;
    }
    m_current_page = 0;

    // Wide DAC support thanks to Mikko Tiihonen.
    // enable wide dac when in indexed mode and not disallowed
    m_wide_dac = 0;
    if (m_bpp == 1 && (m_vib->caps[0]&1) && (flags&2)) {
        // enable 8bit dac
        memset(&dr, 0, sizeof(DPMI::dpmi_regs));
        dr.l.eax = 0x4F08;
        dr.l.ebx = 0x0800;
        dpmi.dos_int(0x10, &dr, &dr);
        if ((!vesa_error(dr.l.eax)) && (dr.b.bh == 8))
            m_wide_dac = 1;
    }
}

void VBE::page_set_init()
{
    int pn;
    if (!m_pitch) throw Error("large scale coder fuckup");

    pn = (((m_mib->y_res * m_pitch) >> 16) + 1) << 16;
    m_page_cx[0] = 0;
    m_page_dx[0] = 0;
    m_page_cx[1] = (pn % m_pitch) / m_bpp;
    m_page_dx[1] = (pn / m_pitch);
    m_page_cx[2] = ((pn*2) % m_pitch) / m_bpp;
    m_page_dx[2] = ((pn*2) / m_pitch);

}

void VBE::page_flip(int page, int wvr)
{
    if ((page != m_current_page) && (page >= 0) && (page < m_pages)) {
        DPMI::dpmi_regs dr;
        memset(&dr, 0, sizeof(DPMI::dpmi_regs));
        dr.l.ecx = m_page_cx[page];
        dr.l.edx = m_page_dx[page];

        if (wvr) {
            if (m_vbe_ver) dr.l.ebx = 0x80;
            else {
                // VBE 1.x -> can assume VGA compatibility.
                while ( inp(0x3DA) & 8) {}
                while (!inp(0x3DA) & 8) {}
            }
        }

        dr.l.eax = 0x4F07;
        dpmi.dos_int(0x10, &dr, &dr);
        m_current_page = page;
    }
}

void VBE::set_bank(int bank)
{
    DPMI::dpmi_regs dr;
    memset(&dr, 0, sizeof(DPMI::dpmi_regs));
    dr.l.eax = 0x4F05;
    dr.l.ebx = 0;
    dr.l.edx = bank;
    dpmi.dos_int(0x10, &dr, &dr);
}

// helpful macros for the next bit
#define MOVES(a)    sr -= (a); src += (a)
#define MOVED(a)    d += (a); dest += (a)
#define MOVEL(a)    dline -= (a); sline -= (a)
#define RESETL      sline = scanline; dline = m_pitch
#define MOVEBANK    bank += bankinc; set_bank(bank)

void VBE::banked_load(char *src, int destpage, int wvr)
{
    // DJGPP: enable near pointers
    char *dest = m_screen;
    int sr = m_height*m_width*m_bpp;
    int d = 0;
    int scanline = m_width*m_bpp;
    int sline;
    int dline;
    RESETL;

    int bank = m_page_bank[destpage];
    int banksize = m_mib->win_size * 1024;
    int bankinc = m_mib->win_size / m_mib->win_gran;
    set_bank(bank);

    while (sr>0) {
        if (d+sline > banksize) {
            // can only draw part of scanline
            int copybytes = banksize - d;
            memcpy(dest, src, copybytes);
            d = 0;
            dest = m_screen;
            MOVES(copybytes);
            MOVEL(copybytes);
            MOVEBANK;
            continue;
        } else if (d+dline > banksize) {
            // can draw all of scanline but not add pitch
            memcpy(dest,src,sline);
            MOVED(dline - banksize);
            MOVES(sline);
            RESETL;
            MOVEBANK;
            continue;
        } else {
            // can copy all of scanline and add pitch
            memcpy(dest,src,sline);
            MOVED(dline);
            MOVES(sline);
            RESETL;
        }
    }
    page_flip(destpage, wvr);
}

void *VBE::lock(int page)
{

    if ((page >= 0) && (page < m_pages) && (!m_locked)) {
        // DJGPP: enable near pointers
        return (void *)m_page_addr[page];
        m_locked = 1;
    }
    else return NULL;
}

void VBE::unlock()
{
    if (m_locked) {
        // DJGPP: disable near pointers
        m_locked = 0;
    }
}

Format &VBE::format()
{
    return m_ptcformat;
}

void VBE::palette(const int32 palette[])
{

    if (m_vbe_ver) {
        // VBE > 1.x palette set

        DPMI::dpmi_regs dr;
        int32 *dest = ((int32 *)m_dosmem.ptr) + 256;

        // DJGPP: enable near pointers (or use movedata)

        // copy data to low memory area
        if (m_wide_dac) {
            memcpy(dest, palette, 1024);
        } else {
            for (int i=0; i<256; i++)
                *dest++ = (((*palette++)>>2)&0x003F3F3F);
        }

        // call palette set function
        memset(&dr, 0, sizeof(DPMI::dpmi_regs));
        dr.l.eax = 0x4F09;
        dr.l.ecx = 256;
        dr.l.ebx = 0;
        dr.l.edx = 0;
        dr.l.es = m_dosmem.seg;
        dr.l.edi = 1024;
        dpmi.dos_int(0x10, &dr, &dr);

        // DJGPP: disable near pointers
    } else {
        // VBE 1.x palette set (direct VGA palette set)
        char *pal = (char *)palette;
        outp(0x3c8, 0);
        if (m_wide_dac) {
            for (int i=0; i<256; i++) {
                outp(0x3c9, pal[2]);
                outp(0x3c9, pal[1]);
                outp(0x3c9, pal[0]);
                pal++;
            }
        } else {
            for (int i=0; i<256; i++) {
                outp(0x3c9, pal[2]>>2);
                outp(0x3c9, pal[1]>>2);
                outp(0x3c9, pal[0]>>2);
                pal+=4;
            }
        }
    }
}
