//
// Console class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// Modified for port to DOS by Jonathan Matthew (jmatthew@uq.net.au)

// adding vga support.

// include files
#include <stdio.h>
#include <conio.h>

#include "Area.h"
#include "Clip.h"
#include "Error.h"
#include "Console.h"
#include "VBE.h"
#include "VGA.h"



Console::Console()
{
    // defaults
    m_width  = 0;
    m_height = 0;
    m_pitch  = 0;
    m_locked = false;
    m_vbe = 0;
    m_vga = 0;

    // configuration defaults:
    m_disallow_banked = false;
    m_force_triple = false;
    m_flexible = true;
    m_wait_retrace = true;
    m_wide_dac = true;
    m_vga_modes = true;
}


Console::~Console()
{
    // close
    close();
}




bool Console::option(const char string[])
{
    // if i implement the protected mode interface,
    // there will be an option to enable it.
    // it's too buggy to have enabled by default.

    if (!strcmp(string, "no banked modes")) {
        m_disallow_banked = true;
    } else if (!strcmp(string, "force triple buffering")) {
        m_force_triple = true;
    } else if (!strcmp(string, "force exact match")) {
        m_flexible = false;
    } else if (!strcmp(string, "no retrace checking")) {
        m_wait_retrace = false;
    } else if (!strcmp(string, "disable wide dac")) {
        m_wide_dac = false;
    } else if (!strcmp(string, "no vga modes")) {
        m_vga_modes = false;
    }

    // option not recognized
    return false;
}




void Console::open(const char title[],const Format &format)
{
    // open console
    open(title,640,400,format);
}


void Console::open(const char title[],int width,int height,const Format &format)
{

    // close
    close();

    // first try VBE
    try {

        m_vbe = new VBE(width, height, format.bytes()-1, m_flexible |
            (m_wide_dac<<1));

        // setup data
        m_width  = m_vbe->width();
        m_height = m_vbe->height();
        m_format = m_vbe->format();
        m_pitch  = m_vbe->pitch();
        m_bufmode = m_vbe->pages();
        m_banked = m_vbe->banked();

        // RGB332 support thanks to Mikko Tiihonen
        // if requested RGB332 and got a 8bit mode then ...
        if (format.bits() == 8 && format.direct() &&
            m_vbe->format().bits() == 8) {
            m_format = Format(8,0xe0,0x1c,0x03);

          // Taken from PTC 0.72 and modified
            for (int i=0;i<256;i++) {
                int r = (int)(((i&0xE0)>>5) * (255.0 / 7.0));
                int g = (int)(((i&0x1C)>>2) * (255.0 / 7.0));
                int b = (int)( (i&0x03)     * (255.0 / 3.0));

                m_palette[i] = (r<<16) | (g<<8) | (b<<0);
            }

            // set 332 palette
            m_vbe->palette((const int32 *)m_palette);
        } else
            m_format = m_vbe->format();


        // check for banked mode when disallowed
        if (m_disallow_banked && m_banked) {
            close();
            throw Error("banked modes disallowed");
        }

        // check for double buffering when triple buffering is forced
        if (m_force_triple && (m_bufmode != 3)) {
            close();
            throw Error("double buffering disallowed");
        }

        if (m_bufmode > 1) {
            // set up double/triple buffering
            m_displaypage = 0;
            m_drawpage = 1;
            m_sparepage = 2;
        } else m_drawpage = 0;

        if ((m_bufmode == 1) || m_banked) {
            m_pitch = m_width * format.bytes();
            m_offscreen = new char[m_pitch*m_height];
            if (!m_offscreen) throw Error("out of memory");
            memset(m_offscreen, 0, m_pitch*m_height);
        }
    }
    catch (Error e)
    {
        // if VGA allowed, try it, otherwise rethrow
        if (!m_vga_modes) e.rethrow("unable to find a video mode");
        delete m_vbe;
        m_vbe = 0;

        int vgamodetype = 0;    // assume fakemode
        if (format.bits() == 8) {
            vgamodetype = 2;
            // rgb332?
            if (format.direct()) vgamodetype = 1;
        }

        m_vga = new VGA(width, height, vgamodetype);

        // setup data
        m_width = m_vga->width();
        m_height = m_vga->height();
        m_format = m_vga->format();
        m_pitch = m_vga->pitch();
        m_bufmode = m_vga->pages();

        // setup offscreen buffer
        m_offscreen = new char[m_height*m_pitch];
        if (!m_offscreen) throw Error("out of memory");
        memset(m_offscreen, 0, m_height*m_pitch);
    }
}


void Console::close()
{
    // unlock the console if required
    if (m_locked) unlock();

    if (m_vbe) {
        delete m_vbe;
        m_vbe  = 0;
    }

    if (m_vga) {
        delete m_vga;
        m_vga = 0;
    }

    // defaults
    m_width  = 0;
    m_height = 0;
    m_pitch  = 0;
    m_format = Format();
    m_locked = false;
}




void Console::update()
{
    if (m_vbe) {
        if (!m_banked) {
            if (m_bufmode > 1) {
                // flip to last drawn page
                m_vbe->page_flip(m_drawpage, m_wait_retrace);

                // update page variables for double or triple buffering
                int tmp = m_displaypage;
                m_displaypage = m_drawpage;

                if (m_bufmode == 3) {
                    m_drawpage = m_sparepage;
                    m_sparepage = tmp;
                } else {
                    m_drawpage = tmp;
                }
            } else {
                // wait for retrace if requested
                if (m_wait_retrace) m_vbe->page_flip(0, 1);

                memcpy(m_vbe->lock(0), m_offscreen, m_pitch*m_height);
                m_vbe->unlock();
            }
        } else {
            // this has to be done by the VBE object, otherwise the
            // console would have to know too much about it all.
            m_vbe->banked_load(m_offscreen, m_drawpage, m_wait_retrace);

            // update page variables
            if (m_bufmode > 1) {
                int tmp = m_displaypage;
                m_displaypage = m_drawpage;
                if (m_bufmode == 3) {
                    m_drawpage = m_sparepage;
                    m_sparepage = tmp;
                } else {
                    m_drawpage = tmp;
                }
            }
        }
    } else if (m_vga) {
        if (m_vga->fakemode()) {
            // fakemode update
            m_vga->fakemode_load(m_offscreen, m_wait_retrace);
        } else {
            memcpy(m_vga->lock(), m_offscreen, m_pitch*m_height);
            m_vga->unlock();
        }
    }
}

void Console::update(const Area &area)
{
    if (m_vbe) {
        if ((m_bufmode > 1) || (m_banked)) {
            // area updates are impossible in double/triple buffered modes
            // and are not supported in banked modes.  may never be.
            update();
        } else {
            // wait for retrace if necessary
            if (m_wait_retrace) m_vbe->page_flip(0, 1);

            // copy area to screen
            char *dest = (char *)m_vbe->lock(0);
            char *src = m_offscreen;
            int btc = (area.right() - area.left()) * m_format.bytes();
            int lbs = area.left() * m_format.bytes();
            int wbs = m_width * m_format.bytes();
            int wbd = m_vbe->pitch();
            dest += area.top() * wbd;
            src += area.top() * wbs;
            for (int y=area.top(); y < area.bottom(); y++) {
                memcpy(dest+lbs, src+lbs, btc);
                dest += wbd;
                src += wbs;
            }
            m_vbe->unlock();
        }
    } else if (m_vga) {
        if (m_vga->fakemode()) {
            // area updates not supported in fakemode
            update();
        } else {
            // wait for retrace if necessary
            if (m_wait_retrace) m_vga->wait_retrace();

            // copy area to screen
            char *dest = (char *)m_vga->lock();
            char *src = m_offscreen;
            int btc = (area.right() - area.left()) * m_format.bytes();
            int lbs = area.left() * m_format.bytes();
            int wbs = m_width * m_format.bytes();
            int wbd = m_vga->pitch();
            dest += area.top() * wbd;
            src += area.top() * wbs;
            for (int y=area.top(); y < area.bottom(); y++) {
                memcpy(dest+lbs, src+lbs, btc);
                dest += wbd;
                src += wbs;
            }
            m_vga->unlock();
        }
    }
}




bool Console::key()
{
    // may implement key() using direct access to the BIOS data area,
    // since it would be much faster, and DOS/4GW has weird bugs
    // with realmode interrupts.
    return kbhit();
}


int Console::read()
{
    // may implement getch() using direct BIOS data area access.
    return getch();
}




void Console::copy(BaseSurface &surface)
{
    // lock console
    void *pixels = lock();

    try
    {
        // palette data
        const int32 *data = 0;

        // get palette for indexed color
        if (format().indexed()) data = palette();

        // load surface pixels to other surface
        surface.load(pixels,width(),height(),format(),data);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console to surface");
    }
}


void Console::copy(BaseSurface &surface,const Area &source,const Area &destination)
{
    // lock console
    void *pixels = lock();

    try
    {
        // palette data
        const int32 *data = 0;

        // get palette for indexed color
        if (format().indexed()) data = palette();

        // load surface pixels to other surface
        surface.load(pixels,width(),height(),source,destination,format(),data);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console area to surface area");
    }
}




void* Console::lock()
{
    // fail if the console is already locked
    if (m_locked) throw Error("console is already locked");

    // we are locked...
    m_locked = true;

    // get pixels pointer
    if (m_vbe) {
        if ((m_bufmode > 1) && (!m_banked)) {
            return m_vbe->lock(m_drawpage);
        } else return (void *)m_offscreen;
    } else if (m_vga) {
        return (void *)m_offscreen;
    }
    return NULL;                // never happens anyway.. hopefully.
}


void Console::unlock()
{
    // fail if the console is not locked
    if (!m_locked) throw Error("console is not locked");

    if (m_vbe) m_vbe->unlock();

    // we are unlocked
    m_locked = false;
}




void Console::load(const void *pixels,int width,int height,const Format &format,const int32 palette[])
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,0,0,width,height,width*format.bytes(),console_pixels,0,0,this->width(),this->height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to load pixels to console");
    }
}


void Console::load(const void *pixels,int width,int height,const Area &source,const Area &destination,const Format &format,const int32 palette[])
{
    // lock console
    void *console_pixels = lock();

    try
    {
	    // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,width,height),source);
        Area clipped_destination = Clip::clip(Area(0,0,this->width(),this->height()),destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),width*format.bytes(),
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to load pixels to console area");
    }
}




void Console::save(void *pixels,int width,int height,const Format &format,const int32 palette[])
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(m_format,format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,width*format.bytes());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to save console pixels");
    }
}


void Console::save(void *pixels,int width,int height,const Area &source,const Area &destination,const Format &format,const int32 palette[])
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,this->width(),this->height()),source);
        Area clipped_destination = Clip::clip(Area(0,0,width,height),destination);

            // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),width*format.bytes());
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to save console area pixels");
    }
}




void Console::clear(const Color &color)
{
    // lock console
    void *pixels = lock();

    try
    {
        // request clear
        m_clear.request(format());

        // clear console
        m_clear.clear(pixels,0,0,width(),height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console");
    }
}


void Console::clear(const Color &color,const Area &area)
{
    // lock console
    void *pixels = lock();

    try
    {
        // clip area
        Area clipped_area = Clip::clip(Area(0,0,width(),height()),area);

        // request clear
        m_clear.request(format());

        // clear console area
        m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console area");
    }
}




void Console::palette(const int32 palette[])
{
    // fail silently if not indexed color
    if (!format().indexed()) return;

    memcpy(m_palette, palette, 1024);
    if (m_vbe) {
        m_vbe->palette(palette);
    } else if (m_vga) {
        m_vga->palette(palette);
    }
}


const int32* Console::palette()
{
    return m_palette;
}




int Console::width() const
{
    return m_width;
}


int Console::height() const
{
    return m_height;
}


int Console::pitch() const
{
    return m_pitch;
}


const Format& Console::format() const
{
    return m_format;
}


const char* Console::name() const
{
    return "DOS";
}

// API extensions:

bool Console::vrstate()
{
    return m_wait_retrace;
}

void Console::vrstate(bool newstate)
{
    m_wait_retrace = newstate;
}
