//
// TGA image reader(8,16,24,32 bit picture) for PTC 2.0 C++ API
// Copyright (c) Robbin Bonthond (elemental@chaos.concepts.nl)
// This source code is licensed under the GNU GPL
//

#include <fstream.h>
#include <stdlib.h>
#include <stdio.h>

#include "ptc.h"
#include "mem.h"
#include "Image.h"

//
//  Image loader
//

Image::Image( char *filename )
{
  m_locked = false;

  // recognize file format and load it
  tga( filename );
}

Image::~Image()
{
  if( m_locked ) unlock();
  delete [] m_pixels;
}

void Image::copy( Surface &surface )
{
  surface.load( m_pixels, m_width, m_height, m_format, m_palette );
}

void Image::copy( Surface &surface, const Area &src, const Area &dst )
{
  surface.load( m_pixels, m_width, m_height, src, dst, m_format, m_palette );
}

int Image::width() const
{
  return m_width;
}

int Image::height() const
{
  return m_height;
}

const Format& Image::format() const
{
  return m_format;
}

const Palette& Image::palette() const
{
  return m_palette;
}

void* Image::lock()
{
  if( m_locked ) throw Error("Image is already locked");
  m_locked = true;
  return m_pixels;
}

void Image::unlock()
{
  if( !m_locked ) throw Error("Image is already unlocked");
  m_locked = false;
}

//
//  TGA loader (8,16,24,32 loader)
//

Image::tga( char *filename )
{
  // open image file
  ifstream is( filename, ios::binary );
  if( !is )
    throw Error( "cannot read file" );

  // read header
  char8 *header = new char8 [18];
  is.seekg(0);
  is.read( header, 18 );
  m_width = (header[13]<<8)+header[12];
  m_height = (header[15]<<8)+header[14];

  // setup format
  switch( header[16] ) {
    case  8: m_format = Format( 8 ); break;
    case 16: m_format = Format( 16, 0x7C00, 0x03E0, 0x001F ); break;
    case 24: m_format = Format( 24, 0x00FF0000, 0x0000FF00, 0x000000FF ); break;
    case 32: m_format = Format( 32, 0x00FF0000, 0x0000FF00, 0x000000FF ); break;
    default: throw Error( "unrecognized tga format" );
  }

  // read palette
  m_palette = Palette();
  int p_flag = header[1];
  unsigned p_offset = (header[4]<<8)+header[3];
  unsigned p_length = (header[6]<<8)+header[5];
  unsigned p_bits = header[7];
  unsigned p_size;

  if( p_flag && p_length ) {

    // reject anything > 256 palette entries
    if( p_length > 256 )
      throw Error("tga has wrong palette");

    // setup temp palette buffer
    p_size = p_length * p_bits / 8;
    char8 *temp = new char8 [p_size];
    int32 *data = new int32 [p_length];

    // read into temp buffer
    is.seekg( 18+p_offset );
    is.read( temp, p_size );

    // initialize palette format
    int i;
    switch( p_bits ) {
      case 16:
        for( i=0; i < p_length; i++ ) {
          int32 r = (temp[i*2+1]&0x7c)<<3;
          int32 g = (((temp[i*2+1]&3)<<3)+(temp[i*2]&0xe0)>>5)<<3;
          int32 b = (temp[i*2]&0x1f)<<3;
          data[i] = (r<<16)|(g<<8)|b;
        }
        break;
      case 24:
        for( i=0; i < p_length; i++ ) {
          int32 r = temp[i*3+2]&0xff;
          int32 g = temp[i*3+1]&0xff;
          int32 b = temp[i*3+0]&0xff;
          data[i] = (r<<16)|(g<<8)|b;
        }
        break;
      case 32:
        for( i=0; i < p_length; i++ ) {
          int32 r = temp[i*4+3]&0xff;
          int32 g = temp[i*4+2]&0xff;
          int32 b = temp[i*4+1]&0xff;
          data[i] = (r<<16)|(g<<8)|b;
        }
        break;
    }

    // load palette
    m_palette.load( data );

    // remove temp data
    delete [] temp;
    delete [] data;
  }

  // calculate size of pixels;
  int32 size = m_width * m_height * m_format.bytes();
  int pitch = m_width * m_format.bytes();

  // allocate image pixels
  m_pixels = new char8 [size];

  // read image pixels one line at a time (upside down)
  if( p_flag )
    is.seekg( 18+p_offset+p_size );
  else
    is.seekg( 18+header[0] );

  for( int y=m_height-1; y>=0; y-- )
    is.read( (char*)m_pixels+y*pitch, pitch );

  // free data
  delete [] header;
}


