//
// Stress test for OpenPTC 1.0 C++ Implementation
// Copyright (c) Glenn Fiedler (ptc@gaffer.org)
// This source code is in the public domain
//

// include files
#include "ptc.h"
#include <time.h>
#include <stdio.h>
#include <stdlib.h>


// configuration defines
#define MAXIMUM_WIDTH  100
#define MAXIMUM_HEIGHT 100


inline int random(int max)
{
    // check max parameter
    if (max<=0) return 0;

    // random number
    return rand() % max;
}


inline float random()
{
    // random float
    return rand() / (float)RAND_MAX;
}


inline bool chance(int number,int probability)
{
    // a chance of number in probability
    if (random(probability)<=number) return true;
    else return false;
}


inline int random_width()
{
    // random width
    return random(MAXIMUM_WIDTH);
}


inline int random_height()
{
    // random height
    return random(MAXIMUM_HEIGHT);
}


inline Format random_format()
{
    // random format
    switch (random(20))
    {
        case 0:  return Format(32,0x00FF0000,0x0000FF00,0x000000FF,0xFF000000);
        case 1:  return Format(24,0x00FF0000,0x0000FF00,0x000000FF);
        case 2:  return Format(16,0xF800,0x07E0,0x001F);
        case 3:  return Format(16,0x0F00,0x00F0,0x000F,0xF000);
        case 4:  return Format(8,0,0,0,0xFF);
        case 5:  return Format(8);
        case 6:  return Format();
        default: return Format(32,0x00FF0000,0x0000FF00,0x000000FF);
    }
}


inline Area random_area(BaseSurface &surface)
{
    // get surface width and height
    const int width  = surface.width();
    const int height = surface.height();
    
    // random area
    switch (10)
    {
        case 0:   return Area();
        case 1:   return Area(10,10,20,20);
        case 2:   return Area(30,30,10,10);
        default:  return Area( random(width*3)-width, random(height*3)-height, random(width*3)-width, random(height*3)-height );
    }
}


inline Color random_color()
{
    // toggle flag
    if (!random(1))
    {
        // direct color
        return Color(random(),random(),random(),random());
    }
    else
    {
        // indexed color
        return Color(random(256));
    }
}


void print(const Color &color)
{
    // check color type
    if (color.direct())
    {
        // check alpha
        if (!color.a())
        {
            // direct color without alpha
            printf("Color(%f,%f,%f)",color.r(),color.g(),color.b());
        }
        else
        {
            // direct color with alpha
            printf("Color(%f,%f,%f,%f)",color.r(),color.g(),color.b(),color.a());
        }
    }
    else if (color.indexed())
    {
        // indexed color
        printf("Color(%d)",color.index());
    }
    else
    {
        // default color
        printf("Color()");
    }
}


void print(const Area &area)
{
    // print area
    printf("Area(%d,%d,%d,%d)",area.left(),area.top(),area.right(),area.bottom());
}


void print(const Format &format)
{
    // check format type
    if (format.direct())
    {
        // check alpha
        if (!format.a())
        {
            // direct color format without alpha
            printf("Format(%d,0x%.8X,0x%.8X,0x%.8X)",format.bits(),format.r(),format.g(),format.b());
        }
        else
        {
            // direct color format with alpha
            printf("Format(%d,0x%.8X,0x%.8X,0x%.8X,0x%.8X)",format.bits(),format.r(),format.g(),format.b(),format.a());
        }
    }
    else if (format.indexed())
    {
        // indexed color format
        printf("Format(%d)",format.bits());
    }
    else
    {
        // default format
        printf("Format()");
    }
}


void main()
{
    try
    {
        // counters
        int i = 0;
        int j = 0;

        // base surface array
        const surfaces = 1000;
        BaseSurface *surface[surfaces];
        for (i=0; i<surfaces; i++) surface[i] = 0;
        
        // static load array
        const int load_width  = MAXIMUM_WIDTH;
        const int load_height = MAXIMUM_HEIGHT;
        const int load_bytes  = 4;
        const int load_size = load_width * load_height * load_bytes;
        const ubyte load[load_size];
        for (i=0; i<load_size; i++) load[i] = random(256);

        // static save array
        const int save_width  = MAXIMUM_WIDTH;
        const int save_height = MAXIMUM_HEIGHT;
        const int save_bytes  = 4;
        const int save_size = save_width * save_height * save_bytes;
        const ubyte save[save_size];

        // stress repeat count
        unsigned count = 100000;

        // stress loop
        while (count--)
        {
            //
            // Surface stress testing
            // ----------------------
            // 

            // create a surface
            if (chance(1,10))
            {
                // random surface array index
                const int index = random(surfaces);

                // check surface is free
                if (!surface[index])
                {
                    // random surface parameters
                    const int width = random_width();
                    const int height = random_height();
                    const Format format = random_format();

                    // print information
                    printf("surface[%d] = new Surface(%d,%d,",index,width,height);
                    print(format);
                    printf(");\n");

                    try
                    {
                        // create new surface
                        surface[index] = new Surface(width,height,format);
                    }
                    catch (Error &error)
                    {
                        // failed to create surface
                        printf("failed to create surface[%d]\nreason: %s\n",index,error.message());

                        // force zero
                        surface[index] = 0;
                    }

                    // end block
                    printf("\n");
                }
            }

            // destroy a surface
            if (chance(1,10))
            {
                // random surface array index
                const int index = random(surfaces);

                // check surface is allocated
                if (surface[index])
                {
                    // print information
                    printf("delete surface[%d];\n",index);

                    try
                    {
                        // delete surface
                        delete surface[index];
                    }
                    catch (Error &error)
                    {
                        // failed to delete surface (?)
                        printf("failed to delete surface[%d]\nreason: %s\n",index,error.message());
                    }

                    // clear pointer
                    surface[index] = 0;

                    // end block
                    printf("\n");
                }
            }

            // clear a surface
            if (chance(1,10))
            {
                // random surface array index
                const int index = random(surfaces);

                // check surface is allocated
                if (surface[index])
                {
                    // print information
                    printf("surface[%d]->clear();\n",index);

                    try
                    {
                        // clear surface
                        surface[index]->clear();
                    }
                    catch (Error &error)
                    {
                        // failed to clear surface
                        printf("failed to clear surface[%d]\nreason: %s\n",index,error.message());
                    }

                    // end block
                    printf("\n");
                }
            }

            // clear a surface area
            if (chance(1,10))
            {
                // random surface array index
                const int index = random(surfaces);

                // check surface is allocated
                if (surface[index])
                {
                    // get random clear parameters
                    const Area area = random_area(*surface[index]);
                    const Color color = random_color();

                    // print information
                    printf("surface[%d]->clear(",index);
                    print(area);
                    printf(",");
                    print(color);
                    printf(");\n");

                    try
                    {
                        // clear surface
                        surface[index]->clear();
                    }
                    catch (Error &error)
                    {
                        // failed to clear surface
                        printf("failed to clear surface[%d]\nreason: %s\n",index,error.message());
                    }

                    // end block
                    printf("\n");
                }
            }

            // copy between surfaces
            if (chance(1,10))
            {
                // random surface array indices
                const int a = random(surfaces);
                const int b = random(surfaces);

                // check surfaces are allocated
                if (surface[a] && surface[b])
                {
                    // print information
                    printf("surface[%d]->copy(*surface[%d]);\n",a,b);

                    try
                    {
                        // copy surface a to b
                        surface[a]->copy(*surface[b]);
                    }
                    catch (Error &error)
                    {
                        // failed to copy surface
                        printf("failed to copy surface[%d] to surface[%d]\nreason: %s\n",a,b,error.message());
                    }

                    // end block
                    printf("\n");
                }
            }

            // area copy between surfaces
            if (chance(1,10))
            {
                // random surface array indices
                const int a = random(surfaces);
                const int b = random(surfaces);

                // check surfaces are allocated
                if (surface[a] && surface[b])
                {
                    // get random copy parameters
                    const Area area = random_area(*surface[a]);

                    // print information
                    printf("surface[%d]->copy(*surface[%d]",a,b);
                    print(area);
                    printf(",");
                    print(area);
                    printf(");\n");

                    try
                    {
                        // copy surface a to b
                        surface[a]->copy(*surface[b],area,area);
                    }
                    catch (Error &error)
                    {
                        // failed to copy surface
                        printf("failed to copy surface[%d] to surface[%d]\nreason: %s\n",a,b,error.message());
                    }

                    // end block
                    printf("\n");
                }
            }

            // area copy between surfaces with stretching
            if (chance(1,10))
            {
                // random surface array indices
                const int a = random(surfaces);
                const int b = random(surfaces);

                // check surfaces are allocated
                if (surface[a] && surface[b])
                {
                    // get random copy parameters
                    const Area area_a = random_area(*surface[a]);
                    const Area area_b = random_area(*surface[b]);

                    // print information
                    printf("surface[%d]->copy(*surface[%d]",a,b);
                    print(area_a);
                    printf(",");
                    print(area_b);
                    printf(");\n");

                    try
                    {
                        // copy surface a to b
                        surface[a]->copy(*surface[b],area_a,area_b);
                    }
                    catch (Error &error)
                    {
                        // failed to copy surface
                        printf("failed to copy surface[%d] to surface[%d]\nreason: %s\n",a,b,error.message());
                    }

                    // end block
                    printf("\n");
                }
            }

            // load surface

            // load surface area

            // save surface

            // save surface area

            // copy surface to console

            // copy surface area to console area

            // lock and unlock surface

            // set surface palette

            // get surface palette

            // get surface information

            //
            // Console stress testing
            // ----------------------
            // 

            // create console

            // destroy console

            // update console

            // flush console

            // close console

            // copy from console to surface
            
            // load console

            // save console
        }

        // free all surfaces
        for (i=0; i<surfaces; i++)
        {
            // print information
            printf("delete surface[%d];\n",i);
            
            // delete surface
            delete surface[i];
        }
    }
    catch (Error &error)
    {
        // report error
        error.report();
    }
}
