//
// X11 console implementation for OpenPTC 1.0 C++ API
// Copyright (c) 1998/99 Christian Nentwich (brn@eleet.mcb.at)
// The OpenPTC 1.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//


// include files
#include <string.h>
#include <stdlib.h>
#include <fstream.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include "ptconfig.h"
#include "../Core/Error.h"
#include "../Core/Base.h"
#include "../Core/Area.h"
#include "../Core/Copy.h"
#include "Console.h"
#include "Display.h"
#include "DGADisplay.h"
#include "WindowDisplay.h"



Console::Console()
{
    // defaults
    x11disp = 0;
    m_flags = 0;
    m_title = 0;

#ifdef HAVE_PTHREADS
    if (!XInitThreads()) {
      throw Error("Cannot initialise X threads");
    }
#endif

    // get user configured options, try user and system options

    configure("/usr/share/ptc/ptc.conf");

    char *homeptr=getenv("HOME");

    // Let's not introduce a security risk and check for size.. (this might
    // run setuid after all)

    if (homeptr && strlen(homeptr)<255) {
      char fullpath[256];

      strcpy(fullpath,homeptr);
      strcat(fullpath,"/.ptc.conf");

      configure(fullpath);
    }
}



Console::~Console()
{
    // Close the console
    close();

    if (m_title) {
      delete [] m_title;
      m_title=0;
    }
}



bool Console::option(const char str[])
{ 
  if(!strcmp(str,"dga pedantic init"))
  { m_flags|=PTC_X11_PEDANTIC_DGA;
    return true;
  }

  if(!strcmp(str,"dga off"))
  { m_flags|=PTC_X11_NODGA;
    return true;
  }

  if(!strcmp(str,"leave window open"))
  { m_flags|=PTC_X11_LEAVE_WINDOW;
    return true;
  }

  if(!strcmp(str,"leave display open"))
  { m_flags|=PTC_X11_LEAVE_DISPLAY;
    return true;
  }

  // As a last resort, pass it to the copy object to handle
  if (x11disp)
  return x11disp->m_copy.option(str);

  return false;
}


// console modes
const Mode* Console::modes()
{ 
  return m_modes;
}


void Console::open(const char title[],const Mode &mode,int pages)
{ setTitle(title);
  return;
}


// TODO: Find current pixel depth
void Console::open(const char title[],int pages)
{ setTitle(title);
  open(title,Format(32,0xff0000,0xff00,0xff));
}


// TODO: Find best DGA mode
void Console::open(const char title[],const Format& format,int pages)
{ setTitle(title);
  open(title,640,480,format);
}


void Console::open(const char title[],int width,int height,
  const Format& format,int pages)
{
    close();

    setTitle(title);

    // Check if we can open an X display
    Display *disp=XOpenDisplay(0);
    if(disp==NULL) throw Error("Cannot open X display");


    // DefaultScreen should be fine
    int screen=DefaultScreen(disp);


    x11disp=0;
#ifndef HAVE_DGA
    m_flags|=PTC_X11_NODGA;
#endif

    if (!(m_flags&PTC_X11_NODGA)) {
      try {
        x11disp=new X11DGADisplay();

        x11disp->flags(m_flags|PTC_X11_LEAVE_DISPLAY);

        x11disp->open(title,width,height,format,disp,screen);
      }
      catch (Error e) 
      { delete x11disp;
        x11disp=0;
      }

      if (x11disp) x11disp->flags(m_flags);
    }
    
    if (!x11disp)
    { x11disp=new X11WindowDisplay();

      x11disp->flags(m_flags);

      x11disp->open(title,width,height,format,disp,screen);
    }

}



void Console::open(Display *disp,int screen,Window w,const Format& format)
{
  // We already have a display and a window, so we create a windowdisplay ;)
  x11disp=new X11WindowDisplay();

  x11disp->open(disp,screen,w,format);

  x11disp->flags(m_flags);
}



void Console::open(Display *disp,int screen,Window window,const Format &format,
  int x,int y,int w,int h)
{
  // We already have a display and a window, so we create a windowdisplay ;)
  x11disp=new X11WindowDisplay();
  
  x11disp->open(disp,screen,window,format,x,y,w,h);

  x11disp->flags(m_flags);
}



void Console::close()
{
    // close display
    if(x11disp)
    { 
      delete x11disp;
      x11disp=0;
    }
}



void Console::copy(BaseSurface &surface)
{ surface.load(x11disp->lock(),x11disp->width(),x11disp->height(),
	       x11disp->pitch(),
               x11disp->format(),x11disp->palette());
}


void Console::copy(BaseSurface &surface,const Area &source,
  const Area &destination)
{
  surface.load(x11disp->lock(),x11disp->width(),x11disp->height(),
	       x11disp->pitch(),
	       x11disp->format(),x11disp->palette(),
               source,destination);
}



bool Console::isWindow() const
{
  if(x11disp->id()==PTC_WINDOW) return true;
  else
  return false;
}



void Console::load(const void *pixels,int width,int height,int pitch,
  const Format &format,const Palette &palette)
{ x11disp->load(pixels,width,height,pitch,format,palette);
}



void Console::load(const void *pixels,int width,int height,int pitch,
  const Format &format,const Palette &palette,
  const Area &source,const Area &destination)
{ x11disp->load(pixels,width,height,pitch,format,palette,source,destination);
}



void Console::save(void *pixels,int width,int height,int pitch,
  const Format &format,const Palette &palette)
{ x11disp->save(pixels,width,height,pitch,format,palette);
}



void Console::save(void *pixels,int width,int height,int pitch,
  const Format &format,const Palette &palette,
  const Area &source,const Area &destination)
 
{ x11disp->save(pixels,width,height,pitch,format,palette,source,destination);
}



void Console::clear()
{ static Color c_index(0);
  static Color c_direct(0,0,0,0);

  if (format().direct()) clear(c_direct);
  else clear(c_index);
}


const Area& Console::area() const
{ static Area area;
  area=Area(0,0,x11disp->width(),x11disp->height()); 
  return area;
}
 

int Console::pages() const
{
  return 1;
}


const char* Console::name() const
{ return "X11";
}


const char* Console::title() const 
{ return m_title;
}


const char* Console::information() const
{ static char info[512];

  
  sprintf(info,"PTC X11, %s, %dx%d, %d bit",isWindow()?
	  "windowed mode":"direct graphics access (DGA) mode",width(),height(),
	  format().bits());
  
  return info;
}


void Console::configure(const char filename[])
{
    // open configuration file
    ifstream file(filename,ios::in);
    
    // check file
    if (!file) {
      //cerr << "Could not open " << filename << "!" << endl;
      return;
    }

    // read option strings
    while (!file.eof())
    {
        // option line
        char line[1024];

        // read line from file
        file.getline(line,1024);

        // process option
        option(line);
    }

}


void Console::setTitle(const char title[])
{
  if (m_title) { 
    delete [] m_title;
    m_title=0;
  }

  m_title=new char[strlen(title)+1];
  strcpy(m_title,title);
}
