/*
    intel2gas.cc   intel to at&t assembly converter
    Copyright (c) 1999 Mikko Tiihonen (mikko.tiihonen@hut.fi)
    This source code is licensed under the GNU LGPL
  
    Please refer to the file COPYING.LIB contained in the distribution for
    licensing conditions
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "intel2gas.h"
#include "loaddata.h"
#include "strhandle.h"

/* Older gas versions went berzerk whith ' characters in comments
 * define this to convert them to ^'s */
#undef OLD_GAS

#define TAG_TEXT    "text"     // text until space or next in parseline
#define TAG_STRING  "string"   // text until eol or next in parseline
#define TAG_NEWLINE "nl"       // newline
#define TAG_COMMENT "cm"       // #
#define TAG_MORE    "more"     // restarts parser
#define TAG_LST     "lt"       // <
#define TAG_GTT     "gt"       // >
#define TAG_AT      "@"        // current position in input line

/* Base pointers to linked lists */
list_t	       *list;
syntaxlist_t   *basesyntaxlist = NULL;
syntaxlist_t   *syntaxlist = NULL;

int             basepos; // first character on line
char            src_comment = ';'; // default comment characters
char            dst_comment = '#'; // can be changed from syntax files

// scratch buffer (can be overwritten any time)
char            tempstr[TEMP_SIZE];



syntaxlist_t *findSyntax(char const *name)
{
    syntaxlist_t *l = basesyntaxlist;
    while (l) {
	if (!strcasecmp(name,l->name))
	    return l;
	l = l->next;
    }
    return NULL;
}


void setVar(char const * const varname, char const * const value)
{
    variable_t *v = syntaxlist->var;
    while (v) {
	if (!strcasecmp(v->name, varname)) {
	    // value already exists
    	    strcpy(v->value, value);
	    return;
	}
	v = v->next;
    }
    v = new variable_t;
    strcpy(v->name, varname);
    strcpy(v->value, value);
    v->next = NULL;

    if (syntaxlist->var == NULL) {
	syntaxlist->var = v;
	return;
    }

    // put last in list
    variable_t *vv = syntaxlist->var;
    while (vv->next)
	vv = vv->next;

    vv->next = v;
}


char const *fetchVar(char const *tag)
{
    variable_t *v = syntaxlist->var;
    while (v) {
	if (!strcasecmp(tag,v->name))
	    return v->value;
	v = v->next;
    }
    return "";
//    printf("Variable not found\n");
//    exit(1);
}


void expand_string(char const *str, char *result)
{
    int pos = 0;
    while (*str != '\0') {
	if (*str == '<') {
	    char tag[STR_LEN+1];
	    str = str_splitat_close(str+1,tag,'<','>');
	    if (!strcasecmp(tag,TAG_NEWLINE)) {
		*result++ = '\n';
		memset(result,' ',basepos);
		result += basepos;
		pos = 0;
		continue;
	    }
	    if (!strcasecmp(tag,TAG_LST)) {
		*result++ = '<';
		continue;
	    }
	    if (!strcasecmp(tag,TAG_GTT)) {
		*result++ = '>';
		continue;
	    }
	    if (!strcasecmp(tag,TAG_COMMENT)) {
	        *result++ = dst_comment;
		continue;
	    }
	    if (tag[0] == '?') { // the mystery equation
	      char t1[STR_LEN], *t2, *values;
	      values = str_splitat(tag+1,t1,':');
	      t2 = str_cutat(t1,'=');

	      char T1[STR_LEN], T2[STR_LEN];
	      expand_string(t1,T1); expand_string(t2,T2);

	      t2 = str_cutat(values,',');
	      if (strcmp(T1,T2)) {
		expand_string(t2,T1);
	      } else {
		expand_string(values,T1);
	      }	      
	      
	      strcpy(result,T1);
	      result += strlen(T1);
	      continue;
	    }
	    char const *var = fetchVar(tag);
	    str_striplastnumber(tag);
	    if (!strcasecmp(tag,TAG_AT)) {
	        int at = atoi(var);
		if (pos < at) {
		    memset(result,' ',at-pos);
		    result += at-pos;
		    pos = at;
		}
		continue;
	    }
	    strcpy(result,var);
	    result += strlen(var);
	    pos += strlen(var);
	    continue;
	}
	*result++ = *str++;
	pos++;
    }
    *result = '\0';
}


void setTheseVars(variable_t const *vars)
{
    while (vars) {
	char temp[TEMP_SIZE];
	expand_string(vars->value,temp);
	setVar(vars->name,temp);
	vars = vars->next;
    }
}


static void find_tag_list(char const *source, char const tag[], 
			  int bits, char *result)
{
    list_t const *l = list;
    int bestsize = 0;
    char const *bestword = 0;
    list_t const *best = 0;

    *result = '\0';
    
    while (l) {
	// correct list == tag & bits
	if ((l->bit & bits) && !strcasecmp(l->name, tag)) {
	    char * const *word = &l->words[0];
	    while (*word != NULL) {
	        // match source chars to a word in list
	        int size = strlen(*word);
		if (!strncasecmp(source, *word, size)) {
		    if (size > bestsize) {
			bestsize = size;
			bestword = *word;
		        best = l;
		    }
		}
		word ++;
	    }
	}
	l = l->next;
    }
    if (bestsize) {
        strcpy(result, bestword);
        // set all variables associated with the list
        setTheseVars(best->assignments);
    }
}


/* Copies numbers (of specified type) to ouput.
 * Returns amount characters found */
static int numberparse(char const *tag, char const *input, char *output)
{
    int base;
    if (!strcasecmp(tag,"hex"))
	base = 16;
    else if (!strcasecmp(tag,"dec"))
	base = 10;
    else if (!strcasecmp(tag,"bin"))
	base = 2;
    else if (!strcasecmp(tag,"oct"))
	base = 8;
    else {
	*output = '\0';
	return 0;
    }
    char *ptr;
    strtol(input,&ptr,base);
    const int len = ptr-input;
    strncpy(output,input,len);
    output[len] = '\0';
    return len;
}

int str_parse(syntaxlist_t *s, char *parse, char *outline);

int parsematch(char *text, char *parse)
{
    char const *oldpos = text;
    while (!str_empty(parse)) {
	if (*parse == '<') {
	    int bits;
	    char tag[STR_LEN+1];
	    char str[TEMP_SIZE];
	    char number;
	    int len;
	    parse = str_splitat(parse+1,tag,'>');
	    str_splitstrat(tag,tag,':',str);
	    if (!str_empty(str)) {
		char *dummy;
		bits = strtol(str,&dummy,16);
		str[0] = '\0';
	    } else bits = 0xff;
	    number = str_striplastnumber(tag);
	    // Try to match a new syntax file
	    syntaxlist_t *l = findSyntax(tag);
	    if (l != NULL) {
		char temp[TEMP_SIZE];
		char stop = *parse;
		if (stop == ' ') {
		    stop = *str_skipspace(parse);
		    if (stop == '<') {
			char *pos = strchr(text,' ');
			if (pos == NULL) {
			    stop = ' ';
			} else {
			    stop = *str_skipspace(pos);
			}
		    }
		}
		variable_t *old = syntaxlist->var;
		syntaxlist->var = NULL;
		str_copyuntil(temp,text,*parse);
		len = str_parse(l,temp,str);
		syntaxlist->var = old;
		if (len==0) return 0;
	    } else {
		find_tag_list(text, tag, bits, str);
		len = strlen(str);
		if (len==0) {
		    len = numberparse(tag, text, str);
		    if (len == 0) {
			if (!strcasecmp(tag,TAG_TEXT)) {
			    str_copyuntil(str,text,*parse,' ');
			    len = strlen(str);
			} else if (!strcasecmp(tag,TAG_STRING)) {
			    // TODO: handle escapes in string
			    if (*parse == ' ') // ends in space: assume eol
  			        strcpy(str,text);
			    else
			        str_copyuntil(str,text,*parse);
			    len = strlen(str);
			} else if (!strcasecmp(tag,TAG_AT)) {
			    sprintf(str,"%d",text-oldpos);
			    len = 0; // don't advance
	 		} else if (!strcasecmp(tag,TAG_MORE)) {
			    if (str_empty(text)) {
				*str = '\0';
				len = 0;
			    } else {
    				variable_t *old = syntaxlist->var;
				syntaxlist->var = NULL;
				len = str_parse(findSyntax("main"),text,str);
				deleteVars(syntaxlist);
				syntaxlist->var = old;
				if (len==0) return 0;
			    }
			} else {
			    return 0;
			}
		    }
		}
	    }    
	    str_catchar(tag,number);
	    setVar(tag,str);
	    text += len;
	    continue;
	}
	if (*parse == ' ') {
	    parse++;
	    text = str_skipspace(text);
	    continue;
	}
	if (tolower(*parse) != tolower(*text)) return 0;
	if (str_empty(text)) return 0;
	parse++; text++;
    }
    text = str_skipspace(text);
    if (!str_empty(text)) return 0;
    return text-oldpos;
}


int str_parse(syntaxlist_t *sl, char *parse, char *outline)
{
    *outline = '\0';
    parse = str_skipspace(parse);
    if (str_empty(parse))
	return 0;
    
    syntaxlist_t *old = syntaxlist;
    syntaxlist = sl;
    syntax_t *s = syntaxlist->syntax;
    
    char tempstr[TEMP_SIZE];
    while (1) {
	if (s == NULL) {
	    syntaxlist = old;
	    return 0;
	}
	strcpy(tempstr, s->parseline);
	int size = parsematch(parse, tempstr);
	if (size) {
	    setTheseVars(s->assignments);
	    if (!str_empty(s->output))
	      expand_string(s->output, outline);
	    setTheseVars(s->assignments_after);
	    syntaxlist = old;
	    return size;
	}
	s = s->next;
    }
    return 0;
}

static void CommentConvert(char *str)
{
    while (*str == src_comment)
	*str++ = dst_comment;
#ifdef OLD_GAS
    while (*str) {
	switch (*str) {
	case '\'': *str = '`';
	  // what else doesn't work
	}
	str++;
    }
#endif
}

void mainparse()
{
    char *pos;
    char *comment;
    char temp[TEMP_SIZE];
    char outline[TEMP_SIZE];
    int  skipline = 0;

    syntaxlist_t *mainsyntax = findSyntax("main");

    pos = tempstr;
    comment = str_cutat(pos, src_comment);
    *outline = '\0';

    pos = str_skipspace(pos);
    basepos = pos-tempstr;
    if (!str_empty(pos)) {
	if (!str_parse(mainsyntax, pos, temp)) {
	    fprintf(stderr,"WARNING, Line %d: no syntax match: \"%s\"\n",
		    current_line,tempstr);
	    printf("MISMATCH: \"%s\"\n",tempstr);
	    return;
	}
	if (str_empty(temp))
	    skipline = 1;
	else
	    str_addatpos(outline, temp, basepos);
    }

    if (comment) {
	CommentConvert(comment);
	// restore & convert also the first ';'
	comment--; *comment = dst_comment;
	str_addatpos(outline, comment, comment-tempstr);
    }

    if (!skipline || comment)
        printf("%s\n",outline);
}

static void printHelp(char const *infostr)
{
    fprintf(stderr,"%s\n"\
	   "usage: intel2gas [options] [infile] [-o outfile]\n"\
	   "where options include:\n"\
	   "\t-h\t\tthis help\n"\
	   "\t-i\t\tconvert from intel to at&t format (default)\n"\
	   "\t-g\t\tconvert from at&t to intel format\n"\
	   "\t-c\t\tenable gcc workarounds\n"\
	   "\t-V\t\tshow version\n"\
	   "\tinfile and outfile default to stdin/stdout\n",
	   infostr);
    exit (*infostr != '\0');
}

int main(int argc, char *argv[])
{
    int ch;
    int gccmode = 0;
    char *modedir = "i2g/";
    
    while ((ch = getopt(argc, argv, "o:Vhgic")) != -1) {
	switch(ch) {
	case 'o':
	    if (strcmp(*argv,"-"))
	        if (!freopen(optarg, "w", stdout)) {
		    fprintf(stderr, "Error: Could not create %s\n",optarg);
		    exit(1);
		}
	    break;
	case 'i':
	    modedir = "i2g/";
	    break;
	case 'c':
	    gccmode = 1;
	case 'g':
	    modedir = "g2i/";
	    break;
	case 'V':
	    printf("intel2gas 1.2\n");
	    exit(0);
	default:
	    printHelp("Error: unrecognized parameter\n");
	case 'h':
	    printHelp("");
	}
    }
    argc -= optind;
    argv += optind;
    if (argc > 1)
        printHelp("Error: too many input files");
    if (*argv) {
        if (strcmp(*argv,"-"))
	    if (!freopen(*argv, "r", stdin)) {
	        fprintf(stderr,"Error: Could not open %s\n",*argv);
		exit(1);
	    }
    }

    // load the syntax files
    {
	char *resource_path = getenv("I2G_DATA");
	if (resource_path == NULL)
	    resource_path = RESOURCE_PATH;
	
	strcpy(tempstr, resource_path);
	strcat(tempstr, modedir);
	LoadWordLists(tempstr);

	strcpy(tempstr, resource_path);
	strcat(tempstr, modedir);
	LoadSyntaxis(tempstr);

	// initialize & check variables
	basesyntaxlist = syntaxlist;
	syntaxlist = findSyntax("main");
	if (basesyntaxlist == NULL || syntaxlist == NULL) {
	    printf("fatal error: no %s%smain.syntax found\n",resource_path,modedir);
	    return 1;
	}
    }

    // get the src and dst comment characters
    {
	str_parse(syntaxlist, "General settings:", tempstr);
	char const *str = fetchVar("src_comment");
	if (str != NULL) src_comment = *str;
	str = fetchVar("dst_comment");
	if (str != NULL) dst_comment = *str;
    }
    
    // set <gcc> to skip some syntaxes that gcc produces
    if (gccmode) {
	tempstr[0] = dst_comment; tempstr[1] = '\0';
        setVar("gcc",tempstr);
    }

    current_line = 0;
    ReadNextLine(stdin);

    while (!feof(stdin)) {
	mainparse();
	ReadNextLine(stdin);
    }
    return 0;
}
