//
// Converter_C class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include <memory.h>
#include <iostream.h>
#include "Converter_C.h"
#include "Error.h"
#include "Format.h"




Converter_C::Converter_C()
{
    // setup table
    table = buffer;
}



inline void Converter_C::convert_copy(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // copy pixels
    int bytes = count * copy.s_bytes;
    memcpy(destination,source,bytes);
}







bool Converter_C::request(const Format &source,const Format &destination,
  const bool stretch)
{

   if(source==s_format && destination==d_format && stretch==m_stretch) return true;
   if(stretch) return false;

    // setup formats
    s_format = source;
    d_format = destination;
    m_stretch = stretch;

    // setup bytes per pixel
    s_bytes = source.bits() / 8;
    d_bytes = destination.bits() / 8;

    // check for copy conversion
    if (source==destination) convert = convert_copy;
    else
    {
        // check format type
        if (source.direct())
        {
            // setup direct color
            setup(source,destination);

            // select conversion functions
            convert = function(source,destination);
            generate = 0;
        }
        else
        {
            // palette pixel format
            Format palette(32,0x00FF0000,0x0000FF00,0x000000FF,0xFF000000);

            // setup indexed color
            setup(palette,destination);

            // setup conversion functions
            convert = function(source,destination);
            generate = function(palette,destination);
        }
    }

  // This converter will convert anything
  cerr << "C conversion" << endl << flush;

  return true;
}




void Converter_C::palette(const int32 palette[])
{
    // generate palette table
    if (s_format.indexed() && d_format.direct()) generate(*this,palette,this->buffer,256);
}




void Converter_C::copy(const void *s_pixels,int s_x,int s_y,int s_width,int s_height,int s_pitch,
                void *d_pixels,int d_x,int d_y,int d_width,int d_height,int d_pitch)
{

    // get memory pointers
    const char8 *s = ((const char8*)s_pixels) + s_pitch*s_y + s_x*s_bytes;
    char8 *d = ((char8*)d_pixels) + d_pitch*d_y + d_x*d_bytes;

	/*
	// adjust source pointer
	s += s_pitch * s_y + s_x * s_format.bits()/8;

	// adjust destination pointer
	d += d_pitch * d_y + d_x * d_format.bits()/8;
    */

    // setup data
    int width  = s_width;
    int height = s_height;

    // copy a line at a time
    for (int y=0; y<height; y++)
    {
        // copy line
        convert(*this,s,d,width);

        // next line
        s += s_pitch;
        d += d_pitch;
    }
}




void Converter_C::setup(const Format &source,const Format &destination)
{
    // setup convert information
    info.r = field(source.r(),destination.r());
    info.g = field(source.g(),destination.g());
    info.b = field(source.b(),destination.b());
    info.a = field(source.a(),destination.a());
}


Converter_C::Field Converter_C::field(int32 source,int32 destination)
{
    // analyse source mask
    int source_base, source_size;
    analyse(source,source_base,source_size);

    // analyse destination mask
    int destination_base, destination_size;
    analyse(destination,destination_base,destination_size);

    // setup field
    Field field;
    field.shift = source_base-destination_base - (destination_size-source_size);
    field.mask  = source & (destination << field.shift);
    return field;
}


void Converter_C::analyse(int32 mask,int &base,int &size)
{
    // clear
    base = 0;
    size = 0;

    // check zero mask
    if (!mask) return;

    // find mask base
    while (!(mask&1))
    {
        mask>>=1;
        base++;
    }

    // find mask size
    while (mask&1)
    {
        mask>>=1;
        size++;
    }
}


Converter_C::Function Converter_C::function(const Format &source,const Format &destination)
{
    // check for muhmuh color source format
    if (source.bits()==32 && source.r()==(0xFF<<20) && source.g()==(0xFF<<10) && source.b()==0xFF && source.a()==0)
    {
        // check for RGB888 32bit
        if (destination.bits()==32 && destination.r()==0x00FF0000 && destination.g()==0x0000FF00 && destination.b()==0x000000FF && destination.a()==0) return convert_muhmuh32_direct32_rgb888;
        if (destination.bits()==24 && destination.r()==0x00FF0000 && destination.g()==0x0000FF00 && destination.b()==0x000000FF && destination.a()==0) return convert_muhmuh32_direct24_rgb888;
        if (destination.bits()==16 && destination.r()==0xF800 && destination.g()==0x07E0 && destination.b()==0x001F && destination.a()==0) return convert_muhmuh32_direct16_rgb565;
        if (destination.bits()==16 && destination.r()==0x7C00 && destination.g()==0x03E0 && destination.b()==0x001F && destination.a()==0) return convert_muhmuh32_direct16_rgb555;
    }

    // check format type
    if (source.direct() && destination.direct())
    {
        // rgba direct color
        switch (source.bits())
        {
            case 32:
            {
                // 32bit -> X
                switch (destination.bits())
                {
                    case 32: return convert_direct32_direct32_rgba;
                    case 24: return convert_direct32_direct24_rgba;
                    case 16: return convert_direct32_direct16_rgba;
                    case 8:  return convert_direct32_direct8_rgba;
                }
            }
            break;

            case 24:
            {
                // 24bit -> X
                switch (destination.bits())
                {
                    case 32: return convert_direct24_direct32_rgba;
                    case 24: return convert_direct24_direct24_rgba;
                    case 16: return convert_direct24_direct16_rgba;
                    case 8:  return convert_direct24_direct8_rgba;
                }
            }
            break;

            case 16:
            {
                // 16bit -> X
                switch (destination.bits())
                {
                    case 32: return convert_direct16_direct32_rgba;
                    case 24: return convert_direct16_direct24_rgba;
                    case 16: return convert_direct16_direct16_rgba;
                    case 8:  return convert_direct16_direct8_rgba;
                }
            }
            break;

            case 8:
            {
                // 8bit -> X
                switch (destination.bits())
                {
                    case 32: return convert_direct8_direct32_rgba;
                    case 24: return convert_direct8_direct24_rgba;
                    case 16: return convert_direct8_direct16_rgba;
                    case 8:  return convert_direct8_direct8_rgba;
                }
            }
            break;
        }
    }
    else if (source.indexed())
    {
        // indexed color -> direct color
        switch (source.bits())
        {
            case 8:
            {
                // 8bit -> X
                switch (destination.bits())
                {
                    case 32: return convert_indexed8_direct32;
                    case 24: return convert_indexed8_direct24;
                    case 16: return convert_indexed8_direct16;
                    case 8:  return convert_indexed8_direct8;
                }
            }
            break;
        }
    }
    
    // error message
    throw Error("unsupported pixel format conversion");

    // failure
    return 0;
}




inline int32 Converter_C::read_24(const char8 *source)
{
    // destination
    int32 pixel;

    // setup destination pointer
    char8 *destination = (char8*) &pixel;

    #if defined __LITTLE_ENDIAN__

        // write pixel (little endian)
        *(destination+0) = *(source+3);
        *(destination+1) = *(source+2);
        *(destination+2) = *(source+1);

    #elif defined __BIG_ENDIAN__

        // write pixel (big endian)
        *(destination+0) = *(source+0);
        *(destination+1) = *(source+1);
        *(destination+2) = *(source+2);

    #else

        // compiler error message
        #error you must configure the processor endianness!

    #endif

    // return pixel
    return pixel;
}


inline void Converter_C::write_24(char8 *destination,int32 pixel)
{
    // setup source pointer
    char8 *source = (char8*) &pixel;

    #if defined __LITTLE_ENDIAN__

        // write pixel (little endian)
		*(destination+0) = *(source+0);
        *(destination+1) = *(source+1);
        *(destination+2) = *(source+2);
        /*
		*(destination+0) = *(source+3);
        *(destination+1) = *(source+2);
        *(destination+2) = *(source+1);
		*/

    #elif defined __BIG_ENDIAN__

        // write pixel (big endian)
        *(destination+0) = *(source+0);
        *(destination+1) = *(source+1);
        *(destination+2) = *(source+2);			// test!

    #else

        // compiler error message
        #error you must configure the processor endianness!

    #endif
}




inline void Converter_C::convert_pixel(Converter_C &copy,int32 &pixel)
{
    // mask color components
    int32 r = pixel & copy.info.r.mask;
    int32 g = pixel & copy.info.g.mask;
    int32 b = pixel & copy.info.b.mask;
    int32 a = pixel & copy.info.a.mask;

    // shift color components
    r >>= copy.info.r.shift;
    g >>= copy.info.g.shift;
    b >>= copy.info.b.shift;
    a >>= copy.info.a.shift;

    // combine components
    pixel = r | g | b | a;
}






void Converter_C::convert_muhmuh32_direct32_rgb888(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const int32 *s = (const int32*) source;
    int32 *d = (int32*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = *s;

        // convert
        int32 r = (pixel & (0xFF<<20)) >> 4;
        int32 g = (pixel & (0xFF<<10)) >> 2;
        int32 b = (pixel & (0xFF));

        // write pixel
        *d = r | g | b;
        
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_muhmuh32_direct24_rgb888(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const int32 *s = (const int32*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = *s;

        // convert
        int32 r = (pixel & (0xFF<<20)) >> 4;
        int32 g = (pixel & (0xFF<<10)) >> 2;
        int32 b = (pixel & (0xFF));

        // write pixel
        write_24(d,r|g|b);
        
        // next
        s++;
        d+=3;
    }
}


void Converter_C::convert_muhmuh32_direct16_rgb565(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const int32 *s = (const int32*) source;
    short16 *d = (short16*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = *s;

        // convert
        int32 r = (pixel & (0xF8<<20)) >> 12;
        int32 g = (pixel & (0xFC<<10)) >> 7;
        int32 b = (pixel & (0xF8))     >> 3;

        // combine components
        pixel = r | g | b;

        // write pixel
        *d = (short16) pixel;
        
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_muhmuh32_direct16_rgb555(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const int32 *s = (const int32*) source;
    short16 *d = (short16*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = *s;

        // convert
        int32 r = (pixel & (0xF8<<20)) >> 13;
        int32 g = (pixel & (0xF8<<10)) >> 8;
        int32 b = (pixel & (0xF8)) >> 3;

        // combine components
        pixel = r | g | b;

        // write pixel
        *d = (short16) pixel;
        
        // next
        s++;
        d++;
    }
}




void Converter_C::convert_direct32_direct32_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const int32 *s = (int32*) source;
    int32 *d = (int32*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = pixel;
        
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_direct32_direct24_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const int32 *s = (const int32*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        write_24(d,pixel);

        // next
        s++;
        d+=3;
    }
}


void Converter_C::convert_direct32_direct16_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const int32 *s = (const int32*) source;
    short16 *d = (short16*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = (short16) pixel;
        
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_direct32_direct8_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const int32 *s = (const int32*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = (char8) pixel;
        
        // next
        s++;
        d++;
    }
}




void Converter_C::convert_direct24_direct32_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    int32 *d = (int32*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = read_24(s);

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = pixel;
                
        // next
        s+=3;
        d++;
    }
}


void Converter_C::convert_direct24_direct24_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = read_24(s);

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        write_24(d,pixel);

        // next
        s+=3;
        d+=3;
    }
}


void Converter_C::convert_direct24_direct16_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    short16 *d = (short16*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = read_24(s);

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = (short16) pixel;
        
        // next
        s+=3;
        d++;
    }
}


void Converter_C::convert_direct24_direct8_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = read_24(s);

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = (char8) pixel;
                
        // next
        s+=3;
        d++;
    }
}




void Converter_C::convert_direct16_direct32_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const short16 *s = (const short16*) source;
    int32 *d = (int32*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = (int32) *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = pixel;
                
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_direct16_direct24_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const short16 *s = (const short16*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = (int32) *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        write_24(d,pixel);

        // next
        s++;
        d+=3;
    }
}


void Converter_C::convert_direct16_direct16_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const short16 *s = (const short16*) source;
    short16 *d = (short16*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = (int32) *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = (short16) pixel;
        
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_direct16_direct8_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const short16 *s = (const short16*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = (int32) *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = (char8) pixel;
                
        // next
        s++;
        d++;
    }
}




void Converter_C::convert_direct8_direct32_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    int32 *d = (int32*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = (int32) *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = pixel;
                
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_direct8_direct24_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = (int32) *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        write_24(d,pixel);

        // next
        s++;
        d+=3;
    }
}


void Converter_C::convert_direct8_direct16_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    short16 *d = (short16*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = (int32) *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = (short16) pixel;
        
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_direct8_direct8_rgba(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    char8 *d = (char8*) destination;

    // pixel loop
    while (count--)
    {
        // read pixel
        int32 pixel = (int32) *s;

        // convert pixel
        convert_pixel(copy,pixel);

        // write pixel
        *d = (char8) pixel;
                
        // next
        s++;
        d++;
    }
}




void Converter_C::convert_indexed8_direct32(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    int32 *d = (int32*) destination;
    
    // setup lookup pointer
    int32 *lookup = (int32*) copy.table;

    // pixel loop
    while (count--)
    {
        // convert pixel
        *d = lookup[*s];
        
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_indexed8_direct24(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    char8 *d = (char8*) destination;
    
    // setup lookup pointer
    char8 *lookup = (char8*) copy.table;

    // pixel loop
    while (count--)
    {
        // calculate offset
        int offset = *s * 3;

        // convert pixel
        *(d+0) = lookup[offset+0];
        *(d+1) = lookup[offset+1];
        *(d+2) = lookup[offset+2];

        // next
        s++;
        d+=3;
    }
}


void Converter_C::convert_indexed8_direct16(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    short16 *d = (short16*) destination;
    
    // setup lookup pointer
    short16 *lookup = (short16*) copy.table;

    // pixel loop
    while (count--)
    {
        // convert pixel
        *d = lookup[*s];
        
        // next
        s++;
        d++;
    }
}


void Converter_C::convert_indexed8_direct8(Converter_C &copy,const void *source,void *destination,unsigned count)
{
    // setup pointers
    const char8 *s = (const char8*) source;
    char8 *d = (char8*) destination;
    
    // setup lookup pointer
    char8 *lookup = (char8*) copy.table;

    // pixel loop
    while (count--)
    {
        // convert pixel
        *d = lookup[*s];
        
        // next
        s++;
        d++;
    }
}
