//
// JNI implementation of "ptc.jni.Console" for PTC 2.0 Java API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "ptc.h"
#include "jni.h"




// global console
static Console *console = 0;

// console functions
Console *lookup(JNIEnv *jni,jobject console);

// translation functions
Area    translate_area(JNIEnv *jni,jobject area);
Format  translate_format(JNIEnv *jni,jobject format);
jobject translate_format(JNIEnv *jni,const Format &format);

// type identification
int type(JNIEnv *jni,jobject object);

// java array locking functions
void*  lock(JNIEnv *jni,jobject array,jboolean &copy);
void unlock(JNIEnv *jni,jobject array,void *data,jboolean copy);

// exception handling functions
void jthrow(JNIEnv *jni,Error error = Error("unknown jni exception"));

// java type constants
enum { JUNKNOWN, JBYTEARRAY, JSHORTARRAY, JINTARRAY, JLONGARRAY };




JNIEXPORT void JNICALL Java_ptc_jni_Console_create(JNIEnv *jni,jobject object,jstring title,jint width,jint height,jobject format)
{
    try
    {
        // create console
        console = new Console;

        // open console
        Java_ptc_jni_Console_open(jni,object,title,width,height,format);
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_destroy(JNIEnv *jni,jobject object)
{
    try
    {
        // destroy console
        delete console;
        console = 0;
    }
    catch (...)
    {
        // no exceptions
    }
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_open(JNIEnv *jni,jobject object,jstring title,jint width,jint height,jobject format)
{
    try
    {
        // copy flag
        jboolean copy;

        // get title string
        const char *string = jni->GetStringUTFChars(title,&copy);

        // open console
        console->open(string,width,height,translate_format(jni,format));

        // free string
        if (copy) jni->ReleaseStringUTFChars(title,string);
    }
    catch (Error error)
    {
        // close console
        console->close();

        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // close console
        console->close();

        // unknown
        jthrow(jni);
    }
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_close(JNIEnv *jni,jobject object)
{
    try
    {
        // close console
        console->close();
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_update(JNIEnv *jni,jobject object)
{
    try
    {
        // update console
        console->update();
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT jboolean JNICALL Java_ptc_jni_Console_key(JNIEnv *jni,jobject object)
{
    try
    {
        // check for keypress
        return console->key();
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT jint JNICALL Java_ptc_jni_Console_read(JNIEnv *jni,jobject object)
{
    try
    {
        // read keypress
        return console->read();
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_copy__Lptc_base_Surface_2(JNIEnv *jni,jobject object,jobject surface)
{
    // copy console to surface
    jthrow(jni,Error("console to surface copy is not implemented"));
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_copy__Lptc_base_Surface_2Lptc_Area_2Lptc_Area_2(JNIEnv *jni,jobject object,jobject surface,jobject source,jobject destination)
{
    // copy console to surface
    jthrow(jni,Error("console to surface copy is not implemented"));
}


JNIEXPORT jobject JNICALL Java_ptc_jni_Console_lock(JNIEnv *jni,jobject object)
{
    // lock console
    jthrow(jni,Error("console lock is not supported"));

    // null pixels
    return (jobject)0;
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_unlock(JNIEnv *jni,jobject object)
{
    // unlock console
    jthrow(jni,Error("console unlock is not supported"));
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_load__Ljava_lang_Object_2IILptc_Format_2_3I(JNIEnv *jni,jobject object,jobject pixels,jint width,jint height,jobject format,jintArray palette)
{
    try
    {
        // copy flag
        jboolean copy;

        // lock java pixel array
        void *pixel_array = (char*) lock(jni,pixels,copy);

        // lock java palette array
        jint *palette_data = jni->GetIntArrayElements(palette,&copy);

        // convert palette
        int32 palette_array[256];
        for (int i=0; i<256; i++) palette_array[i] = (int32)palette_data[i];

        // write to console
        console->load(pixel_array,width,height,translate_format(jni,format),palette_array);
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_load__Ljava_lang_Object_2IILptc_Format_2_3ILptc_Area_2(JNIEnv *jni,jobject object,jobject pixels,jint width,jint height,jobject format,jintArray palette,jobject area)
{
    try
    {
        // copy flag
        jboolean copy;

        // lock java pixel array
        void *pixel_array = (char*) lock(jni,pixels,copy);

        // lock java palette array
        jint *palette_data = jni->GetIntArrayElements(palette,&copy);

        // convert palette
        int32 palette_array[256];
        for (int i=0; i<256; i++) palette_array[i] = (int32)palette_data[i];

        // write to console
        console->load(pixel_array,width,height,translate_format(jni,format),palette_array,translate_area(jni,area));
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_save__Ljava_lang_Object_2IILptc_Format_2_3I(JNIEnv *jni,jobject object,jobject pixels,jint width,jint height,jobject format,jintArray palette)
{
    // save console pixels
    jthrow(jni,Error("console save is not implemented"));
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_save__Ljava_lang_Object_2IILptc_Format_2_3ILptc_Area_2(JNIEnv *jni,jobject object,jobject pixels,jint width,jint height,jobject format,jintArray palette,jobject area)
{
    // save console area pixels
    jthrow(jni,Error("console area save is not implemented"));
}


JNIEXPORT void JNICALL Java_ptc_jni_Console_palette(JNIEnv *jni,jobject object,jintArray palette)
{
    try
    {
        // copy flag
        jboolean copy;

        // lock java array
        int32 *array = (int32*) jni->GetIntArrayElements(palette,&copy);

        // set palette
        console->palette(array);
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT jint JNICALL Java_ptc_jni_Console_width(JNIEnv *jni,jobject object)
{
    try
    {
        // get width
        return console->width();
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT jint JNICALL Java_ptc_jni_Console_height(JNIEnv *jni,jobject object)
{
    try
    {
        // get height
        return console->height();
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT jobject JNICALL Java_ptc_jni_Console_format(JNIEnv *jni,jobject object)
{
    try
    {
        // get format
        return translate_format(jni,console->format());
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}


JNIEXPORT jstring JNICALL Java_ptc_jni_Console_name(JNIEnv *jni,jobject object)
{
    try
    {
        // get name
        return jni->NewStringUTF(console->name());
    }
    catch (Error error)
    {
        // known error
        jthrow(jni,error);
    }
    catch (...)
    {
        // unknown
        jthrow(jni);
    }
}




Area translate_area(JNIEnv *jni,jobject area)
{
    // get ptc.Area class
    jclass cls = jni->GetObjectClass(area);

    // get method ids
    jmethodID area_left   = jni->GetMethodID(cls,"left","()I");
    jmethodID area_top    = jni->GetMethodID(cls,"top","()I");
    jmethodID area_right  = jni->GetMethodID(cls,"right","()I");
    jmethodID area_bottom = jni->GetMethodID(cls,"bottom","()I");

    // get area data
    int left   = (int) jni->CallIntMethod(area,area_left);
    int top    = (int) jni->CallIntMethod(area,area_top);
    int right  = (int) jni->CallIntMethod(area,area_right);
    int bottom = (int) jni->CallIntMethod(area,area_bottom);

	// create area object
	return Area(left,top,right,bottom);
}


Format translate_format(JNIEnv *jni,jobject format)
{
    // get ptc.Format class
    jclass cls = jni->GetObjectClass(format);

    // get method ids
    jmethodID format_r       = jni->GetMethodID(cls,"r","()I");
    jmethodID format_g       = jni->GetMethodID(cls,"g","()I");
    jmethodID format_b       = jni->GetMethodID(cls,"b","()I");
    jmethodID format_a       = jni->GetMethodID(cls,"a","()I");
    jmethodID format_bits    = jni->GetMethodID(cls,"bits","()I");
    jmethodID format_direct  = jni->GetMethodID(cls,"direct","()Z");
    jmethodID format_indexed = jni->GetMethodID(cls,"indexed","()Z");

    // get format data
    int r    = (int) jni->CallIntMethod(format,format_r);
    int g    = (int) jni->CallIntMethod(format,format_g);
    int b    = (int) jni->CallIntMethod(format,format_b);
    int a    = (int) jni->CallIntMethod(format,format_a);
    int bits = (int) jni->CallIntMethod(format,format_bits);
 
    // check format type
    if (jni->CallBooleanMethod(format,format_direct))
    {
        // direct color
        return Format(bits,r,g,b,a);
    }
    else 
    {
        // indexed color
        return Format(bits);
    }
}


jobject translate_format(JNIEnv *jni,const Format &format)
{
    // get ptc.Format class
    jclass cls = jni->FindClass("ptc/Format");

    // check format type
    if (format.direct())
    {
        // get constructor method id
        jmethodID method = jni->GetMethodID(cls,"<init>","(IIIII)V");

        // construct format object
        jobject object = jni->NewObject(cls,method,(jint)format.bits(),(jint)format.r(),(jint)format.g(),(jint)format.b(),(jint)format.a());

        // success
        return object;
    }
    else
    {
        // get constructor method id
        jmethodID method = jni->GetMethodID(cls,"<init>","(I)V");

        // construct format object
        jobject object = jni->NewObject(cls,method,(jint)format.bits());

        // success
        return object;
    }
}




int type(JNIEnv *jni,jobject object)
{
    // get java array classes
    jclass jbyte_class  = jni->FindClass("[B");
    jclass jshort_class = jni->FindClass("[S");
    jclass jint_class   = jni->FindClass("[I");
    jclass jlong_class  = jni->FindClass("[J");
    
    // return array type id
    if (jni->IsInstanceOf(object,jbyte_class))  return JBYTEARRAY;
    if (jni->IsInstanceOf(object,jshort_class)) return JSHORTARRAY;
    if (jni->IsInstanceOf(object,jint_class))   return JINTARRAY;
    if (jni->IsInstanceOf(object,jlong_class))  return JLONGARRAY;

    // failure
    return JUNKNOWN;
}




void* lock(JNIEnv *jni,jobject array,jboolean &copy)
{
    // get array type id
    int id = type(jni,array);
    
    // data pointer
    void *data = 0;

    // lock array
    switch (id)
    {
        case JBYTEARRAY:  data = jni->GetByteArrayElements((jbyteArray)array,&copy);   break;
        case JSHORTARRAY: data = jni->GetShortArrayElements((jshortArray)array,&copy); break;
        case JINTARRAY:   data = jni->GetIntArrayElements((jintArray)array,&copy);     break;
        case JLONGARRAY:  data = jni->GetLongArrayElements((jlongArray)array,&copy);   break;
        default:          throw Error("invalid java array type");
    }
    
    // success
    return data;
}


void unlock(JNIEnv *jni,jobject array,void *data,jboolean copy)
{
    if (copy)
    {
        // get array type id
        int id = type(jni,array);
    
        // release java array
        switch (id)
        {
            case JBYTEARRAY:  jni->ReleaseByteArrayElements((jbyteArray)array,(jbyte*)data,0);    break;
            case JSHORTARRAY: jni->ReleaseShortArrayElements((jshortArray)array,(jshort*)data,0); break;
            case JINTARRAY:   jni->ReleaseIntArrayElements((jintArray)array,(jint*)data,0);       break;
            case JLONGARRAY:  jni->ReleaseLongArrayElements((jlongArray)array,(jlong*)data,0);    break;
        }
    }
}




void jthrow(JNIEnv *jni,Error error)
{
    // get ptc.Error class
    jclass cls = jni->FindClass("ptc/Error");

    // get constructor method id
    jmethodID method = jni->GetMethodID(cls,"<init>","(Ljava/lang/String;)V");

    // create error string
    jstring string = jni->NewStringUTF(error.message());

    // construct exception object
    jthrowable object = (jthrowable) jni->NewObject(cls,method,string);

    // throw error
    jni->Throw(object);
}
