//
// Link test for OpenPTC 1.0 C++ Implementation
// Copyright (c) Glenn Fiedler (ptc@gaffer.org)
// This source code is in the public domain
//

// include files
#include "ptc.h"
#include <stdio.h>


void TestDefines()
{
    // check ptc signature define
    #ifndef __PTC__
    #error you need to add the ptc signature define __PTC__ in your ptc.h
    #endif
    
    // check platform signature define
    #ifdef __PTC_PORTING__
    #error you need to change the ptc platform signature define in ptc.h
    #endif

    // check version define
    #ifndef __PTC_VERSION__
    #error you do not have a ptc version define in ptc.h
    #endif

    // print version define
    printf(__PTC_VERSION__);
}


void TestTypes()
{
    // test integral types
    char8 test_char8 = 0;
    short16 test_short16 = 0;
    int32 test_int32 = 0;
}


void TestBaseSurface(BaseSurface &surface)
{
    // create other surface
    Surface other(100,100,Format(32,0x00FF0000,0x0000FF00,0x000000FF));

    // copy surface to another surface
    surface.copy(other);

    // copy surface area to other area
    surface.copy(other,Area(0,0,10,10),Area(10,10,20,20));

    // lock surface
    void *data = surface.lock();

    // unlock surface
    surface.unlock();

    // load pixels to surface
    surface.load(0,10,10,40,Format(32,0x00FF0000,0x0000FF00,0x000000FF),Palette());

    // load pixels to console area
    surface.load(0,10,10,40,Format(32,0x00FF0000,0x0000FF00,0x000000FF),Palette(),Area(0,0,10,10),Area(10,10,20,20));

    // save console pixels
    surface.save(0,10,10,40,Format(32,0x00FF0000,0x0000FF00,0x000000FF),Palette());

    // save console area pixels
    surface.save(0,10,10,40,Format(32,0x00FF0000,0x0000FF00,0x000000FF),Palette(),Area(0,0,10,10),Area(10,10,20,20));

    // clear surface
    surface.clear();
    
    // clear surface to color
    surface.clear(Color(0,0,1));

    // clear surface area to color
    surface.clear(Color(0,0,1),Area(0,0,10,10));

    // set surface palette
    surface.palette(Palette());

    // get surface palette
    const Palette &palette = surface.palette();

    // set surface clip area
    surface.clip(Area(0,0,10,10));
    
    // get surface width
    int width = surface.width();

    // get surface height
    int height = surface.height();

    // get surface area
    Area area = surface.area();
    
    // get surface clip area
    Area clip = surface.clip();

    // get surface format
    const Format &format = surface.format();

    // pass surface option string
    bool success = surface.option("all those moments will be lost in time like tears in the rain");
}


void TestBaseConsole(BaseConsole &console)
{
    // configure the console
    console.configure("fake.cfg");

    // get console modes
    const Mode *modes = console.modes();

    // open the console
    console.open("Title");
    console.open("Title",3);
    console.open("Title",Format(32,0x00FF0000,0x0000FF00,0x000000FF));
    console.open("Title",Format(32,0x00FF0000,0x0000FF00,0x000000FF),3);
    console.open("Title",100,100,Format(32,0x00FF0000,0x0000FF00,0x000000FF));
    console.open("Title",100,100,Format(32,0x00FF0000,0x0000FF00,0x000000FF),3);
    console.open("Title",Mode(100,100,Format()));
    console.open("Title",Mode(100,100,Format()),3);

    // close the console
    console.close();

    // flush console
    console.flush();

    // finish console
    console.finish();

    // update console
    console.update();

    // update console area
    console.update(Area(0,0,10,10));

    // check if a key has been pressed
    bool keypressed = console.key();

    // read a key press
    Key key = console.read();

    // get the number of pages
    int pages = console.pages();

    // get the console title
    const char *title = console.title();

    // get the console name
    const char *name = console.name();

    // get the console information
    const char *information = console.information();
}


void TestConsole()
{
    // create console
    Console console;

    // test base surface
    TestBaseSurface(console);

    // test base console
    TestBaseConsole(console);
}


void TestSurface()
{
    // create surface
    Surface surface(100,100,Format(32,0x00FF0000,0x0000FF00,0x000000FF));

    // test base surface
    TestBaseSurface(surface);
}


void TestPalette()
{
    // default constructor
    Palette a;

    // data constructor
    int32 data[256];
    Palette b(data);

    // copy constructor
    Palette palette(b);

    // lock palette
    int32 *pal = palette.lock();

    // unlock palette
    palette.unlock();

    // load palette data
    palette.load(data);

    // save palette data
    palette.save(data);

    // get palette data
    const int32 *const_pal = palette.data();

    // assignment operator
    palette = a;

    // is equal to
    if (palette==a) return;

    // is not equal to
    if (palette!=b) return;
}


void TestFormat()
{
    // default constructor
    Format defaults;

    // indexed constructor
    Format indexed(8);

    // direct constructor
    Format direct(32,0x00FF0000,0x0000FF00,0x000000FF);

    // copy constructor
    Format format(direct);

    // get red mask
    int32 r = format.r();

    // get green mask
    int32 g = format.g();

    // get blue mask
    int32 b = format.b();

    // get alpha mask
    int32 a = format.a();

    // get bits per pixel
    int32 bits = format.bits();

    // get bytes per pixel
    int32 bytes = format.bytes();

    // get direct color flag
    bool is_direct = format.direct();

    // get indexed color flag
    bool is_indexed = format.indexed();

    // assignment operator
    format = indexed;

    // is equal to 
    if (format==indexed) return;

    // not equal to
    if (format!=direct) return;
}


void TestColor()
{
    // default constructor
    Color defaults;

    // indexed constructor
    Color indexed(100);

    // direct constructor
    Color direct(0.2f,0.6f,1.0f);

    // copy constructor
    Color color(direct);

    // get index
    int index = color.index();

    // get red component
    float r = color.r();

    // get green component
    float g = color.g();

    // get blue component
    float b = color.b();

    // get alpha component
    float a = color.a();

    // get direct color flag
    bool is_direct = color.direct();

    // get indexed color flag
    bool is_indexed = color.indexed();

    // assignment operator
    color = indexed;

    // is equal to
    if (color==indexed) return;

    // not equal to
    if (color!=direct) return;
}


void TestError()
{
    // default constructor
    Error defaults;

    // message constructor
    Error message("this is the original error message");

    // cascading error constructor
    Error cascading("this is a cascading error message",message); 

    // copy constructor
    Error error(cascading);

    try
    {
        // throw error
        throw error;
    }
    catch (Error &error)
    {
        // get error message
        const char *message = error.message();

        // report error
        error.report();
    }
    
    // assignment operator
    error = message;

    // is equal to
    if (error==cascading) return;

    // is not equal to
    if (error!=message) return;
}


void TestTimer()
{
    // default constructor
    Timer defaults;

    // set time constructor
    Timer time(2.0);

    // copy constructor
    Timer timer = defaults;

    // start timer
    timer.start();

    // stop timer
    timer.stop();

    // get time
    double t = timer.time();

    // get delta
    double delta = timer.delta();

    // get timer resolution
    double resolution = timer.resolution();

    // assignment operator
    timer = defaults;

    // is equal to
    if (timer==time) return;

    // not equal to 
    if (timer!=defaults) return;
}


void TestArea()
{
    // default constructor
    Area defaults;

    // setup constructor
    Area setup(0,0,100,100);

    // copy constructor
    Area area(setup);

    // get area left
    int left = area.left();

    // get area top
    int top = area.top();

    // get area right
    int right = area.right();

    // get area bottom
    int bottom = area.bottom();

    // get area width
    int width = area.width();

    // get area height
    int height = area.height();

    // assignment operator
    area = defaults;

    // is equal to
    if (area==setup) return;

    // not equal to
    if (area!=defaults) return;
}


void TestMode()
{
    // default constructor
    Mode defaults;

    // setup constructor
    Mode setup(640,480,Format(32,0x00FF0000,0x0000FF00,0x000000FF));

    // copy constructor
    Mode mode(setup);

    // get mode width
    int width = mode.width();

    // get mode height
    int height = mode.height();

    // get mode format
    Format format = mode.format();

    // is mode valid?
    bool valid = mode.valid();

    // assignment operator
    mode = defaults;

    // is equal to
    if (mode==setup) return;

    // not equal to
    if (mode!=defaults) return;
}


void TestClip()
{
    // setup areas
    Area a(10,10,20,20);
    Area b(20,20,50,100);
    Area c(0,0,30,20);
    Area d(0,0,100,100);
    Area e;
    Area f;
    
    // clip a single area
    a = Clipper::clip(b,c);
    
    // clip source and destination areas
    Clipper::clip(a,b,c,d,e,f);
}


void TestClear()
{
    // create clear
    Clear clear;

    // request clear
    clear.request(Format());

    // perform clear
    clear.clear(0,0,0,10,10,10,Color());
}


void TestCopy()
{
    // create copy
    Copy copy;

    // request copy
    copy.request(Format(),Format());

    // set copy palettes
    copy.palette(Palette(),Palette());

    // perform copy
    copy.copy(0,0,0,10,10,10,0,0,0,10,10,10);

    // set copy option
    copy.option("some copy option");
}


void TestKey()
{
    // default constructor
    Key defaults;

    // code constructor
    Key keycode(Key::ENTER);

    // modifiers constructor
    Key modifiers(Key::ENTER,true,true,true);

    // copy constructor
    Key key(modifiers);

    // get key code
    int code = key.code();

    // get alt modifier
    bool alt = key.alt();

    // get shift modifier
    bool shift = key.shift();

    // get control modifier
    bool control = key.control();

    // assignment operator
    key = defaults;

    // is equal to
    if (key==keycode) return;

    // not equal to
    if (key!=defaults) return;

    // test all key codes exist
    code = Key::ENTER;
    code = Key::BACKSPACE;
    code = Key::TAB;
    code = Key::CANCEL;
    code = Key::CLEAR;
    code = Key::SHIFT;
    code = Key::CONTROL;
    code = Key::ALT;
    code = Key::PAUSE;
    code = Key::CAPSLOCK;
    code = Key::ESCAPE;
    code = Key::SPACE;
    code = Key::PAGEUP;
    code = Key::PAGEDOWN;
    code = Key::END;
    code = Key::HOME;
    code = Key::LEFT;
    code = Key::UP;
    code = Key::RIGHT;
    code = Key::DOWN;
    code = Key::COMMA;
    code = Key::PERIOD;
    code = Key::SLASH;
    code = Key::ZERO;
    code = Key::ONE;
    code = Key::TWO;
    code = Key::THREE;
    code = Key::FOUR;
    code = Key::FIVE;
    code = Key::SIX;
    code = Key::SEVEN;
    code = Key::EIGHT;
    code = Key::NINE;
    code = Key::SEMICOLON;
    code = Key::EQUALS;
    code = Key::A;
    code = Key::B;
    code = Key::C;
    code = Key::D;
    code = Key::E;
    code = Key::F;
    code = Key::G;
    code = Key::H;
    code = Key::I;
    code = Key::J;
    code = Key::K;
    code = Key::L;
    code = Key::M;
    code = Key::N;
    code = Key::O;
    code = Key::P;
    code = Key::Q;
    code = Key::R;
    code = Key::S;
    code = Key::T;
    code = Key::U;
    code = Key::V;
    code = Key::W;
    code = Key::X;
    code = Key::Y;
    code = Key::Z;
    code = Key::OPENBRACKET;
    code = Key::BACKSLASH;
    code = Key::CLOSEBRACKET;
    code = Key::NUMPAD0;
    code = Key::NUMPAD1;
    code = Key::NUMPAD2;
    code = Key::NUMPAD3;
    code = Key::NUMPAD4;
    code = Key::NUMPAD5;
    code = Key::NUMPAD6;
    code = Key::NUMPAD7;
    code = Key::NUMPAD8;
    code = Key::NUMPAD9;
    code = Key::MULTIPLY;
    code = Key::ADD;
    code = Key::SEPARATOR;
    code = Key::SUBTRACT;
    code = Key::DECIMAL;
    code = Key::DIVIDE;
    code = Key::F1;
    code = Key::F2;
    code = Key::F3;
    code = Key::F4;
    code = Key::F5;
    code = Key::F6;
    code = Key::F7;
    code = Key::F8;
    code = Key::F9;
    code = Key::F10;
    code = Key::F11;
    code = Key::F12;
    code = Key::DELETE;
    code = Key::NUMLOCK;
    code = Key::SCROLLLOCK;
    code = Key::PRINTSCREEN;
    code = Key::INSERT;
    code = Key::HELP;
    code = Key::META;
    code = Key::BACKQUOTE;
    code = Key::QUOTE;
    code = Key::FINAL;
    code = Key::CONVERT;
    code = Key::NONCONVERT;
    code = Key::ACCEPT;
    code = Key::MODECHANGE;
    code = Key::KANA;
    code = Key::KANJI;
    code = Key::UNDEFINED;
}


void main()
{
    // this program does nothing but test linkage with OpenPTC
    printf("congratulations you passed the link test\n");
}
