//
// Save example for OpenPTC 1.0 Java Implementation
// Copyright (c) Glenn Fiedler (ptc@gaffer.org)
// This source code is in the public domain
//

// import classes
import ptc.Area;
import ptc.Error;
import ptc.Format;
import ptc.Surface;
import ptc.Console;



class Save
{
    public static void main(String argv[])
    {
        try
        {
            // create console
            Console console = new Console();
        
            // create format
            Format format = new Format(32,0x00FF0000,0x0000FF00,0x000000FF);
 
            // open the console with a single page
            console.open("Save example",format,1);

            // create surface matching console dimensions
            Surface surface = new Surface(console.width(),console.height(),format);

            // setup viewing area
            final float x1 = -2.00f;
            final float y1 = -1.25f;
            final float x2 = +1.00f;
            final float y2 = +1.25f;

            // render the mandelbrot fractal
            mandelbrot(console,surface,x1,y1,x2,y2);

            // save mandelbrot image
            save(surface,"Save.jpg");

            // read key
            console.read();

            // close console
            console.close();
        }
        catch (Error error)
        {
            // report error
            error.report();
        }
        
        // exit program
        System.exit(0);
    }

    static void mandelbrot(Console console,Surface surface,float x1,float y1,float x2,float y2) throws Error
    {
        // constant values
        final int entries = 1024;
        final int maximum = 1024;

        // fractal color table
        int table[] = new int[entries];

        // generate fractal color table
        for (int i=0; i<entries; i++)
        {
            // useful pi constant
            final float pi = 3.1415926f;

            // calculate normalized index
            final float index = i / (float) entries;

            // calculate sine curve time value
            final float time = index * pi - pi/2;

            // lookup sine curve intensity at time and scale to [0,1]
            float intensity = (float) ( Math.sin(time) + 1.0f ) / 2.0f;

            // raise the intensity to a power
            intensity = (float) Math.pow(intensity,0.1);

            // store intensity as a shade of blue
            table[i] = (int) ( 255 * intensity );
        }

        // lock surface pixels
        int pixels[] = (int[]) surface.lock();

        // get surface dimensions
        final int width  = surface.width();
        final int height = surface.height();

        // pixel index
        int pixel = 0;

        // calculate real x,y deltas
        final float dx = (x2 - x1) / width;
        final float dy = (y2 - y1) / height;

        // imaginary axis
        float imaginary = y1;

        // iterate down surface y
        for (int y=0; y<height; y++)
        {
            // real axis
            float real = x1;

            // iterate across surface x
            for (int x=0; x<width; x++)
            {
                // calculate the mandelbrot interation count
                final int count = calculate(real,imaginary,maximum);

                // calculate color table index
                final int index = count % entries;

                // lookup color from iteration
                final int color = table[index];

                // store color
                pixels[pixel] = color;

                // next pixel
                pixel ++;

                // update real
                real += dx;
            }

            // update imaginary
            imaginary += dy;

            // setup line area
            Area area = new Area(0,y,width,y+1);

            // copy surface area to console
            surface.copy(console,area,area);

            // update console area
            console.update(area);
        }

        // unlock surface
        surface.unlock();
    }

    static int calculate(float real,float imaginary,int maximum)
    {
        // complex number 'c'
        final float c_r = real;
        final float c_i = imaginary;

        // complex 'z'
        float z_r = 0;
        float z_i = 0;
    
        // complex 'z' squares
        float z_r_squared = 0;
        float z_i_squared = 0;

        // mandelbrot function iteration loop
        for (int count=0; count<maximum; count++)
        {
            // square 'z' and add 'c'
            z_i = 2 * z_r * z_i + c_i;
            z_r = z_r_squared - z_i_squared + c_r;

            // update 'z' squares
            z_r_squared = z_r * z_r;
            z_i_squared = z_i * z_i;

            // calculate squared magnitude of complex 'z'
            final float z_squared_magnitude = z_r_squared + z_i_squared;

            // stop iterating if the magnitude of 'z' is greater than two
            if (z_squared_magnitude>=4) return count;
        }

        // maximum
        return 0;
    }

    static void save(Surface surface,String filename) throws Error
    {
        // todo: save surface image
    }
}
