//
// Fire demo for OpenPTC 1.0 Java Implementation
// Copyright (c) Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU GPL
// See http://www.gnu.org/copyleft/gpl.html for details
//

// import classes
import ptc.Area;
import ptc.Error;
import ptc.Format;
import ptc.Palette;
import ptc.Surface;


public class Fire extends ptc.java.Applet
{
    public static void main(String argv[])
    {
        try
        {
            // create console
            ptc.Console console = new ptc.Console();
        
            // create format
            Format format = new Format(8);
 
            // open the console
            console.open("Fire demo",format);

            // create fire
            Fire fire = new Fire();

            // run fire demo
            fire.run(console);

            // close console
            console.close();
        }
        catch (Error error)
        {
            // report error
            error.report();
        }

        // exit program
        System.exit(0);
    }

    public ptc.base.Console create() throws Error
    {
        // create format
        Format format = new Format(8);

        // create console
        ptc.java.Console console = new ptc.java.Console();

        // open console on applet
        console.open(this,format);

        // return console
        return console;
    }

    public void run(ptc.base.Console console) throws Error
    {
        // create format
        Format format = new Format(8);

        // create surface
        Surface surface = new Surface(320,208,format);

        // create palette
        Palette palette = new Palette();

        // generate palette
        generate(palette);

        // set console palette
        console.palette(palette);

        // set surface palette
        surface.palette(palette);

        // flame data
        int state = 0;
        float intensity = 0;

        // main loop
        while (true)
        {
            // lower flame on keypress
            if (console.key()) state = 2;

            // state machine
            switch (state)
            {
                case 0:
                {
                    // raise flame
                    intensity += 0.007f;
            
                    // maximum flame height
                    if (intensity>0.8f) state = 1;
                }
                break;

                case 1:
                {
                    // constant flame
                }
                break;

                case 2:
                {
                    // lower flame
                    intensity -= 0.005f;

                    // exit program when flame is out
                    if (intensity<0.01f) return;
                }
                break;
            }

            // lock surface pixels
            byte pixels[] = (byte[]) surface.lock();

            // get surface dimensions
            final int width  = surface.width();
            final int height = surface.height();

            // setup flame buffer pitch
            final int pitch = width << 1;

            // flame vertical loop
            for (int y=1; y<height-4; y+=2)
            {
                // setup current line index
                final int line = y * width;

                // flame horizontal loop
                for (int x=0; x<width; x++)
                {
                    // setup pixel out index
                    final int out = line + x;

                    // setup pixel in index
                    final int in = out + pitch;

                    // read in pixels
                    int l = pixels[in-1];
                    int t = pixels[in];
                    int r = pixels[in+1];
                    int b = pixels[in+pitch];

                    // adjust signed values
                    l &= 0xFF;
                    t &= 0xFF;
                    r &= 0xFF;
                    b &= 0xFF;
                    
                    // setup top sum
                    final int top = l + t + r;

                    // setup bottom sum
                    final int bottom = b;

                    // combine top and bottom
                    int combined = (top + bottom) >> 2;

                    // cool down intensity
                    if (combined>0) combined--;

                    // interpolate intensity between top and bottom
                    final int interpolated = (combined + bottom) >> 1;

                    // store pixels
                    pixels[out] = (byte) combined;
                    pixels[out+width] = (byte) interpolated;
                }
            }

            // setup flame generator start index
            int generator = width*(height-4);

            // update flame generator bar
            for (int x=0; x<width; x+=4)
            {
                // random block color taking intensity into account
                final byte color = (byte) random((int)(255.0f*intensity));

                // write 4x4 color blocks
                pixels[generator+0]         = color;
                pixels[generator+1]         = color;
                pixels[generator+2]         = color;
                pixels[generator+3]         = color;
                pixels[generator+width+0]   = color;
                pixels[generator+width+1]   = color;
                pixels[generator+width+2]   = color;
                pixels[generator+width+3]   = color;
                pixels[generator+width*2+0] = color;
                pixels[generator+width*2+1] = color;
                pixels[generator+width*2+2] = color;
                pixels[generator+width*2+3] = color;
                pixels[generator+width*3+0] = color;
                pixels[generator+width*3+1] = color;
                pixels[generator+width*3+2] = color;
                pixels[generator+width*3+3] = color;

                // next block
                generator += 4;
            }

            // unlock surface
            surface.unlock();

            // setup surface copy area
            Area area = new Area(0,1,320,201);

            // copy surface to console
            surface.copy(console,area,console.area());

            // update console
            console.update();
        }
    }

    static int random(int number)
    {
        // random number routine
        return (int) ( Math.random() * number );
    }

    static int pack(int r,int g,int b)
    {
        // pack color integer
        return (r<<16) | (g<<8) | b;
    }

    static void generate(Palette palette) throws Error
    {
        // lock palette data
        int data[] = palette.lock();
    
        // black to red
        int i=0;
        int c=0;
        while (i<64)
        {
            data[i] = pack(c,0,0);
            c+=4;
            i++;
        }

        // red to yellow
        c=0;
        while (i<128)
        {
            data[i] = pack(255,c,0);
            c+=4;
            i++;
        }

        // yellow to white
        c=0;
        while (i<128)
        {
            data[i] = pack(255,255,c);
            c+=4;
            i++;
        }

        // white
        while (i<256) 
        {
            data[i] = pack(255,255,255);
            i++;
        }

        // unlock palette
        palette.unlock();
    }
}
