//
// Tiny Console class for OpenPTC 1.0 Java Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// package name
package tinyptc;

// import classes
import java.awt.Image;
import java.awt.Graphics;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.image.ImageProducer;
import java.awt.image.ImageConsumer;
import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;
import java.awt.image.DirectColorModel;



public class Console extends Component implements ImageProducer
{
    public void open(Component component,Format format)
    {
        // get component size
        Dimension size = component.size();

        // setup console data
        this.width = size.width;
        this.height = size.height;
        this.format = format;
        this.component = component;

        // setup color model
        this.model = model(format,new Palette());

        // create image
        image = createImage(this);
    }

    public void update()
    {
        // paint
        paint();
    }

    public synchronized void load(Object pixels,int width,int height,int pitch,Format format,Palette palette)
    {
        // check consumer reference
        if (consumer==null) return;

        // check bits
        switch (format.bits())
        {
            case 32:
            {
                // copy integer pixel data to image consumer
                consumer.setPixels(0,0,width,height,this.model(format,palette),(int[])pixels,0,pitch/4);
            }
            break;

            case 8:
            {
                // copy byte pixel data to image consumer
                consumer.setPixels(0,0,width,height,model,(byte[])pixels,0,pitch);
            }
            break;
        }

        // notify image consumer that the frame is done
        consumer.imageComplete(ImageConsumer.SINGLEFRAMEDONE);
    }

    public void palette(Palette palette)
    {
        // update color model palette
        this.model = model(format,palette);

        // check consumer reference
        if (consumer==null) return;

        // set consumer palette model
        consumer.setColorModel(model); 
    }

    public int width()
    {
        // get width
        return width;
    }

    public int height()
    {
        // get height
        return height;
    }

    public Format format()
    {
        // get format
        return format;
    }

    public synchronized void addConsumer(ImageConsumer ic) 
    {
        // register image consumer
        consumer = ic;

        // set image dimensions
        consumer.setDimensions(width,height);

        // set image consumer hints for speed
        consumer.setHints(ImageConsumer.TOPDOWNLEFTRIGHT|ImageConsumer.COMPLETESCANLINES|ImageConsumer.SINGLEPASS|ImageConsumer.SINGLEFRAME);

        // set image color model
        consumer.setColorModel(model);
    }

    public synchronized boolean isConsumer(ImageConsumer ic) 
    {
        // check if consumer is registered
        return true;
    }

    public synchronized void removeConsumer(ImageConsumer ic) 
    {
        // remove image consumer
    }

    public void startProduction(ImageConsumer ic) 
    {
        // add consumer
     addConsumer(ic);
    }

    public void requestTopDownLeftRightResend(ImageConsumer ic) 
    {
        // ignore resend request
    }

    private synchronized void paint()
    {
        // get component graphics object
        Graphics graphics = component.getGraphics();

        // draw image to graphics context
        graphics.drawImage(image,0,0,width,height,null);
    }

    private ColorModel model(Format format,Palette palette)
    {
        // check format
        if (format.direct())
        {
            // create direct color model
            return new DirectColorModel(format.bits(),format.r(),format.g(),format.b(),format.a());
        }
        else
        {
            // get palette data array
            int data[] = palette.data();

            // create color arrays
            byte r[] = new byte[256];
            byte g[] = new byte[256];
            byte b[] = new byte[256];

            // setup color byte arrays
            for (int i=0; i<256; i++)
            {
                // unpack color integer to a,r,g,b
                r[i] = (byte) ( (data[i] & 0x00FF0000) >> 16 );
                g[i] = (byte) ( (data[i] & 0x0000FF00) >> 8  );
                b[i] = (byte) ( (data[i] & 0x000000FF)       );
            }

            // create indexed color model
            return new IndexColorModel(format.bits(),256,r,g,b);
        }
    }

    // console data
    private int width;
    private int height;
    private Image image;
    private Format format;
    private ColorModel model;
    private Component component;
    private ImageConsumer consumer;
}
