//
// Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

#ifndef __PTC_CONSOLE_H
#define __PTC_CONSOLE_H

// include files
#include <ptc.h>
#include <stdlib.h>
#include <string.h>
#include <fstream.h>
#include <ggi/ggi.h>
#include <ggi/gii.h>

#define BASE_FLAG 1
#define EXPLICIT_MODE ( BASE_FLAG << 0 )

class Console : public BaseConsole
{
    public:

        // setup
        Console();
        ~Console();

        // console configuration
        virtual void configure(const char file[]);

        // console option string
        virtual bool option(const char *str);

        // console modes
        virtual const Mode* modes();

        // console management
        virtual void open(const char title[],int pages = 0);
        virtual void open(const char title[],const Format &format,int pages = 0);
        virtual void open(const char title[],int width,int height,const Format &format,int pages = 0);
        virtual void open(const char title[],const Mode &mode,int pages = 0);
        virtual void close();

        // synchronization
        virtual void flush();
        virtual void finish();
        virtual void update();
        virtual void update(const Area &area);

        // keyboard input
        virtual bool key();
        virtual Key read();

        // copy to surface
        virtual void copy(BaseSurface &surface);
        virtual void copy(BaseSurface &surface,const Area &source,const Area &destination);

        // memory access
        virtual void* lock();
        virtual void unlock();

        // load pixels to console
        virtual void load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette);
        virtual void load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination);

        // save console pixels
        virtual void save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette);
        virtual void save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination);

        // clear console
        virtual void clear();
        virtual void clear(const Color &color);
        virtual void clear(const Color &color,const Area &area);

        // console palette
        virtual void palette(const Palette &palette);
        virtual const Palette& palette() const;

        // console clip area
        virtual void clip(const Area &area);

        // data access
        virtual int width() const;
        virtual int height() const;
        virtual int pages() const;
        virtual int pitch() const;
        virtual const Area& area() const;
        virtual const Area& clip() const;
        virtual const Format& format() const;
        virtual const char* name() const;
        virtual const char* title() const;
        virtual const char* information() const;

        // GGI Specific

        // return a point to the GGI visual
        ggi_visual_t get_visual();

    private:
        // translate between GGI modes and PTC modes
        ggi_mode *translate( const Mode &l_ptc_mode, int pages = 0 );
        const Mode &translate( const ggi_mode &l_ggi_mode );

        // disallow copy construction
        Console(const Console &console);

        // disallow console assignment
        Console& operator =(const Console &console);

        // generic set_string
        void set_string( const char *source, char **dest );
         
        // console debug checks
        const bool check_open() const;
        const bool check_unlocked() const;

        // data
        int m_width;
        int m_height;
        int m_pages;
        int m_pitch;
        Area m_area;
        Area m_clip;
        Format m_format;

        // GGI Data
        ggi_visual_t                m_ggi_visual;
        char                       *m_ggi_visual_text;
        const ggi_directbuffer     *m_ggi_db;           // GGI pixel buffer

        // array data
        Mode                        m_modes;
        char                       *m_title;

        // flags
        int                         m_open;
        int                         m_locked;
        unsigned int                m_flags;

        // helper objects
        Copy                        m_copy;
        Clear                       m_clear;
        Palette                     m_palette;
        Mode                        m_mode;
};

#endif
