//
// Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//
// GGI implementation by giles francis hall <ghall@csh.rit.edu>
// This port is brought to you by Morcheeba and the letter 't'
// Copyright (c) 1999 giles francis hall
//


// include files
#include "Console.h"

// TODO: migrate this out into config.h.in
#define PTC_CONFIG_FILE "/usr/share/ptc/ptc.conf"

Console::Console()
{
    // defaults
    m_width  = 0;
    m_height = 0;
    m_pages  = 0;
    m_pitch  = 0;
    m_open   = 0;
    m_locked = 0;
    m_flags  = 0;

    // GGI specific stuff
    m_ggi_db = 0;
    m_ggi_visual = 0;
    m_ggi_visual_text = 0;
    
    // clear strings
    m_title = 0;

    // local variables
    char *home_env = getenv( "HOME" );

    configure( PTC_CONFIG_FILE );
    
    if( home_env && strlen( home_env ) < 255 ) {
        char fullpath[256];
        
        strcpy( fullpath, home_env );
        strcat( fullpath, "/.ptc.conf" );
        configure( fullpath );
    }

}


Console::~Console()
{
   close();
}

void Console::configure( const char file[] )
{
    // open configuration file
    ifstream f( file, ios::in );
    
    // check file
    if( !f ) {
        return;
    }

    // read option strings
    while( !f.eof() )
    {
        // option line
        char line[1024];

        // read line from file
        f.getline( line, 1024 );

        // process option
        option( line );
    }
}

bool Console::option( const char *str )
{
    if( !strcmp( str, "explicit mode" ) ){
        m_flags |= EXPLICIT_MODE;
        return true;
    }

    // last resort
    return m_copy.option( str );
}

const Mode* Console::modes()
{
    // return modes
    return &m_modes;
}

void Console::open(const char title[],int pages)
{
    // open console
    open( title, Format( 32, 0xff0000, 0xff00, 0xff ), pages );
}

void Console::open(const char title[],const Format &format,int pages)
{
    // default width and height
    int width  = 640;
    int height = 480;

    // open console
    open( title, width, height, format, pages );
}

void Console::open(const char title[],int width,int height,const Format &format,int pages)
{
    // open console
    open( title, Mode( width, height, format ), pages );
}

void Console::open(const char title[],const Mode &mode,int pages)
{
    ggi_mode                *l_ggi_mode;

    // check that the mode is valid
    if( !mode.valid() ){
        throw Error( "invalid mode" );
    }

    // sanity close
    close();

    // check pages
    if (pages==0)
    {
        // default
        pages = 1;
    }

    try
    {
        // Fire up the libraries ...
        if( ggiInit() < 0 ){
            throw Error( "ggiInit() failed!" );
        }
        m_open++;

        m_ggi_visual = ggiOpen( m_ggi_visual_text, NULL );
        if ( !m_ggi_visual ) {
            throw Error("Can't open GGI visual");
        }

        // TODO: this should be an option
        ggiAddFlags( m_ggi_visual, GGIFLAG_ASYNC );

        // get GGI friendly bit depth
        l_ggi_mode = translate( mode, pages );

        if( ggiCheckMode( m_ggi_visual, l_ggi_mode ) != 0 ){
            if( m_flags & EXPLICIT_MODE ){
                throw Error("Could not set desired mode!");
            }
        }

        ggiSetMode( m_ggi_visual, l_ggi_mode );

        m_mode = translate( *l_ggi_mode );
        
        // must suck the pages out by hand
        pages = l_ggi_mode->frames; 
        
        // configure backing plane
        // TODO: BE SMART ABOUT PAGES!
        m_ggi_db = ggiDBGetBuffer( m_ggi_visual, 0 );

        if( !m_ggi_db ){
            throw Error( "could not allocate a direct buffer!" );
        }
    }

    catch (Error &error)
    {
        // close
        close();

        // error message
        throw Error( "could not open GGI library", error );
    }

    // clear internal palette
    palette( Palette() );

    // setup console data
    m_width  = m_mode.width();
    m_height = m_mode.height();
    m_format = m_mode.format();
    m_pitch  = m_mode.width() * m_mode.format().bytes();
    m_pages  = pages;

    // setup console area
    m_area = Area( 0, 0, width(), height() );

    // setup clip area
    m_clip = m_area;

    // check length of console title
    set_string( title, &m_title );
   
    // free up useless GGI mode
    delete l_ggi_mode;
}

void Console::close()
{
    // check if open
    if( check_open() )
    {
        // the console must be unlocked when closed
        if( check_unlocked() ){
            throw Error("console is still locked");
        }

        if( m_ggi_visual ){

            // flush all key presses
            while( key() ){
                read();
            }

            if( m_ggi_visual ){
                if( ggiClose( m_ggi_visual ) < 0 ){
                    throw Error( "call to ggiClose failed" );
                }
                m_ggi_visual = 0;
            }
        }

        if( ggiExit() < 0 ){
            throw Error( "ggiExit failed!" );
        }
        m_open--;
                    
        // defaults
        m_width  = 0;
        m_height = 0;
        m_pages  = 0;
        m_pitch  = 0;
        m_open   = 0;
        m_locked = 0;
        m_flags  = 0;

        // GGI specific stuff -- MEMORY LEAK???
        m_ggi_db = 0;
        m_ggi_visual = 0;
        m_ggi_visual_text = 0;
            
        // clear strings
        m_title = 0;

        // clear areas
        m_area = Area();
        m_clip = Area();
    }
}

void Console::flush()
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif
    
    // TODO: I'm unclear if how ggiFlush() returns.
    ggiFlush( m_ggi_visual );
}

void Console::finish()
{
 #ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif
    
    // TODO: I'm unclear if how ggiFlush() returns.
    ggiFlush( m_ggi_visual );
}

void Console::update()
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif
    
    // TODO: I'm unclear if how ggiFlush() returns.
    ggiFlush( m_ggi_visual );
}

void Console::update( const Area &area )
{
    // just to be _safe_
    Area clipped = Clipper::clip( area, this->clip() );

    ggiFlushRegion( m_ggi_visual, clipped.left(), clipped.top(), 
        clipped.width(), clipped.height() );
}

bool Console::key()
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    return ggiKbhit( m_ggi_visual );
}

Key Console::read()
{
#ifdef __DEBUG__
    assert( check_open() );
#endif

    gii_event       l_ggi_event;
    // only way i know how to get input handle for a specific visual
    gii_input_t     l_ggi_input = ggiJoinInputs( m_ggi_visual, NULL );

    // Block until we get a key.
    giiEventRead( l_ggi_input, &l_ggi_event, ( ggi_event_mask )
                                        emKeyPress | emKeyRepeat );

    int code = l_ggi_event.key.sym;
    bool alt = l_ggi_event.key.modifiers & GII_MOD_ALT;
    bool shift = l_ggi_event.key.modifiers & GII_MOD_SHIFT;
    bool control = l_ggi_event.key.modifiers & GII_MOD_CTRL;

    // return key object
    return Key( code, alt, shift, control );
}

void Console::copy( BaseSurface &surface )
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif
    
    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load( pixels, width(), height(), pitch(), format(), palette() );

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console to surface",error);
    }
}

void Console::copy(BaseSurface &surface,const Area &source,const Area &destination)
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif

    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette(),source,destination);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console area to surface area",error);
    }
}

void* Console::lock()
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // TODO: ggi_directbuffer returns a lot of shit, including stuff
    // about alignment, format and page_size.  the code below is a hopeful
    // simple case.  i also need to deal with paging!
    
    try
    {
        // fail if the console is already locked
        if( m_locked ){
            throw Error("console is already locked");
        }

        if( ggiResourceAcquire( m_ggi_db->resource, GGI_ACTYPE_WRITE ) != 0 ){
            throw Error( "could not acquire directbuffer during lock()" );
        }
    }

    catch (Error &error)
    {
        // close
        close();

        // error message
        throw Error( "could not acquire directbuffer during lock()", error );
    }

    m_locked++;
    return (void *) m_ggi_db->write;
}

void Console::unlock()
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // fail if the console is not locked
    if( !m_locked ){
        throw Error( "console is not locked" );
    }

    try
    {
        if( m_ggi_db ){
            if( ggiResourceRelease( m_ggi_db->resource ) != 0 ){
                throw Error( "Could not release directbuffer durring unlock()" );
            }
            m_locked--;
        } else {
            throw Error( "GGI directbuffer is invalid!" );
        }
    }

    catch (Error &error)
    {
        // close
        close();

        // error message
        throw Error( "Could not acquire directbuffer during lock()", error );
    }
}

void Console::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif
    // check clip area
    if (clip()==area())
    {
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(format,this->format());
    
            // set copy palettes
            m_copy.palette(palette,this->palette());

            // copy pixels to surface
            m_copy.copy(pixels,0,0,width,height,pitch,console_pixels,0,0,this->width(),this->height(),this->pitch());

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();

            // error message
            throw Error("failed to load pixels to console",error);
        }
    }
    else
    {
        // load explicit areas
        load(pixels,width,height,pitch,format,palette,Area(0,0,width,height),area());
    }
}

void Console::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif
   
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,Area(0,0,width,height),clipped_source,destination,clip(),clipped_destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch,
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        throw Error("failed to load pixels to console area",error);
    }
}

void Console::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif

    // check clip area
    if (clip()==area())
    {
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(this->format(),format);
    
            // set copy palettes
            m_copy.palette(this->palette(),palette);

            // copy console pixels to 'pixels' buffer
            m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,pitch);

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();
        
            // error message
            throw Error("failed to save console pixels",error);
        }
    }
    else
    {
        // save explicit areas
        save(pixels,width,height,pitch,format,palette,area(),Area(0,0,width,height));
    }
}

void Console::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{

#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif

    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,clip(),clipped_source,destination,Area(0,0,width,height),clipped_destination);

        // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),this->pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch);
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to save console area pixels",error);
    }
}

void Console::clear()
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif
   
    // check console format
    if (format().direct())
    {
        // direct color
        clear(Color(0,0,0,0));
    }
    else
    {
        // indexed color
        clear(Color(0));
    }
}

void Console::clear(const Color &color)
{
    // clear console
    clear(color,area());
}

void Console::clear(const Color &color,const Area &area)
{
#ifdef __DEBUG__
    assert( check_open() );
    assert( check_unlocked() );
#endif
    
   // lock console
    void *pixels = lock();

    try
    {
        // clip area
        Area clipped_area = Clipper::clip(area,this->clip());

        // request clear
        m_clear.request(format());

        // clear console area
        m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to clear console area",error);
    }
}

void Console::palette(const Palette &palette)
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // update cached palette
    m_palette = palette;

    // fail silently if not indexed color
    if ( !format().indexed() ){
        return;
    }
    
    // get palette data
    const int32 *data = ( int32 * )palette.data();

    // convert palette data
    // TODO: the following code is /untested/
    ggi_color temp[256];
    for( int i = 0; i < 256; i++ )
    {
        temp[i].a   =   ( ( data[i] & 0xff000000 ) >> 16 );  
        temp[i].r   =   ( ( data[i] & 0x00ff0000 ) >> 8  );
        temp[i].g   =   ( ( data[i] & 0x0000ff00 ) );
        temp[i].b   =   ( ( data[i] & 0x000000ff ) << 8 );
    }

    // set palette entries
    ggiSetPalette( m_ggi_visual, 0, 256, temp );

    // just in case
    ggiFlush( m_ggi_visual );
}

const Palette& Console::palette() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // get palette
    return m_palette;
}

void Console::clip(const Area &area)
{
#ifdef __DEBUG__
    assert( check_open() );
#endif

    // set new clip area
    m_clip = Clipper::clip(area,this->area());
}




int Console::width() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // get width
    return m_width;
}


int Console::height() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif    
    // get height
    return m_height;
}


int Console::pages() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // get height
    return m_pages;
}


int Console::pitch() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // get pitch
    return m_pitch;
}


const Area& Console::area() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    // get area
    return m_area;
}


const Area& Console::clip() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif

    // get area
    return m_area;
}


const Format& Console::format() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // get format
    return m_format;
}


const char* Console::name() const
{
    // get name
    return "GGI";
}


const char* Console::title() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    
    // get title
    return m_title;
}


const char* Console::information() const
{
#ifdef __DEBUG__
    assert( check_open() );
#endif
    static char info[512];
    
    // TODO: jeez, not a whole lot of info!
    sprintf( info, "PTC GGI ( %d x %d %d bits )\n", 
                    width(), height(), format().bits() ); 
    
    // get information
    return info;
}

// ------------------------------------------------------------------------ //
// GGI PTC specific

ggi_visual_t Console::get_visual()
{
    return m_ggi_visual;
}


// ------------------------------------------------------------------------ //
// PRIVATE FUNCTIONS

const bool Console::check_open() const
{
    return m_open ? 1 : 0;
}


const bool Console::check_unlocked() const
{
    return m_locked ? 1 : 0;
}

const Mode &Console::translate( const ggi_mode &l_ggi_mode )
{
    // PTC mode for your returning pleasure
    static Mode l_ptc_mode;
    Format l_ptc_format;

    const ggi_pixelformat *l_ggi_format = ggiGetPixelFormat( m_ggi_visual );

    switch( GT_SIZE( l_ggi_mode.graphtype ) )
    {
        case 32:
            l_ptc_format = Format( 32, l_ggi_format->red_mask, 
                                        l_ggi_format->green_mask, 
                                        l_ggi_format->blue_mask, 
                                        l_ggi_format->alpha_mask );
            break;
        case 24:
            l_ptc_format = Format( 24, l_ggi_format->red_mask, 
                                        l_ggi_format->green_mask, 
                                        l_ggi_format->blue_mask, 
                                        l_ggi_format->alpha_mask );
            break;
        case 16:
            l_ptc_format = Format( 16, l_ggi_format->red_mask, 
                                        l_ggi_format->green_mask, 
                                        l_ggi_format->blue_mask, 
                                        l_ggi_format->alpha_mask );
           break;
        case 15:
            l_ptc_format = Format( 15, l_ggi_format->red_mask, 
                                        l_ggi_format->green_mask, 
                                        l_ggi_format->blue_mask, 
                                        l_ggi_format->alpha_mask );
            break;
        case 8:
            l_ptc_format = Format( 8 );
            break;
        default: 
            throw Error( "invalid PTC pixel depth" );
            break;
    }

    // we totally ignore virtual pixels here since PTC doesn't
    // really grok the virtual thing.
    l_ptc_mode = Mode( ( int ) l_ggi_mode.visible.x, 
                            ( int ) l_ggi_mode.visible.y, l_ptc_format );

    return l_ptc_mode;
}

ggi_mode *Console::translate( const Mode &l_ptc_mode, int pages = 0 )
{
    
    ggi_mode    *l_ggi_mode = new ggi_mode;
    
    switch( l_ptc_mode.format().bits() )
    {
        case 32:
            l_ggi_mode->graphtype = GT_32BIT;
            break;
        case 24:
            l_ggi_mode->graphtype = GT_24BIT;
            break;
        case 16:
            l_ggi_mode->graphtype = GT_16BIT;
            break;
        case 15:
            l_ggi_mode->graphtype = GT_15BIT;
            break;
        case 8:
            l_ggi_mode->graphtype = GT_8BIT;
            break;
        default:
            throw Error( "invalid GGI pixel depth" );
    }

    if( pages <= 0 ){
        // hmm, probably didn't change the default
        l_ggi_mode->frames = m_pages;
    } else {
        l_ggi_mode->frames = pages;
    }

    l_ggi_mode->visible.x = l_ptc_mode.width();
    l_ggi_mode->visible.y = l_ptc_mode.height();
    
    // virtual?  we don't need no stinking virtual!
    l_ggi_mode->virt.x = GGI_AUTO;
    l_ggi_mode->virt.y = GGI_AUTO;

    return l_ggi_mode;
}

void Console::set_string( const char *source, char **dest )
{
    if( *dest ){
        delete[] ( *dest );
        *dest = 0;
    }
    
    ( *dest ) = new char[strlen( source ) + 1];
    strcpy( ( *dest ), source );
}
