//
// Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include <stdio.h>
#include <string.h>
#include <fstream.h>
#include "Core/Key.h"
#include "Core/Area.h"
#include "Core/Color.h"
#include "Core/Error.h"
#include "Core/Config.h"
#include "Core/Clipper.h"
#include "Core/Console.h"
#include "DirectX/Hook.h"

// define defaults
#define DEFAULT_WIDTH 320;
#define DEFAULT_HEIGHT 200;
#define DEFAULT_FORMAT Format(32,0x00FF0000,0x0000FF00,0x000000FF);
#ifdef __DEBUG__
#define DEFAULT_OUTPUT WINDOWED
#else
#define DEFAULT_OUTPUT DEFAULT
#endif



DLLAPI PTCAPI Console::Console()
{
    // defaults
    m_open   = false;
    m_locked = false;

    // clear strings
    m_title[0] = 0;
    m_information[0] = 0;

    // clear objects
    m_window = 0;
    m_primary = 0;
    m_keyboard = 0;

    // default option data
    m_default_width = DEFAULT_WIDTH;
    m_default_height = DEFAULT_HEIGHT;
    m_default_format = DEFAULT_FORMAT;
    m_output_mode = DEFAULT_OUTPUT;

    // configure console
    configure("ptc.cfg");

    // setup display object
    m_display.setup(m_library.lpDD2());
}


DLLAPI PTCAPI Console::~Console()
{
    // close
    close();
}




DLLAPI void PTCAPI Console::configure(const char file[])
{
    // open configuration file
    ifstream is(file,ios::in | ios::nocreate);
    
    // check stream
    if (!is) return;

    // read option strings
    while (!is.eof())
    {
        // option line
        char line[1024];

        // read line from stream
        is.getline(line,1024);

        // process option
        option(line);
    }
}




DLLAPI bool PTCAPI Console::option(const char option[])
{
    // check for "default output" option
    if (stricmp(option,"default output")==0)
    {
        // default output mode
        m_output_mode = DEFAULT;

        // recognized
        return true;
    }

    // check for "windowed output" option
    if (stricmp(option,"windowed output")==0)
    {
        // windowed output mode
        m_output_mode = WINDOWED;

        // recognized
        return true;
    }

    // check for "fullscreen output" option
    if (stricmp(option,"fullscreen output")==0)
    {
        // fullscreen output mode
        m_output_mode = FULLSCREEN;
       
        // recognized
        return true;
    }

    // check for "default width" option
    if (strnicmp(option,"default width",13)==0)
    {
        // look for parameter
        if (option[13]!=0)
        {
            // we have a parameter
            m_default_width = atoi(&option[13]);
        }
        else
        {
            // no parameter
            m_default_width = DEFAULT_WIDTH;
        }
    }

    // check for "default height" option
    if (strnicmp(option,"default height",14)==0)
    {                            
        // look for parameter
        if (option[14]!=0)
        {
            // we have a parameter
            m_default_height = atoi(&option[14]);
        }
        else
        {
            // no parameter
            m_default_height = DEFAULT_HEIGHT;
        }

        // recognized
        return true;
    }

    // check for "default bits" option
    if (strnicmp(option,"default bits",12)==0)
    {                            
        // look for parameter
        if (option[12]!=0)
        {
            // we have a parameter
            int bits = atoi(&option[12]);
            
            // setup format
            switch (bits)
            {
                case 8:  m_default_format = Format(8); break;
                case 16: m_default_format = Format(16,0xF800,0x07E0,0x001F); break;
                case 24: m_default_format = Format(24,0x00FF0000,0x0000FF00,0x000000FF); break;
                case 32: m_default_format = Format(32,0x00FF0000,0x0000FF00,0x000000FF); break;
                default: return false;
            }
        }
        else
        {
            // no parameter
            m_default_height = DEFAULT_HEIGHT;
        }

        // recognized
        return true;
    }

    // pass the option to the copy object
    return m_copy.option(option);
}




DLLAPI const Mode* PTCAPI Console::modes()
{
    // bogus imitation mode list! :)
    m_modes[0] = Mode(640,480,Format(8));
    m_modes[1] = Mode(800,600,Format(8));
    m_modes[2] = Mode(1024,768,Format(8));
    m_modes[3] = Mode(640,480,Format(16,0xF800,0x07E0,0x001F));
    m_modes[4] = Mode(800,600,Format(16,0xF800,0x07E0,0x001F));
    m_modes[5] = Mode(1024,768,Format(16,0xF800,0x07E0,0x001F));
    m_modes[6] = Mode(640,480,Format(32,0x00FF0000,0x0000FF00,0x000000FF));
    m_modes[7] = Mode(800,600,Format(32,0x00FF0000,0x0000FF00,0x000000FF));
    m_modes[8] = Mode(1024,768,Format(32,0x00FF0000,0x0000FF00,0x000000FF));

    // return modes
    return m_modes;
}




DLLAPI void PTCAPI Console::open(const char title[],int pages)
{
    // open console
    open(title,m_default_format,pages);
}


DLLAPI void PTCAPI Console::open(const char title[],const Format &format,int pages)
{
    // open console
    open(title,m_default_width,m_default_height,m_default_format,pages);
}


DLLAPI void PTCAPI Console::open(const char title[],int width,int height,const Format &format,int pages)
{
    // open exact mode (windowed output is our backup...)
    open(title,Mode(width,height,format),pages);
}


DLLAPI void PTCAPI Console::open(const char title[],const Mode &mode,int pages)
{
    // check that the mode is valid
    if (!mode.valid()) throw Error("invalid mode");

    // get mode information
    int width = mode.width();
    int height = mode.height();
    Format format = mode.format();

    // setup before open
    internal_pre_open_setup(title);

    // check output mode
    switch (m_output_mode)
    {
        case DEFAULT:
        {
            try
            {
                // start internal fullscreen open
                internal_open_fullscreen_start();
        
                // internal open fullscreen mode
                internal_open_fullscreen(width,height,format);

                // finish internal fullscreen open
                internal_open_fullscreen_finish(pages);
            }
            catch (Error&)
            {
                // internal reset
                internal_reset();

                // internal open windowed
                internal_open_windowed(width,height,pages);
            }
        }
        break;

        case WINDOWED:
        {
            // internal open windowed
            internal_open_windowed(width,height,pages);
        }
        break;

        case FULLSCREEN:
        {
            // start internal fullscreen open
            internal_open_fullscreen_start();
    
            // internal open fullscreen mode
            internal_open_fullscreen(width,height,format);

            // finish internal fullscreen open
            internal_open_fullscreen_finish(pages);
        }
        break;
    }

    // setup after open
    internal_post_open_setup();
}


DLLAPI void PTCAPI Console::close()
{
    // check if open
    if (m_open)
    {
        // the console must be unlocked when closed
        if (m_locked) throw Error("console is still locked");

        // flush all key presses
        while (key()) read();
    }

    // close console
    internal_close();
}





DLLAPI void PTCAPI Console::flush()
{
    // debug checks
    check_open();
    check_unlocked();

    // [platform dependent code to flush all console operations]

    // update window
    m_window->update();
}


DLLAPI void PTCAPI Console::finish()
{
    // debug check
    check_open();
    check_unlocked();

    // [platform dependent code to finish all console operations]

    // update window
    m_window->update();
}


DLLAPI void PTCAPI Console::update()
{
    // debug check
    check_open();
    check_unlocked();

    // update primary surface
    m_primary->update();

    // update window
    m_window->update();
}


DLLAPI void PTCAPI Console::update(const Area &area)
{
    // update
    update();
}




DLLAPI bool PTCAPI Console::key()
{
    // debug check
    check_open();
    
    // check for key press
    return m_keyboard->key();
}

                         
DLLAPI Key PTCAPI Console::read()
{
    // debug check
    check_open();

    // read key code from keyboard
    int code = m_keyboard->read(*m_window);

    // [platform dependent code to determine what modifier keys are pressed]
    bool alt = false;
    bool shift = false;
    bool control = false;

    // return key object
    return Key(code,alt,shift,control);
}




DLLAPI void PTCAPI Console::copy(BaseSurface &surface)
{
    // debug check
    check_open();
    check_unlocked();
    
    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console to surface",error);
    }
}


DLLAPI void PTCAPI Console::copy(BaseSurface &surface,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();
    
    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),pitch(),format(),palette(),source,destination);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to copy console area to surface area",error);
    }
}




DLLAPI void* PTCAPI Console::lock()
{
    // debug check
    check_open();
    
    // fail if the console is already locked
    if (m_locked) throw Error("console is already locked");

    // lock primary surface
    void *pixels = m_primary->lock();

    // surface is locked
    m_locked = true;

    // return pixels
    return pixels;
}


DLLAPI void PTCAPI Console::unlock()
{
    // debug check
    check_open();
    
    // fail if the console is not locked
    if (!m_locked) throw Error("console is not locked");

    // unlock primary surface
    m_primary->unlock();

    // we are unlocked
    m_locked = false;
}




DLLAPI void PTCAPI Console::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
    // check clip area
    if (clip()==area())
    {
        // debug check
        check_open();
        check_unlocked();
    
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(format,this->format());
    
            // set copy palettes
            m_copy.palette(palette,this->palette());

            // copy pixels to surface
            m_copy.copy(pixels,0,0,width,height,pitch,console_pixels,0,0,this->width(),this->height(),this->pitch());

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();

            // error message
            throw Error("failed to load pixels to console",error);
        }
    }
    else
    {
        // load explicit areas
        load(pixels,width,height,pitch,format,palette,Area(0,0,width,height),area());
    }
}


DLLAPI void PTCAPI Console::load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();
    
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,Area(0,0,width,height),clipped_source,destination,clip(),clipped_destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch,
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        throw Error("failed to load pixels to console area",error);
    }
}




DLLAPI void PTCAPI Console::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette)
{
    // debug check
    check_open();
    check_unlocked();
    
    // check clip area
    if (clip()==area())
    {
        // lock console
        void *console_pixels = lock();

        try
        {
            // request format conversion
            m_copy.request(this->format(),format);
    
            // set copy palettes
            m_copy.palette(this->palette(),palette);

            // copy console pixels to 'pixels' buffer
            m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,pitch);

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();
        
            // error message
            throw Error("failed to save console pixels",error);
        }
    }
    else
    {
        // save explicit areas
        save(pixels,width,height,pitch,format,palette,area(),Area(0,0,width,height));
    }
}


DLLAPI void PTCAPI Console::save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination)
{
    // debug check
    check_open();
    check_unlocked();
    
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source,clipped_destination;
        Clipper::clip(source,clip(),clipped_source,destination,Area(0,0,width,height),clipped_destination);

        // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),this->pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch);
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        throw Error("failed to save console area pixels",error);
    }
}




DLLAPI void PTCAPI Console::clear()
{
    // debug check
    check_open();
    check_unlocked();
    
    // check console format
    if (format().direct())
    {
        // direct color
        clear(Color(0,0,0,0));
    }
    else
    {
        // indexed color
        clear(Color(0));
    }
}


DLLAPI void PTCAPI Console::clear(const Color &color)
{
    // debug check
    check_open();
    check_unlocked();

    // clear console
    clear(color,area());
}


DLLAPI void PTCAPI Console::clear(const Color &color,const Area &area)
{
    // debug check
    check_open();
    check_unlocked();

    // clear primary surface
    m_primary->clear(color,area);
}




DLLAPI void PTCAPI Console::palette(const Palette &palette)
{
    // debug check
    check_open();
    
    // set primary surface palette
    m_primary->palette(palette);
}


DLLAPI const Palette& PTCAPI Console::palette() const
{
    // debug check
    check_open();
    
    // get primary surface palette
    return m_primary->palette();
}




DLLAPI void PTCAPI Console::clip(const Area &area)
{
    // debug check
    check_open();

    // set clip area
    m_primary->clip(area);
}




DLLAPI int PTCAPI Console::width() const
{
    // debug check
    check_open();
    
    // get primary width
    return m_primary->width();
}


DLLAPI int PTCAPI Console::height() const
{
    // debug check
    check_open();
    
    // get primary height
    return m_primary->height();
}


DLLAPI int PTCAPI Console::pages() const
{
    // debug check
    check_open();
    
    // get primary pages
    return m_primary->pages();
}


DLLAPI int PTCAPI Console::pitch() const
{
    // debug check
    check_open();
    
    // get primary pitch
    return m_primary->pitch();
}


DLLAPI const Area& PTCAPI Console::area() const
{
    // debug check
    check_open();
    
    // get primary area
    return m_primary->area();
}


DLLAPI const Area& PTCAPI Console::clip() const
{
    // debug check
    check_open();
    
    // get primary clip
    return m_primary->clip();
}


DLLAPI const Format& PTCAPI Console::format() const
{
    // debug check
    check_open();
    
    // get primary format
    return m_primary->format();
}


DLLAPI const char* PTCAPI Console::name() const
{
    // get name
    return "DirectX";
}


DLLAPI const char* PTCAPI Console::title() const
{
    // debug check
    check_open();
    
    // get title
    return m_title;
}


DLLAPI const char* PTCAPI Console::information() const
{
    // debug check
    check_open();
    
    // get information
    return m_information;
}




DLLAPI HWND PTCAPI Console::window() const
{
    // debug check
    check_open();
    
    // get window handle
    return m_window->handle();
}


DLLAPI LPDIRECTDRAW Console::lpDD() const
{
    // debug check
    check_open();

    // get directdraw interface
    return m_library.lpDD();
}



DLLAPI LPDIRECTDRAW2 Console::lpDD2() const
{
    // debug check
    check_open();

    // get directdraw 2 interface
    return m_library.lpDD2();
}



DLLAPI LPDIRECTDRAWSURFACE Console::lpDDS() const
{
    // debug check
    check_open();

    // get directdraw surface interface
    return m_primary->lpDDS();
}




void Console::internal_pre_open_setup(const char title[])
{
    // close down
    internal_close();

    // check length of console title
    if (strlen(title)+1<=sizeof(m_title))
    {
        // update console title
        strcpy(m_title,title);
    }
    else
    {
        // update console title truncated
        memcpy(m_title,title,sizeof(m_title));
        m_title[sizeof(m_title)-1] = 0;
    }
}


void Console::internal_open_fullscreen_start()
{
    // create fullscreen window
    m_window = new Win32Window("PTC_DIRECTX_FULLSCREEN",m_title,0,/*WS_SHOW|*/WS_POPUP|WS_SYSMENU,SW_NORMAL,0,0,0,0,false,false);

    // set cooperative level
    m_display.cooperative(m_window->handle(),true);
}


void Console::internal_open_fullscreen(int width,int height,const Format &format)
{
    // open display
    m_display.open(width,height,format);
}


void Console::internal_open_fullscreen_finish(int pages)
{
    try
    {
        // n buffered video memory primary surface
        m_primary = new DirectXPrimary(m_window->handle(),m_library.lpDD2(),pages,true,true);
    }
    catch (Error&)
    {
        // failure
        m_primary = 0;
    }

    // check primary
    if (!m_primary)
    {
        try
        {
            // triple buffered video memory primary surface
            m_primary = new DirectXPrimary(m_window->handle(),m_library.lpDD2(),3,true,true);
        }
        catch (Error&)
        {
            // double buffered primary surface
            m_primary = new DirectXPrimary(m_window->handle(),m_library.lpDD2(),2,false,true);
        }
    }
}


void Console::internal_open_windowed(int width,int height,int pages)
{
    // get system metrics
    int frame_x = GetSystemMetrics(SM_CXFIXEDFRAME);
    int frame_y = GetSystemMetrics(SM_CYFIXEDFRAME);
    int title_y = GetSystemMetrics(SM_CYCAPTION);

    // adjust size
    width  += frame_x*2;
    height += frame_y*2 + title_y;

    // create windowed window
    m_window = new Win32Window("PTC_DIRECTX_WINDOWED",m_title,0,WS_VISIBLE|WS_SYSMENU|WS_CAPTION|WS_MINIMIZE,SW_NORMAL,CW_USEDEFAULT,CW_USEDEFAULT,width,height,true,false);

    // set cooperative level for windowed output
    m_display.cooperative(m_window->handle(),false);

    // open windowed display
    m_display.open();

    // create primary surface
    m_primary = new DirectXPrimary(m_window->handle(),m_library.lpDD2(),1,false,false);
}


void Console::internal_post_open_setup()
{
    // create win32 keyboard
    m_keyboard = new Win32Keyboard(m_window->handle(),m_window->thread(),false);

    // temporary platform dependent information fudge
    sprintf(m_information,"windows version x.xx.x\nddraw version x.xx\ndisplay driver name xxxxx\ndisplay driver vendor xxxxx\ncertified driver? x\n");

    // set open flag
    m_open = true;
}


void Console::internal_reset()
{
    // delete primary
    delete m_primary;

    // delete keyboard
    delete m_keyboard;

    // delete window
    delete m_window;

    // clear pointers
    m_primary = 0;
    m_keyboard = 0;
    m_window = 0;
}


void Console::internal_close()
{
    // delete primary
    delete m_primary;

    // close display
    m_display.close();

    // delete keyboard
    delete m_keyboard;

    // delete window
    delete m_window;

    // clear pointers
    m_primary = 0;
    m_keyboard = 0;
    m_window = 0;
}




void Console::check_open() const
{
    #ifdef __DEBUG__

        // check that console is open
        if (!m_open) throw Error("console is not open");

    #else

        // no checking in release build

    #endif
}


void Console::check_unlocked() const
{
    #ifdef __DEBUG__

        // check that console is unlocked
        if (m_locked) throw Error("console is not unlocked");

    #else

        // no checking in release build

    #endif
}
