//
// Window class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include <memory.h>
#include "Base/Types.h"
#include "Core/Error.h"
#include "Win32/Window.h"

// using directive
using namespace ptc::Win32;

// using declarations
using ptc::Core::Error;




Window::Window(HWND window)
{
    // defaults
    defaults();

    // setup window
    m_window = window;
}


Window::Window(const char wndclass[],const char title[],unsigned extra,unsigned style,int show,int x,int y,int width,int height,bool cursor,bool multithreaded,void *data)
{
    // defaults
    defaults();

    // setup multithreaded flag
    m_multithreaded = multithreaded;

    try
    {
        // get module instances
        HINSTANCE program_instance = GetModuleHandle(0);
		HINSTANCE library_instance = GetModuleHandle("ptc.dll");
		if (!library_instance) library_instance = program_instance;

		// get cursor dimensions
		int cursor_width = GetSystemMetrics(SM_CXCURSOR);
		int cursor_height = GetSystemMetrics(SM_CYCURSOR);

		// setup cursor bits
        char8 and_map[256];
		memset(and_map,0xFF,256);
        char8 xor_map[256];
		memset(xor_map,0x00,256);

        // create blank cursor
        m_cursor = CreateCursor(program_instance,0,0,cursor_width,cursor_height,and_map,xor_map);
		if (!m_cursor) throw Error("could not create cursor");

        // register window class
        WNDCLASSEX wc;
        wc.cbSize        = sizeof(WNDCLASSEX);
        wc.hInstance     = program_instance;
        wc.lpszClassName = wndclass;
        wc.style         = CS_VREDRAW|CS_HREDRAW;
        wc.hIcon         = LoadIcon(library_instance,"IDI_PTC_ICON");
        wc.hIconSm       = 0;
        wc.lpszMenuName  = 0;
        wc.cbClsExtra    = 0;
        wc.cbWndExtra    = 0;
        wc.hbrBackground = 0;//GetStockObject(BLACK_BRUSH);
        if (multithreaded) wc.lpfnWndProc = (WNDPROC)WndProcMultiThreaded;
        else wc.lpfnWndProc = (WNDPROC)WndProcSingleThreaded;
        if (!cursor) wc.hCursor = m_cursor;
        else wc.hCursor = LoadCursor(0,IDC_ARROW);
        RegisterClassEx(&wc);

        // setup window data
        strcpy(m_name,wndclass);
        strcpy(m_title,title);
        m_extra  = extra;
        m_style  = style;
        m_show   = show;
        m_x      = x;
        m_y      = y;
        m_width  = width;
        m_height = height;
        m_data   = data;

        // check multithread flag
        if (multithreaded)
        {
            // create window event
            m_event = CreateEvent(0,0,0,0);

            // create window thread
            m_thread = CreateThread(0,0,(LPTHREAD_START_ROUTINE)ThreadFunction,this,0,&m_id);

            // wait for window signal
            WaitForSingleObject(m_event,INFINITE);

            // check window handle
            if (!IsWindow(m_window)) throw Error("could not create window");
        }
        else
        {
            // create window
            m_window = CreateWindowEx(m_extra,m_name,m_title,m_style,m_x,m_y,m_width,m_height,0,0,0,m_data);
    
            // check window handle
            if (!IsWindow(m_window)) throw Error("could not create window");

            // show window
            ShowWindow(m_window,m_show);

            // set window focus
            SetFocus(m_window);

            // set foreground window
            SetForegroundWindow(m_window);
        }
    }
    catch (Error error)
    {
        // close
        close();

        // error message
        error.rethrow("could not create window");
    }
}


Window::~Window()
{
    // close
    close();
}




void Window::update()
{
    // check multithread flag
    if (!m_multithreaded)
    {   
        // message
        MSG message;
        
        // process all messages
        while (PeekMessage(&message,m_window,0,0,PM_REMOVE))
        {
            // handle WM_QUIT message
            if (message.message==WM_QUIT) return;

            // translate and dispatch
            TranslateMessage(&message);
            DispatchMessage(&message);
        }
    }
    else
    {
        // sleep
        Sleep(0);
    }
}




HWND Window::handle()
{
    // get handle
    return m_window;
}


DWORD Window::thread()
{
    // get thread id
    if (m_multithreaded) return m_id;
    else return GetCurrentThreadId();
}


bool Window::multithreaded()
{
    // get multithreaded flag
    return m_multithreaded;
}




void Window::defaults()
{
    // defaults
    m_window = 0;
    m_cursor = 0;
    m_event = 0;
    m_thread = 0;
    m_id = 0;
    m_multithreaded = false;
}


void Window::close()
{
    // check multithread flag
    if (m_multithreaded)
    {
        // check window and thread
        if (m_thread && IsWindow(m_window))
        {
            // close window
            PostMessage(m_window,WM_DESTROY,0,0);
            
            // wait for window thread to exit
            WaitForSingleObject(m_thread,INFINITE);
        }

        // release objects
        if (m_event)  CloseHandle(m_event);
        if (m_thread) CloseHandle(m_thread);

        // set normal priority class
        SetPriorityClass(GetCurrentProcess(),NORMAL_PRIORITY_CLASS);
    }
    else
    {
        // destroy window
        DestroyWindow(m_window);

        // update
        update();
    }

    // destroy cursor
    if (m_cursor) DestroyCursor(m_cursor);

    // reset data
    m_window = 0;
    m_cursor = 0;
    m_event = 0;
    m_thread = 0;
    m_id = 0;

    // unregister window class
    UnregisterClass("PTC_WINDOW",GetModuleHandle(0));
}




void WINAPI Window::ThreadFunction(Window *owner)
{
    // create window
    owner->m_window = CreateWindowEx(owner->m_extra,owner->m_name,owner->m_title,owner->m_style,owner->m_x,owner->m_y,owner->m_width,owner->m_height,0,0,0,owner->m_data);
    
    // check window handle
    if (IsWindow(owner->m_window))
    {
        // show window
        ShowWindow(owner->m_window,owner->m_show);

        // set window focus
        SetFocus(owner->m_window);

        // set foreground window
        SetForegroundWindow(owner->m_window);

        // signal window event
        SetEvent(owner->m_event);

        // message loop
        MSG message;
        while (GetMessage(&message,0,0,0)==TRUE)
        {
            // process window message
            TranslateMessage(&message);
            DispatchMessage(&message);
        }
    }
    else
    {
        // window failure
        SetEvent(owner->m_event);
    }
}


LRESULT CALLBACK Window::WndProcSingleThreaded(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam)
{
    // result data
    int result = 0;

    // handle message
    switch (message)
    {
        case WM_CLOSE:
        {
            // exit
            exit(0);
        }

        default:
        {
            // unhandled messages
            result = DefWindowProc(hWnd,message,wParam,lParam);
        }
    }

    // return value
    return result;
}


LRESULT CALLBACK Window::WndProcMultiThreaded(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam)
{
    // result data
    int result = 0;

    // handle message
    switch (message)
    {
        case WM_DESTROY:
        {
            // shutdown window
            PostQuitMessage(0);
        }
        break;

        case WM_CLOSE:
        {
            // exit
            exit(0);
        }

        default:
        {
            // unhandled messages
            result = DefWindowProc(hWnd,message,wParam,lParam);
        }
    }

    // return value
    return result;
}
