//
// Primary class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "Core/Area.h"
#include "Core/Clip.h"
#include "Core/Error.h"
#include "Core/Color.h"
#include "Core/Palette.h"
#include "Base/Surface.h"
#include "DirectX/Check.h"
#include "DirectX/Primary.h"

// using directive
using namespace ptc::DirectX;

// using declarations
using ptc::Core::Area;
using ptc::Core::Clip;
using ptc::Core::Error;
using ptc::Core::Color;
using ptc::Core::Format;
using ptc::Core::Palette;




Primary::Primary(HWND window,LPDIRECTDRAW ddraw,int pages,bool indexed,bool video,bool fullscreen)
{
    // setup ddraw
    m_ddraw = ddraw;

    // setup window
    m_window = window;

    // defaults
    m_width   = 0;
    m_height  = 0;
    m_back    = 0;
    m_front   = 0;
    m_ddpal   = 0;
    m_surface = 0;
    m_pages   = 0;

    // setup fullscreen flag
    m_fullscreen = fullscreen;
    
    try
    {
        // clear indexed if windowed
        if (!fullscreen) indexed = false;

        // allocate surface array
        m_surface = new LPDIRECTDRAWSURFACE[pages];
        memset(m_surface,0,sizeof(LPDIRECTDRAWSURFACE)*pages);

        // create primary surface
        DDSURFACEDESC descriptor;
        descriptor.dwSize  = sizeof(descriptor);
        descriptor.dwFlags = DDSD_CAPS;
        if (!indexed && pages>1) descriptor.dwFlags |= DDSD_BACKBUFFERCOUNT;
        descriptor.dwBackBufferCount = pages - 1;
        descriptor.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
        if (video) descriptor.ddsCaps.dwCaps |= DDSCAPS_VIDEOMEMORY;
        if (!indexed && pages>1) descriptor.ddsCaps.dwCaps |= DDSCAPS_COMPLEX | DDSCAPS_FLIP;
        check(m_ddraw->CreateSurface(&descriptor,&m_surface[0],0));
    
        // get surface descriptor
        check(m_surface[0]->GetSurfaceDesc(&descriptor));
        
        // get pixel format
        DDPIXELFORMAT ddpf;
        ddpf.dwSize = sizeof(ddpf);
        check(m_surface[0]->GetPixelFormat(&ddpf));

        // setup data
        m_front  = m_surface[0];
        m_pages  = pages;
        m_width  = descriptor.dwWidth;
        m_height = descriptor.dwHeight;
        m_format = translate(ddpf);

        // check if fullscreen
        if (!fullscreen)
        {
            // get window client area
            RECT rectangle;
            GetClientRect(m_window,&rectangle);

            // set data to match window
            m_width  = rectangle.right;
            m_height = rectangle.bottom;
        }

        // setup palette
        if (indexed)
        {
            // setup black palette
            PALETTEENTRY data[256];
            memset(data,0,256*sizeof(PALETTEENTRY));

            // create palette object
            check(m_ddraw->CreatePalette(DDPCAPS_8BIT|DDPCAPS_ALLOW256|DDPCAPS_INITIALIZE,data,&m_ddpal,0));

            // set palette of front surface
            check(m_front->SetPalette(m_ddpal));

            // iterate number of pages
            for (int i=1; i<pages; i++)
            {
                // create offscreen surface
                descriptor.dwFlags  = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
                descriptor.dwWidth  = m_width;
                descriptor.dwHeight = m_height;
                if (!video) descriptor.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
                else descriptor.ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY;
                check(m_ddraw->CreateSurface(&descriptor,&m_surface[i],0));

                // attach to front surface
                check(m_front->AddAttachedSurface(m_surface[i]));
            }
        }
        
        // check pages
        if (pages>1)
        {
            // get back surface
            DDSCAPS capabilities;
            capabilities.dwCaps = DDSCAPS_BACKBUFFER;
            check(m_front->GetAttachedSurface(&capabilities,&m_back));
        }
        else
        {
            // only one surface
            m_back = m_front;
        }
        
        // clear all pages
        while (pages--)
		{
            /*
			DDBLTFX fx;
			fx.dwSize = sizeof(fx);
			fx.dwFillColor = 0;
			m_back->Blt(0,0,0,DDBLT_COLORFILL,&fx);
            */
            clear(Color(0,0,0));
			update();
		}
    }
    catch (Error &error)
    {
        // close
        close();

        // error message
        error.rethrow("could not create primary surface");
    }
}


Primary::~Primary()
{
    // close
    close();
}




void Primary::update()
{
    // flip surface
    if (m_front!=m_back) check(m_front->Flip(0,DDFLIP_WAIT));

    // sleep
    Sleep(0);
}




void Primary::copy(ptc::Base::Surface &surface)
{
    // lock primary
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console to surface");
    }
}


void Primary::copy(ptc::Base::Surface &surface,const ptc::Base::Area &source,const ptc::Base::Area &destination)
{
    // lock primary
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),source,destination,format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console area to surface area");
    }
}




void* Primary::lock()
{
    // setup surface descriptor
    DDSURFACEDESC descriptor;
    descriptor.dwSize = sizeof descriptor;

    // lock surface
    check(m_back->Lock(0,&descriptor,DDLOCK_WAIT,0));

    // setup lock address pointer
    char8 *address = (char8*) descriptor.lpSurface;

    // if windowed add window offset
    if (!m_fullscreen)
    {
        // get system metrics
        int frame_x = GetSystemMetrics(SM_CXFIXEDFRAME);
        int frame_y = GetSystemMetrics(SM_CYFIXEDFRAME);
        int title_y = GetSystemMetrics(SM_CYCAPTION);

        // get window rectangle
        RECT rectangle;
        GetWindowRect(m_window,&rectangle);
        
        // calculate client area origin
        int x = rectangle.left + frame_x;
        int y = rectangle.top + frame_y + title_y;

        // adjust lock address to window client area
        address += y*descriptor.lPitch + x * descriptor.ddpfPixelFormat.dwRGBBitCount/8;  
    }

    // return locked buffer
    return address;
}


void Primary::unlock()
{
    // unlock surface
    m_back->Unlock(0);
}




void Primary::load(const void *pixels,int width,int height,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,0,0,width,height,width*format.bytes(),console_pixels,0,0,this->width(),this->height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to load pixels to console");
    }
}


void Primary::load(const void *pixels,int width,int height,const ptc::Base::Area &source,const ptc::Base::Area &destination,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
	    // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,width,height),source);
        Area clipped_destination = Clip::clip(Area(0,0,this->width(),this->height()),destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),width*format.bytes(),
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to load pixels to console area");
    }
}




void Primary::save(void *pixels,int width,int height,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(m_format,format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,width*format.bytes());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to save console pixels");
    }
}


void Primary::save(void *pixels,int width,int height,const ptc::Base::Area &source,const ptc::Base::Area &destination,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,this->width(),this->height()),source);
        Area clipped_destination = Clip::clip(Area(0,0,width,height),destination);

        // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),width*format.bytes());
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to save console area pixels");
    }
}




void Primary::clear(const ptc::Base::Color &color)
{
    // lock console
    void *pixels = lock();

    try
    {
        // request clear
        m_clear.request(format());

        // clear console
        m_clear.clear(pixels,0,0,width(),height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console");
    }
}


void Primary::clear(const ptc::Base::Color &color,const ptc::Base::Area &area)
{
    // lock console
    void *pixels = lock();

    try
    {
        // clip area
        Area clipped_area = Clip::clip(Area(0,0,width(),height()),area);

        // request clear
        m_clear.request(format());

        // clear console area
        m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console area");
    }
}




void Primary::palette(const ptc::Base::Palette &palette)
{
    // assign palette
    m_palette = palette;

    // get palette data
    const int32 *data = palette.data();

    // check format and palette
    if (!m_format.indexed() || !m_ddpal) return;

    // convert palette data
    PALETTEENTRY temp[256];
    for (int i=0; i<256; i++)
    {
        temp[i].peRed   = (BYTE) ( (data[i] & 0x00FF0000) >> 16 );
        temp[i].peGreen = (BYTE) ( (data[i] & 0x0000FF00) >> 8  );
        temp[i].peBlue  = (BYTE) ( (data[i] & 0x000000FF) >> 0  );
        temp[i].peFlags = 0;
    }

    // set palette entries
    check(m_ddpal->SetEntries(0,0,256,temp));
}


const ptc::Base::Palette& Primary::palette() const
{
    // get palette
    return m_palette;
}




int Primary::width() const
{
    // get width
    return m_width;
}


int Primary::height() const
{
    // get height
    return m_height;
}


int Primary::pitch() const
{
    // setup surface descriptor
    DDSURFACEDESC descriptor;
    descriptor.dwSize = sizeof(descriptor);

    // get surface descriptor
    check(m_back->GetSurfaceDesc(&descriptor));

    // return surface pitch
    return descriptor.lPitch;
}


const ptc::Base::Format& Primary::format() const
{
    // get format
    return m_format;
}




Format Primary::translate(DDPIXELFORMAT const &ddpf)
{
    // pixel format
    Format format;

    // check rgb data
    if (ddpf.dwFlags & DDPF_PALETTEINDEXED8)
    {
        // indexed color
        format = Format(8);
    }
    else if (ddpf.dwFlags & DDPF_RGB)
    {
        // direct color
        format = Format(ddpf.dwRGBBitCount,ddpf.dwRBitMask,ddpf.dwGBitMask,ddpf.dwBBitMask);
    }
    else
    {
        // error message
        throw Error("invalid pixel format");
    }

    // note: i should add alpha component support

    // return format
    return format;
}




void Primary::close()
{
    // wait for retrace

    // check if fullscreen
    if (m_fullscreen)
    {
        // clear palette
        palette(Palette());

        // check back buffer
        if (m_back)
        {
            // clear all pages
            while (m_pages--)
		    {
			    DDBLTFX fx;
			    fx.dwSize = sizeof(fx);
			    fx.dwFillColor = 0;
			    m_back->Blt(0,0,0,DDBLT_COLORFILL,&fx);
			    update();
		    }
        }
    }

    // wait a bit
    //Sleep(100);
   
    // iterate number of pages
    for (int i=0; i<m_pages; i++)
    {
        // release surface
        if (m_surface[i]) m_surface[i]->Release();
    }

    // delete array
    delete[] m_surface;
}
