//
// DirectX Console implementation for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "Core/Error.h"
#include "Core/Format.h"
#include "DirectX/Console.h"

// using directive
using namespace ptc::DirectX;

// using declarations
using ptc::Core::Error;
using ptc::Core::Format;
using ptc::Win32::Window;
using ptc::Win32::Keyboard;




Console::Console()
{
    // defaults
    m_hook = 0;
    m_window = 0;
    m_library = 0;
    m_display = 0;
    m_primary = 0;
    m_keyboard = 0;
    m_foreground = 0;
    m_fullscreen = false;

    // default console options
    m_multithreaded_option = false;

    // save foreground window
    m_foreground = GetForegroundWindow();

    // save foreground window rectangle
    GetWindowRect(m_foreground,&m_rectangle);
}


Console::~Console()
{
    // close
    close();

    // restore foreground position
    SetWindowPos(m_foreground,HWND_BOTTOM,m_rectangle.left,m_rectangle.top,m_rectangle.right-m_rectangle.left,m_rectangle.bottom-m_rectangle.top,SWP_NOACTIVATE|SWP_NOZORDER);

    // set back to foreground
    SetForegroundWindow(m_foreground);
}




bool Console::option(const char string[])
{
    // multithreaded option
    if (strcmp(string,"multithreaded window")==0)
    {
        // set option flag
        m_multithreaded_option = true;
        return true;
    }

    // not recognized
    return false;
}




void Console::open(const char title[])
{
    // todo: smart default format
    
    // default format
    Format format(32,0x00FF0000,0x0000FF00,0x000000FF);

    // open console
    open(title,format);
}


void Console::open(const char title[],const ptc::Base::Format &format)
{
    // todo: smart default width and height

    // default width and height
    int width  = 640;
    int height = 480;

    // open console
    open(title,width,height,format);
}


void Console::open(const char title[],int width,int height,const ptc::Base::Format &format)
{
    try
    {
        // try fullscreen output
        open(title,width,height,format,true);
    }
    catch (Error&)
    {
        // try windowed output
        open(title,width,height,format,false);
    }
}


void Console::open(const char title[],int width,int height,const ptc::Base::Format &format,bool fullscreen)
{
    // close
    close();

    // create library
    m_library = new Library();

    // create display
    m_display = new Display(m_library->lpDD());

    // check if fullscreen
    if (fullscreen)
    {
        // hide cursor
        ShowCursor(0);

        // test display
        m_display->test(title,width,height,format);

        // create fullscreen window
        m_window = new Window("PTC_DIRECTX_FULLSCREEN",title,0,WS_VISIBLE|WS_POPUP|WS_SYSMENU,SW_NORMAL,0,0,0,0,false,m_multithreaded_option);

        // create window hook
        Hook *hook = new Hook(*this,m_window->handle(),m_window->thread());
    
        // window message data
        Display::Data data;
        data.display = m_display;
        data.title   = title;
        data.width   = width;
        data.height  = height;
        data.format  = &format;

        // display open result
        bool result = false;
    
        // post window message to open display
        PostMessage(m_window->handle(),WM_DIRECTX_OPEN,(WPARAM)&data,(LPARAM)&result);

        // wait for open to complete if multithreaded
        if (m_window->multithreaded()) data.event.wait();

        // update window
        m_window->update();

        // check result
        if (!result) throw Error("could not set display mode");
        
        // destroy hook
        delete hook;
        
        // fullscreen output
        m_fullscreen = true;

        try
        {
            // triple buffered video memory primary surface
            m_primary = new Primary(m_window->handle(),m_library->lpDD(),3,format.indexed(),true,true);
        }
        catch (Error&)
        {
            // double buffered primary surface
            m_primary = new Primary(m_window->handle(),m_library->lpDD(),2,format.indexed(),false,true);
        }
    }
    else
    {
        // todo: adjust window width and height

        // get system metrics
        int frame_x = GetSystemMetrics(SM_CXFIXEDFRAME);
        int frame_y = GetSystemMetrics(SM_CYFIXEDFRAME);
        int title_y = GetSystemMetrics(SM_CYCAPTION);

        // adjust size
        width  += frame_x*2;
        height += frame_y*2 + title_y;

        // create windowed window
        m_window = new Window("PTC_DIRECTX_WINDOWED",title,0,WS_VISIBLE|WS_SYSMENU|WS_CAPTION|WS_MINIMIZE,SW_NORMAL,CW_USEDEFAULT,CW_USEDEFAULT,width,height,true,m_multithreaded_option,0);
        
        // open windowed display
        m_display->open(m_window->handle());

        // create primary surface
        m_primary = new Primary(m_window->handle(),m_library->lpDD(),1,format.indexed(),false,false);

        // not fullscreen output
        m_fullscreen = false;
    }

    // create keyboard
    m_keyboard = new Keyboard(m_window->handle(),m_window->thread(),m_multithreaded_option);
}


void Console::close()
{
    // check if open
    bool open = false;
    if (m_display) open = true;
    
    // free keyboard
    delete m_keyboard;
    m_keyboard = 0;

    // free primary
    delete m_primary;
    m_primary = 0;

    // close display
    if (m_fullscreen && m_display && m_window)
    {
        // create window hook
        Hook *hook = new Hook(*this,m_window->handle(),m_window->thread());
        
        // setup display data
        Display::Data data; 
        data.display = m_display;

        // post message to close display
        PostMessage(m_window->handle(),WM_DIRECTX_CLOSE,(WPARAM)&data,0);

        // update window
        m_window->update();

        // wait for close to complete if multithreaded
        if (m_window->multithreaded()) data.event.wait();

        // destroy hook
        delete hook;
        //delete m_hook;
        //m_hook = 0;
    }

    // free display
    delete m_display;
    m_display = 0;

    // free window
    delete m_window;
    m_window = 0;

    // free library
    delete m_library;
    m_library = 0;

    // show cursor
    if (open && m_fullscreen) ShowCursor(1);
}




void Console::update()
{
    // update console
    m_primary->update();

    // update window
    m_window->update();
}


void Console::update(const ptc::Base::Area &area)
{
    // update
    update();
}




bool Console::key()
{
    // check for keypress
    return m_keyboard->key();
}


int Console::read()
{
    // read keypress
    return m_keyboard->read(*m_window);
}




void Console::copy(ptc::Base::Surface &surface)
{
    // copy primary to surface
    m_primary->copy(surface);
}


void Console::copy(ptc::Base::Surface &surface,const ptc::Base::Area &source,const ptc::Base::Area &destination)
{
    // copy primary area to surface area
    m_primary->copy(surface,source,destination);
}




void* Console::lock()
{
    // lock primary
    return m_primary->lock();
}


void Console::unlock()
{
    // unlock console
    m_primary->unlock();
}




void Console::load(const void *pixels,int width,int height,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // load pixels to primary
    m_primary->load(pixels,width,height,format,palette);
}


void Console::load(const void *pixels,int width,int height,const ptc::Base::Area &source,const ptc::Base::Area &destination,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // load pixels to primary
    m_primary->load(pixels,width,height,source,destination,format,palette);
}




void Console::save(void *pixels,int width,int height,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // save primary pixels
    m_primary->save(pixels,width,height,format,palette);
}


void Console::save(void *pixels,int width,int height,const ptc::Base::Area &source,const ptc::Base::Area &destination,const ptc::Base::Format &format,const ptc::Base::Palette &palette)
{
    // save primary pixels
    m_primary->save(pixels,width,height,source,destination,format,palette);
}




void Console::clear(const ptc::Base::Color &color)
{
    // clear primary
    m_primary->clear(color);
}


void Console::clear(const ptc::Base::Color &color,const ptc::Base::Area &area)
{
    // clear primary area
    m_primary->clear(color,area);
}




void Console::palette(const ptc::Base::Palette &palette)
{
    // set palette
    m_primary->palette(palette);
}


const ptc::Base::Palette& Console::palette() const
{
    // get palette
    return m_primary->palette();
}




int Console::width() const
{
    // get width of console
    return m_primary->width();
}


int Console::height() const
{
    // get height of console
    return m_primary->height();
}


int Console::pitch() const
{
    // get pitch of console
    return m_primary->pitch();
}


const ptc::Base::Format& Console::format() const
{
    // get format of console
    return m_primary->format();
}


const char* Console::name() const
{
    // get console name
    return "DirectX";
}
